package org.gcube.spatial.data.geonetwork.configuration;

import java.util.Set;

import org.gcube.spatial.data.geonetwork.extension.ServerAccess.Version;
import org.gcube.spatial.data.geonetwork.model.Account;
import org.gcube.spatial.data.geonetwork.model.ScopeConfiguration;
import org.gcube.spatial.data.geonetwork.model.faults.EncryptionException;
import org.gcube.spatial.data.geonetwork.model.faults.MissingConfigurationException;
import org.gcube.spatial.data.geonetwork.model.faults.MissingServiceEndpointException;

/**
 * Interface for managing GeoNetwork configurations.
 */
public interface Configuration {

	/**
	 * Gets the GeoNetwork endpoint URL.
	 * @return the GeoNetwork endpoint URL
	 * @throws MissingServiceEndpointException if the service endpoint cannot be found
	 */
	public String getGeoNetworkEndpoint()throws MissingServiceEndpointException;	
	
	/**
	 * Gets the GeoNetwork version.
	 * @return the GeoNetwork version
	 * @throws MissingServiceEndpointException if the service endpoint cannot be found
	 */
	public Version getGeoNetworkVersion()throws MissingServiceEndpointException;
	
	/**
	 * Gets the current scope configuration.
	 * @return current scope configuration
	 * @throws MissingServiceEndpointException if the service endpoint cannot be found
	 * @throws MissingConfigurationException if the configuration is missing
	 */
	public ScopeConfiguration getScopeConfiguration() throws MissingServiceEndpointException,MissingConfigurationException;
	
	/**
	 * Assigns an available configuration to the current scope and returns it.
	 * @return the acquired scope configuration
	 * @throws MissingServiceEndpointException if the service endpoint cannot be found
	 * @throws MissingConfigurationException if no available configuration is found
	 */
	public ScopeConfiguration acquireConfiguration() throws MissingServiceEndpointException,MissingConfigurationException;
	
	/**
	 * Gets the admin account.
	 * @return the admin account
	 * @throws MissingServiceEndpointException if the service endpoint cannot be found
	 */
	public Account getAdminAccount() throws MissingServiceEndpointException;
		
	/**
	 * Gets all existing configurations.
	 * @return set of existing scope configurations
	 * @throws MissingServiceEndpointException if the service endpoint cannot be found
	 */
	public Set<ScopeConfiguration> getExistingConfigurations()throws MissingServiceEndpointException;
	
	/**
	 * Gets configurations from parent scopes.
	 * @return set of parent scope configurations
	 * @throws MissingServiceEndpointException if the service endpoint cannot be found
	 */
	public Set<ScopeConfiguration> getParentScopesConfiguration()throws MissingServiceEndpointException;
	/**
	 * Store the passed configuration
	 * 
	 * @param toCreate the configuration to create and store
	 * @throws MissingServiceEndpointException if the service endpoint cannot be found
	 */
	public void createScopeConfiguration(ScopeConfiguration toCreate) throws MissingServiceEndpointException ;
}
