package org.gcube.social_networking.liferay.ws;

import static org.gcube.resources.discovery.icclient.ICFactory.clientFor;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;

import java.rmi.ServerException;
import java.util.List;

import org.gcube.common.resources.gcore.ServiceEndpoint;
import org.gcube.common.security.secrets.Secret;
import org.gcube.idm.common.is.InfrastrctureServiceClient;
import org.gcube.idm.common.is.IsServerConfig;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.api.SimpleQuery;
import org.gcube.social_networking.utils.InfrastructureUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.ws.rs.NotFoundException;

/**
 * This is a singleton bean instantiated at service start up. It contains
 * the credentials of the user who is allowed to perform calls to Liferay.
 * Its credentials are looked up from the infrastructure.
 * 
 * @author Costantino Perciante at ISTI-CNR (costantino.perciante@isti.cnr.it)
 */
public class LiferayJSONWsCredentials {

	private static final Logger logger = LoggerFactory.getLogger(LiferayJSONWsCredentials.class);

	// the singleton obj
	private static LiferayJSONWsCredentials singleton = new LiferayJSONWsCredentials();

	// properties that it contains
	private String host;
	private String schema;
	private String user;
	private String password;
	private int port;

	// The token of the user used to send notifications/messages when an application
	// token is provided. (will be read from web.xml)
	private String notifierUserToken;

	// Service endpoint properties

	private final static String RUNTIME_RESOURCE_NAME = "D4Science Infrastructure Gateway";
	private final static String CATEGORY = "Portal";
	private final static String END_POINT_NAME = "JSONWSUser";
	private final static boolean IS_ROOT_SERVICE = true;

	/**
	 * Private constructor
	 */
	private LiferayJSONWsCredentials() {
		logger.debug("Building LiferayJSONWsCredentials object");
		loadNotifierToken();
		lookupPropertiesFromIs();
		logger.debug("LiferayJSONWsCredentials object built");
	}

	/**
	 * Load the token of the notifier user
	 */
	private void loadNotifierToken() {
		try {
			notifierUserToken = ServletContextClass.getNotifierToken();
			logger.debug("Token read " + notifierUserToken.substring(0, 5) + "*********************");
		} catch (Exception e) {
			logger.error("Failed to read notifier user token!", e);
		}
	}

	/**
	 * Read the properties from the infrastructure
	 */
	private void lookupPropertiesFromIs() {
		Secret secret = InfrastructureUtils.getSecretForInfrastructure();

		IsServerConfig cfg;
		try {
			cfg = InfrastrctureServiceClient.serviceConfigFromIS(RUNTIME_RESOURCE_NAME, CATEGORY,
					END_POINT_NAME, IS_ROOT_SERVICE, secret);
		} catch (NotFoundException | ServerException e) {
			logger.error("Unable to retrieve such service endpoint information!", e);
			e.printStackTrace();
			return;
		}

		host = cfg.getServerUrl();
		schema = cfg.getProperty("schema");
		user = cfg.getProperty("username");

		password = cfg.getProperty("password");

		// TODO: check if needs to be decrypted
		// password = StringEncrypter.getEncrypter().decrypt(passsword);

		port = Integer.valueOf(cfg.getProperty("port"));

		logger.debug("Bean built " + toString());
	}

	/**
	 * Retrieve endpoints information from IS for DB
	 * 
	 * @return list of endpoints for ckan database
	 * @throws Exception
	 */
	private List<ServiceEndpoint> getConfigurationFromIS() throws Exception {

		SimpleQuery query = queryFor(ServiceEndpoint.class);
		query.addCondition("$resource/Profile/Name/text() eq '" + RUNTIME_RESOURCE_NAME + "'");
		query.addCondition("$resource/Profile/Category/text() eq '" + CATEGORY + "'");
		DiscoveryClient<ServiceEndpoint> client = clientFor(ServiceEndpoint.class);
		List<ServiceEndpoint> toReturn = client.submit(query);
		return toReturn;

	}

	public static LiferayJSONWsCredentials getSingleton() {
		return singleton;
	}

	public String getHost() {
		return host;
	}

	public String getSchema() {
		return schema;
	}

	public String getUser() {
		return user;
	}

	public String getPassword() {
		return password;
	}

	public int getPort() {
		return port;
	}

	public String getNotifierUserToken() {
		return notifierUserToken;
	}

	@Override
	public String toString() {
		return "LiferayJSONWsCredentials [host=" + host + ", schema=" + schema
				+ ", user=" + user + ", password=" + password + ", port="
				+ port + ", notifierUserToken=" + notifierUserToken + "]";
	}

}
