package org.gcube.smartgears.handler.resourceregistry;

import static java.util.concurrent.TimeUnit.SECONDS;
import static org.gcube.common.events.Observes.Kind.resilient;
import static org.gcube.smartgears.handlers.ProfileEvents.addToContext;
import static org.gcube.smartgears.handlers.ProfileEvents.removeFromContext;
import static org.gcube.smartgears.lifecycle.container.ContainerLifecycle.activation;
import static org.gcube.smartgears.lifecycle.container.ContainerLifecycle.failure;
import static org.gcube.smartgears.lifecycle.container.ContainerLifecycle.part_activation;
import static org.gcube.smartgears.lifecycle.container.ContainerLifecycle.shutdown;
import static org.gcube.smartgears.lifecycle.container.ContainerLifecycle.stop;
import static org.gcube.smartgears.lifecycle.container.ContainerState.active;
import static org.gcube.smartgears.utils.Utils.rethrowUnchecked;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.bind.annotation.XmlRootElement;

import org.gcube.common.authorization.client.proxy.AuthorizationProxy;
import org.gcube.common.authorization.library.provider.SecurityTokenProvider;
import org.gcube.common.events.Observes;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.informationsystem.model.impl.embedded.HeaderImpl;
import org.gcube.informationsystem.model.impl.relation.IsIdentifiedByImpl;
import org.gcube.informationsystem.model.reference.embedded.Header;
import org.gcube.informationsystem.model.reference.entity.Facet;
import org.gcube.informationsystem.model.reference.entity.Resource;
import org.gcube.informationsystem.model.reference.relation.ConsistsOf;
import org.gcube.informationsystem.model.reference.relation.IsIdentifiedBy;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceAvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relation.RelationAvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.client.ResourceRegistryClient;
import org.gcube.informationsystem.resourceregistry.client.ResourceRegistryClientFactory;
import org.gcube.informationsystem.resourceregistry.publisher.ResourceRegistryPublisher;
import org.gcube.informationsystem.resourceregistry.publisher.ResourceRegistryPublisherFactory;
import org.gcube.resourcemanagement.model.impl.entity.facet.CPUFacetImpl;
import org.gcube.resourcemanagement.model.impl.entity.facet.ContainerStateFacetImpl;
import org.gcube.resourcemanagement.model.impl.entity.facet.MemoryFacetImpl;
import org.gcube.resourcemanagement.model.impl.entity.facet.NetworkingFacetImpl;
import org.gcube.resourcemanagement.model.impl.entity.facet.SimplePropertyFacetImpl;
import org.gcube.resourcemanagement.model.impl.entity.facet.SoftwareFacetImpl;
import org.gcube.resourcemanagement.model.impl.entity.resource.HostingNodeImpl;
import org.gcube.resourcemanagement.model.impl.relation.consistsof.HasPersistentMemoryImpl;
import org.gcube.resourcemanagement.model.impl.relation.consistsof.HasVolatileMemoryImpl;
import org.gcube.resourcemanagement.model.reference.entity.facet.CPUFacet;
import org.gcube.resourcemanagement.model.reference.entity.facet.ContainerStateFacet;
import org.gcube.resourcemanagement.model.reference.entity.facet.MemoryFacet;
import org.gcube.resourcemanagement.model.reference.entity.facet.NetworkingFacet;
import org.gcube.resourcemanagement.model.reference.entity.facet.SimplePropertyFacet;
import org.gcube.resourcemanagement.model.reference.entity.facet.SoftwareFacet;
import org.gcube.resourcemanagement.model.reference.entity.facet.MemoryFacet.MemoryUnit;
import org.gcube.resourcemanagement.model.reference.entity.resource.HostingNode;
import org.gcube.resourcemanagement.model.reference.relation.consistsof.HasPersistentMemory;
import org.gcube.resourcemanagement.model.reference.relation.consistsof.HasVolatileMemory;
import org.gcube.smartgears.configuration.container.ContainerConfiguration;
import org.gcube.smartgears.configuration.library.SmartGearsConfiguration;
import org.gcube.smartgears.context.Property;
import org.gcube.smartgears.context.container.ContainerContext;
import org.gcube.smartgears.handlers.container.ContainerHandler;
import org.gcube.smartgears.handlers.container.ContainerLifecycleEvent.Start;
import org.gcube.smartgears.lifecycle.container.ContainerLifecycle;
import org.gcube.smartgears.provider.ProviderFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Manages the {@link HostingNode} {@link Resource} of the application.
 * <p>
 * The manager:
 * <ul>
 * <li>
 * creates the {@link HostingNode} {@link Resource} and the facets it
 * {@link ConsistsOf} when the container starts for the first time;</li>
 * <li>
 * update the {@link ContainerStateFacet} when the application becomes active,
 * and at any lifecycle change thereafter;</li>
 * <li>
 * schedule a periodic update of {@link Facet}s containing variables
 * information.</li>
 * </ul>
 * </p>
 * 
 * @author Luca Frosini
 */
@XmlRootElement(name = Constants.RESOURCE_MANAGEMENT)
public class HostingNodeManager extends ContainerHandler {

	private static Logger logger = LoggerFactory
			.getLogger(HostingNodeManager.class);

	public static final String MEMORY_TYPE = "memoryType";
	public static final String MEMORY_TYPE_RAM = "RAM";
	public static final String MEMORY_TYPE_JVM = "JVM";
	public static final String JVM_MAX_MEMORY = "jvmMaxMemory";

	public static final String MESSAGE = "message";

	private ContainerContext context;
	private AuthorizationProxy authorizationProxy;

	private ScheduledFuture<?> periodicUpdates;

	public HostingNodeManager() {
		super();
		this.authorizationProxy = ProviderFactory.provider()
				.authorizationProxy();
	}

	private void setContextFromToken(String token) {
		if (token == null || token.compareTo("") == 0) {
			SecurityTokenProvider.instance.reset();
			ScopeProvider.instance.reset();
		} else {
			SecurityTokenProvider.instance.set(token);
			String scope = getCurrentContextName(token);
			ScopeProvider.instance.set(scope);
		}

	}

	@Override
	public void onStart(Start e) {
		try{
			logger.info("onStart started");
			context = e.context();
			HostingNode hostingNode = instantiateHostingNode();
			hostingNode = publishHostingNode(hostingNode);
			registerObservers();
			schedulePeriodicUpdates();
			logger.info("onStart finished");
		}catch(Throwable re){
			logger.error("onStart failed",re);
		}
	}

	private void share(HostingNode hostingNode) {
		logger.trace("sharing {} {}", HostingNode.NAME, Resource.NAME);
		context.properties().add(
				new Property(Constants.HOSTING_NODE_PROPERTY, hostingNode));
	}

	private void registerObservers() {

		context.events().subscribe(new Object() {

			@Observes({ activation, part_activation, shutdown, stop, failure })
			void onChanged(ContainerLifecycle lc) {
				HostingNode hostingNode = context.properties().lookup(Constants.HOSTING_NODE_PROPERTY).value(HostingNode.class);
				updateHostingNode(hostingNode);
			}

			@Observes(value = addToContext)
			void addTo(String token) {
				HostingNode hostingNode = context.properties().lookup(Constants.HOSTING_NODE_PROPERTY).value(HostingNode.class);
				addToContext(hostingNode, token);
			}

			@Observes(value = removeFromContext)
			void removeFrom(String token) {
				HostingNode hostingNode = context.properties().lookup(Constants.HOSTING_NODE_PROPERTY).value(HostingNode.class);
				removeFromContext(hostingNode, token);
			}

		});
	}

	private void schedulePeriodicUpdates() {

		// register to cancel updates
		context.events().subscribe(

				new Object() {

					final ScheduledExecutorService service = Executors
							.newScheduledThreadPool(1);

					// we register it in response to lifecycle events so that we can
					// stop and resume along with application
					@Observes(value = { activation, part_activation }, kind = resilient)
					synchronized void restartPeriodicUpdates(ContainerLifecycle lc) {

						// already running
						if (periodicUpdates != null) {
							return;
						}

						if (lc.state() == active) {
							logger.info("scheduling periodic updates of {}",
									HostingNode.NAME);
						} else {
							logger.info("resuming periodic updates of {}",
									HostingNode.NAME);
						}

						final Runnable updateTask = new Runnable() {
							public void run() {
								HostingNode hostingNode = context.properties().lookup(Constants.HOSTING_NODE_PROPERTY).value(HostingNode.class);
								try {
									updateHostingNode(hostingNode);
								} catch (Exception e) {
									logger.error(
											"cannot complete periodic update of {}",
											HostingNode.NAME, e);
								}
							}
						};

						periodicUpdates = service
								.scheduleAtFixedRate(updateTask, 3, context
										.configuration().publicationFrequency(),
										SECONDS);

					}

					@Observes(value = { stop, failure, shutdown }, kind = resilient)
					synchronized void cancelPeriodicUpdates(ContainerLifecycle ignore) {

						if (periodicUpdates != null) {
							logger.trace("stopping periodic updates of {}",
									HostingNode.NAME);
							try {
								periodicUpdates.cancel(true);
								service.shutdownNow();
								periodicUpdates = null;
							} catch (Exception e) {
								logger.warn("could not stop periodic updates of {}",
										HostingNode.NAME, e);
							}
						}
					}

				});

	}

	private HostingNode publishHostingNode(HostingNode hostingNode) {
		ClassLoader contextCL = Thread.currentThread().getContextClassLoader();
		String previousToken = SecurityTokenProvider.instance.get();
		try {
			Thread.currentThread().setContextClassLoader(
					HostingNodeManager.class.getClassLoader());

			boolean create = true;

			List<String> startTokens = context.configuration().startTokens();
			for (String token : startTokens) {
				setContextFromToken(token);

				try {
					ResourceRegistryPublisher resourceRegistryPublisher = ResourceRegistryPublisherFactory
							.create();

					if (create) {

						try {
							hostingNode = resourceRegistryPublisher
									.createResource(hostingNode);
							share(hostingNode);

							// TODO Add a Reference to Site
							/*
							 * node.profile().newSite().country(cfg.site().country
							 * ()).location(cfg.site ().location())
							 * .latitude(cfg
							 * .site().latitude()).longitude(cfg.site
							 * ().longitude
							 * ()).domain(domainIn(cfg.hostname()));
							 */

						} catch (ResourceAlreadyPresentException e) {
							ResourceRegistryClient registryClient = ResourceRegistryClientFactory
									.create();
							hostingNode = registryClient.getInstance(
									HostingNode.class, hostingNode.getHeader()
									.getUUID());
							updateHostingNode(hostingNode);
						} catch (ResourceAvailableInAnotherContextException e) {
							addToContext(hostingNode, token);
						}

						create = false;

					} else {
						addToContext(hostingNode, token);
					}

				} catch (Exception e) {
					logger.error("Unable to add {} to current context ({})",
							hostingNode, getCurrentContextName(token), e);
				}

			}

		} catch (Throwable e) {
			rethrowUnchecked(e);
		} finally {
			setContextFromToken(previousToken);
			Thread.currentThread().setContextClassLoader(contextCL);
		}
		logger.info("hosting node published");
		return hostingNode;
	}

	private String getCurrentContextName(String token) {
		try {
			return this.authorizationProxy.get(token).getContext();
		} catch (Exception e) {
			logger.error("Error retrieving context form token {}, it should never happen",
					token, e);
			return null;
		}
	}

	private void removeFromContext(HostingNode hostingNode, String token) {
		ClassLoader contextCL = Thread.currentThread().getContextClassLoader();
		String previousToken = SecurityTokenProvider.instance.get();
		try {
			Thread.currentThread().setContextClassLoader(
					HostingNodeManager.class.getClassLoader());
			setContextFromToken(token);

			boolean removed = false;

			ResourceRegistryPublisher resourceRegistryPublisher = 
					ResourceRegistryPublisherFactory.create();
			removed = resourceRegistryPublisher.
					removeResourceFromCurrentContext(hostingNode);

			if (removed) {
				logger.info(
						"{} successfully removed from current context ({})",
						hostingNode, getCurrentContextName(token));
				share(hostingNode);
			} else {
				logger.error("Unable to remove {} from current context ({})",
						hostingNode, getCurrentContextName(token));
			}

			share(hostingNode);

		} catch (Exception e) {
			logger.error("Unable to remove {} from current context ({})",
					hostingNode, getCurrentContextName(token), e);
			rethrowUnchecked(e);
		} finally {
			setContextFromToken(previousToken);
			Thread.currentThread().setContextClassLoader(contextCL);
		}
	}

	private void addToContext(HostingNode hostingNode, String token) {
		ClassLoader contextCL = Thread.currentThread().getContextClassLoader();
		String previousToken = SecurityTokenProvider.instance.get();
		try {
			Thread.currentThread().setContextClassLoader(
					HostingNodeManager.class.getClassLoader());
			setContextFromToken(token);

			ResourceRegistryPublisher resourceRegistryPublisher = ResourceRegistryPublisherFactory
					.create();
			boolean added = resourceRegistryPublisher
					.addResourceToCurrentContext(hostingNode);

			if (added) {
				logger.info("{} successfully added to current context ({})",
						hostingNode, getCurrentContextName(token));
				share(hostingNode);
			} else {
				logger.error("Unable to add {} to current context ({})",
						hostingNode, getCurrentContextName(token));
			}

			share(hostingNode);

		} catch (Exception e) {
			logger.error("Unable to add {} to current context ({})",
					hostingNode, getCurrentContextName(token), e);
			rethrowUnchecked(e);
		} finally {
			setContextFromToken(previousToken);
			Thread.currentThread().setContextClassLoader(contextCL);
		}
	}

	private HostingNode updateHostingNode(HostingNode hostingNode) {
		ClassLoader contextCL = Thread.currentThread().getContextClassLoader();
		String previousToken = SecurityTokenProvider.instance.get();
		try {

			if (previousToken == null) {
				setContextFromToken((String) context.configuration()
						.startTokens().toArray()[0]);
			}
			Thread.currentThread().setContextClassLoader(
					HostingNodeManager.class.getClassLoader());

			hostingNode = updateFacets(hostingNode);

			share(hostingNode);

		} catch (Exception e) {
			rethrowUnchecked(e);
		} finally {
			setContextFromToken(previousToken);
			Thread.currentThread().setContextClassLoader(contextCL);
		}
		return hostingNode;

	}

	private HostingNode updateFacets(HostingNode hostingNode) throws ResourceRegistryException {
		logger.debug("Updating HostingNode");

		ResourceRegistryPublisher resourceRegistryPublisher = ResourceRegistryPublisherFactory
				.create();

		ContainerStateFacet containerStateFacet = null;
		MemoryFacet ramFacet = null;
		MemoryFacet jvmMemoryFacet = null;
		MemoryFacet disk = null;

		List<ConsistsOf<? extends Resource, ? extends Facet>> consistsOfToRemove = new ArrayList<>();

		List<ConsistsOf<? extends Resource, ? extends Facet>> consistsOfList = hostingNode
				.getConsistsOf();
		for (ConsistsOf<? extends Resource, ? extends Facet> c : consistsOfList) {
			if (c.getTarget() instanceof ContainerStateFacet) {
				containerStateFacet = (ContainerStateFacet) c.getTarget();
				containerStateFacet = getContainerStateFacet(containerStateFacet);
				continue;
			}

			if (c instanceof HasVolatileMemory) {
				String memoryType = (String) c
						.getAdditionalProperty(MEMORY_TYPE);
				if (memoryType.compareTo(MEMORY_TYPE_RAM) == 0) {
					ramFacet = (MemoryFacet) c.getTarget();
					ramFacet = getRamInfo(ramFacet);
					continue;
				}

				if (memoryType.compareTo(MEMORY_TYPE_JVM) == 0) {
					jvmMemoryFacet = (MemoryFacet) c.getTarget();
					jvmMemoryFacet = getJVMMemoryInfo(jvmMemoryFacet);
					continue;
				}

			}

			if (c instanceof HasPersistentMemory) {
				disk = (MemoryFacet) c.getTarget();
				disk = getDiskSpace(disk);
				continue;
			}

			consistsOfToRemove.add(c);

		}

		// Resource Update has effect only on specified facet.
		// Removing the one that have not to be changed
		consistsOfList.removeAll(consistsOfToRemove);

		try {
			hostingNode = resourceRegistryPublisher.updateResource(hostingNode);
		} catch (ResourceNotFoundException e) {
			/* Update failed trying to recreate it */
			// ReAdding the removed relations to recreate all 
			consistsOfList.addAll(consistsOfToRemove);
			hostingNode = resourceRegistryPublisher.createResource(hostingNode);
		} catch (ResourceAvailableInAnotherContextException | RelationAvailableInAnotherContextException e) {
			addToContext(hostingNode, SecurityTokenProvider.instance.get());
			hostingNode = resourceRegistryPublisher.updateResource(hostingNode);
		} catch (ResourceRegistryException e) {
			logger.error("error trying to publish hosting node",e);
		}

		return hostingNode;
	}

	private HostingNode instantiateHostingNode() {
		logger.info("Creating {} {}", HostingNode.NAME, Resource.NAME);

		ContainerConfiguration containerConfiguration = context.configuration();

		UUID uuid = UUID.fromString(this.context.id());
		HostingNode hostingNode = new HostingNodeImpl();
		Header header = new HeaderImpl(uuid);
		hostingNode.setHeader(header);

		NetworkingFacet networkingFacet = new NetworkingFacetImpl();
		try {
			networkingFacet.setIPAddress(InetAddress.getLocalHost()
					.getHostAddress());
		} catch (UnknownHostException e) {
			logger.warn("unable to detect the IP address of the host");
		}
		String hostname = containerConfiguration.hostname();
		networkingFacet.setHostName(hostname);
		networkingFacet.setDomainName(getDomain(hostname));

		networkingFacet.setAdditionalProperty("Port",
				containerConfiguration.port());
		IsIdentifiedBy<HostingNode, NetworkingFacet> isIdentifiedBy = new IsIdentifiedByImpl<>(
				hostingNode, networkingFacet, null);
		hostingNode.addFacet(isIdentifiedBy);

		List<CPUFacet> cpuFacets = getCPUFacets();
		for (CPUFacet cpuFacet : cpuFacets) {
			hostingNode.addFacet(cpuFacet);
		}

		SoftwareFacet softwareFacet = new SoftwareFacetImpl();
		OperatingSystemMXBean mxbean = ManagementFactory
				.getOperatingSystemMXBean();
		softwareFacet.setGroup(mxbean.getName()); // softwareFacet.setGroup(System.getProperty("os.name"));
		softwareFacet.setName(mxbean.getArch()); // softwareFacet.setName(System.getProperty("os.arch"));
		softwareFacet.setVersion(mxbean.getVersion()); // softwareFacet.setName(System.getProperty("os.version"));
		hostingNode.addFacet(softwareFacet);

		SimplePropertyFacet simplePropertyFacet = addEnvironmentVariables();
		hostingNode.addFacet(simplePropertyFacet);

		ContainerStateFacet containerStateFacet = getContainerStateFacet(null);
		hostingNode.addFacet(containerStateFacet);

		MemoryFacet ramFacet = getRamInfo(null);
		HasVolatileMemory<HostingNode, MemoryFacet> hasVolatileRAMMemory = new HasVolatileMemoryImpl<HostingNode, MemoryFacet>(
				hostingNode, ramFacet, null);
		hasVolatileRAMMemory
		.setAdditionalProperty(MEMORY_TYPE, MEMORY_TYPE_RAM);
		hostingNode.addFacet(hasVolatileRAMMemory);

		MemoryFacet jvmMemoryFacet = getJVMMemoryInfo(null);
		HasVolatileMemory<HostingNode, MemoryFacet> hasVolatileJVMMemory = new HasVolatileMemoryImpl<HostingNode, MemoryFacet>(
				hostingNode, jvmMemoryFacet, null);
		hasVolatileJVMMemory
		.setAdditionalProperty(MEMORY_TYPE, MEMORY_TYPE_JVM);
		hostingNode.addFacet(hasVolatileJVMMemory);

		MemoryFacet diskFacet = getDiskSpace(null);
		HasPersistentMemory<HostingNode, MemoryFacet> hasPersistentMemory = new HasPersistentMemoryImpl<HostingNode, MemoryFacet>(
				hostingNode, diskFacet, null);
		hostingNode.addFacet(hasPersistentMemory);
		logger.info("hostingNode instanciated");
		return hostingNode;
	}

	private ContainerStateFacet getContainerStateFacet(
			ContainerStateFacet containerStateFacet) {
		if (containerStateFacet == null) {
			containerStateFacet = new ContainerStateFacetImpl();
		}
		containerStateFacet.setValue(context.lifecycle().state().remoteForm());
		return containerStateFacet;
	}

	private MemoryFacet getDiskSpace(MemoryFacet memoryFacet) {
		if (memoryFacet == null) {
			memoryFacet = new MemoryFacetImpl();
		}

		long free = 0;
		long total = 0;
		try {
			FileStore fileStore = Files.getFileStore(Paths.get(context
					.configuration().persistence().location()));
			free = fileStore.getUsableSpace() / 1048576; // 1048576 = 1024*1024
			// user to convert
			// bytes in MByte
			total = fileStore.getTotalSpace() / 1048576; // 1048576 = 1024*1024
			// user to convert
			// bytes in MByte
		} catch (IOException ioe) {
			logger.warn("Unable to detect disk space information", ioe);
			memoryFacet.setAdditionalProperty(MESSAGE,
					"Unable to detect disk space information.");
		}

		memoryFacet.setUnit(MemoryUnit.MB);
		memoryFacet.setSize(total);
		memoryFacet.setUsed(total - free);

		return memoryFacet;
	}

	@SuppressWarnings("restriction")
	private MemoryFacet getRamInfo(MemoryFacet memoryFacet) {
		if (memoryFacet == null) {
			memoryFacet = new MemoryFacetImpl();
		}

		OperatingSystemMXBean mxbean = ManagementFactory
				.getOperatingSystemMXBean();
		com.sun.management.OperatingSystemMXBean sunmxbean = (com.sun.management.OperatingSystemMXBean) mxbean;
		long freeMemory = sunmxbean.getFreePhysicalMemorySize() / 1048576; // in
		// MB
		long totalMemory = sunmxbean.getTotalPhysicalMemorySize() / 1048576; // in
		// MB

		memoryFacet.setUnit(MemoryUnit.MB);
		memoryFacet.setSize(totalMemory);
		memoryFacet.setUsed(totalMemory - freeMemory);

		return memoryFacet;
	}

	private MemoryFacet getJVMMemoryInfo(MemoryFacet memoryFacet) {
		if (memoryFacet == null) {
			memoryFacet = new MemoryFacetImpl();
		}

		/* 1048576 = 1024*1024 used to convert bytes in MByte */
		long jvmFreeMemory = Runtime.getRuntime().freeMemory() / 1048576;
		long jvmTotalMemory = Runtime.getRuntime().totalMemory() / 1048576;
		long jvmMaxMemory = Runtime.getRuntime().maxMemory() / 1048576;

		memoryFacet.setUnit(MemoryUnit.MB);
		memoryFacet.setSize(jvmTotalMemory);
		memoryFacet.setUsed(jvmTotalMemory - jvmFreeMemory);
		memoryFacet.setAdditionalProperty(JVM_MAX_MEMORY, jvmMaxMemory);

		return memoryFacet;
	}

	private static String sanitizeKey(String key) {
		return key.trim().replace(" ", "_");
	}

	private SimplePropertyFacet addEnvironmentVariables() {

		ContainerConfiguration cfg = context.configuration();

		Map<String, String> map = new HashMap<String, String>();
		map.putAll(cfg.properties());
		map.putAll(System.getenv());

		SimplePropertyFacet simplePropertyFacet = new SimplePropertyFacetImpl();
		simplePropertyFacet.setName("ENVIRONMENT_VARIABLES");
		simplePropertyFacet.setValue("");

		for (Map.Entry<String, String> entry : map.entrySet()) {
			String varname = entry.getKey();
			if ((varname.compareToIgnoreCase("CLASSPATH") == 0)
					|| (varname.compareToIgnoreCase("PATH") == 0)
					|| (varname.contains("SSH")) || (varname.contains("MAIL"))
					|| (varname.compareToIgnoreCase("LS_COLORS") == 0)) {
				continue;
			}

			simplePropertyFacet.setAdditionalProperty(
					sanitizeKey(entry.getKey()), entry.getValue());

		}

		simplePropertyFacet.setAdditionalProperty("Java",
				System.getProperty("java.version"));
		SmartGearsConfiguration config = ProviderFactory.provider()
				.smartgearsConfiguration();

		simplePropertyFacet.setAdditionalProperty("SmartGears",
				config.version());
		simplePropertyFacet.setAdditionalProperty(
				"ghn-update-interval-in-secs",
				String.valueOf(cfg.publicationFrequency()));

		return simplePropertyFacet;
	}

	private static String getDomain(String hostname) {
		try {
			Pattern pattern = Pattern
					.compile("([0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3})");
			Matcher regexMatcher = pattern.matcher(hostname);
			if (regexMatcher.matches()) { // it's an IP address, nothing to trim
				return hostname;
			}
			return hostname.substring(hostname.indexOf(".") + 1);
		} catch (Exception e) {
			logger.warn("Error while getting domain from hostname");
			return hostname;
		}
	}

	public static final String CPU_PROCESSOR = "processor";
	public static final String CPU_VENDOR_ID = "vendor_id";
	public static final String CPU_MODEL_NAME = "model name";
	public static final String CPU_CPU_MHZ = "cpu MHz";
	public static final String CPU_MODEL_T = "model\t";
	public static final String CPU_MODEL_B = "model\b";
	public static final String CPU_MODEL_NUMBER = "modelNumber";

	public static List<CPUFacet> getCPUFacets() {

		List<CPUFacet> cpuFacets = new ArrayList<>();

		File file = new File("/proc/cpuinfo");

		if (!file.exists()) {
			logger.warn("cannot acquire CPU info (no /proc/cpuinfo)");
			return cpuFacets;
		}

		BufferedReader input = null;

		try {
			input = new BufferedReader(new FileReader(file));

			String line = null;

			CPUFacet cpuFacet = null;

			while ((line = input.readLine()) != null) {

				if ((line.startsWith(CPU_PROCESSOR))) { // add the current
					// processor to the map
					cpuFacet = new CPUFacetImpl();
					cpuFacets.add(cpuFacet);
				}

				try {
					if (line.contains(CPU_VENDOR_ID)) {
						cpuFacet.setVendor(line.split(":")[1].trim());
						continue;
					}
				} catch (Exception e) {
					continue;
				}

				try {
					if (line.contains(CPU_MODEL_NAME)) {
						cpuFacet.setModel(line.split(":")[1].trim());
						continue;
					}
				} catch (Exception e) {
					continue;
				}

				try {
					if (line.contains(CPU_CPU_MHZ)) {
						cpuFacet.setClockSpeed(line.split(":")[1].trim());
						continue;
					}
				} catch (Exception e) {
					continue;
				}

				try {
					if ((line.contains(CPU_MODEL_T))
							|| (line.contains(CPU_MODEL_B))) {
						cpuFacet.setAdditionalProperty(CPU_MODEL_NUMBER,
								line.split(":")[1].trim());
						continue;
					}
				} catch (Exception e) {
					continue;
				}

				try {
					String[] nameValue = line.split(":");
					cpuFacet.setAdditionalProperty(sanitizeKey(nameValue[0]),
							line.split(":")[1].trim());
				} catch (Exception e) {

				}

			}
		} catch (Exception e) {
			logger.warn("unable to acquire CPU info", e);
		} finally {
			if (input != null) {
				try {
					input.close();
				} catch (IOException e) {
					logger.warn("unable to close stream", e);
				}
			}
		}
		return cpuFacets;
	}

	@Override
	public String toString() {
		return Constants.RESOURCE_MANAGEMENT;
	}
}
