package org.gcube.vremanagement.virtualplatform.tomcat;

import java.io.File;
import java.io.FileFilter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.List;

import org.apache.catalina.ant.DeployTask;
import org.apache.catalina.ant.StartTask;
import org.apache.catalina.ant.StopTask;
import org.apache.catalina.ant.UndeployTask;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.gcube.vremanagement.virtualplatform.model.DeployedPackage;
import org.gcube.vremanagement.virtualplatform.model.PackageSet;
import org.gcube.vremanagement.virtualplatform.model.TargetPlatform;
import org.gcube.vremanagement.virtualplatform.model.UndeployedPackage;

/**
 * An abstraction over the Tomcat container. 
 * Performs deployment, activation, deactivation and undeployment of Web Applications.s
 * 
 * @author Manuele Simi (ISTI-CNR), @author Roberto Cirillo (ISTI CNR)
 *
 */
public class Container extends TargetPlatform<Webapp> {
	
	String user;
	String password;
	String manager;
	URL baseURL;
	static final String platform = "TOMCAT";
	static final int platformVersion = 6;
	static final int platformMinorVersion = 0;

	//default constructor for reflection instantiation
	public Container() {}
	
	/**
	 * {@inheritDoc}
	 */	
	@Override
	public DeployedPackage deploy(Webapp app) throws Exception {
		DeployedWebapp deployedApp = new DeployedWebapp(app);
		try {
			DeployTask task = new DeployTask();
			task.setProject(new Project());
			task.setUsername(user);
			task.setPassword(password);
			task.setWar(app.getFile().getAbsolutePath());
			task.setPath(app.getTargetPath());
			task.setUrl(manager);
			task.setDescription(app.getDescription());
			task.execute();
		} catch (Exception e) {
			for (StackTraceElement st : e.getStackTrace()) {
				System.err.println("Container says: " + st);	
			}
			throw new Exception(e);
		}
		List<String> sourceEprs = app.getEntrypoints();
		String[] eprs = new String[sourceEprs.size()+1];
		//root as first entry point
		eprs[0] = baseURL.toString() + ((app.getTargetPath().startsWith("/")) ?app.getTargetPath():"/"+app.getTargetPath());
		//then the others
		for (int i = 0; i < sourceEprs.size(); i++)
			eprs[i+1] = eprs[0] + ((sourceEprs.get(i).startsWith("/")) ? sourceEprs.get(i): "/" +sourceEprs.get(i));
		deployedApp.setEndpoints(eprs);
		deployedApp.setSuccess(true);
		return deployedApp;
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public UndeployedPackage undeploy(Webapp app) throws Exception {
		UndeployedWebapp undeployedApp = new UndeployedWebapp(app);
		try {
			UndeployTask task = new UndeployTask();
			task.setProject(new Project());
			task.setUrl(manager);
			task.setUsername(user);
			task.setPassword(password);
			task.setPath(app.getTargetPath());
			task.execute();
		} catch (BuildException e) {
			throw new Exception(e);		
		}
		undeployedApp.setSuccess(true);
		return undeployedApp;
	}


	/**
	 * {@inheritDoc}
	 */
	public String getPlatform() {
		return platform;
	}


	/**
	 * {@inheritDoc}
	 */
	public int getPlatformVersion() {
		return platformVersion;
	}


	/**
	 * {@inheritDoc}
	 */
	public int getPlatformMinorVersion() {
		return platformMinorVersion;
	}


	/**
	 * {@inheritDoc}
	 */
	public void setUser(String user) {
		this.user = user;
	}


	/**
	 * {@inheritDoc}
	 */
	public void setPassword(String password) {
		this.password = password;
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBaseURL(URL baseURL) {
		this.baseURL = baseURL;
		this.manager = baseURL.toString() + "/manager";
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public Class<Webapp> getResourceClass() {
		return Webapp.class;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean activate(Webapp resource) throws Exception {
		StartTask task = new StartTask();
		task.setPath(resource.getTargetPath());
		task.setUsername(this.user);
		task.setPassword(this.password);
		task.setUrl(this.manager);
		try {
			task.execute();
		} catch (BuildException e) {
			throw new Exception(e);		
		}
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean deactivate(Webapp resource) throws Exception {
		StopTask task = new StopTask();
		task.setPath(resource.getTargetPath());
		task.setUsername(this.user);
		task.setPassword(this.password);
		task.setUrl(this.manager);
		try {
			task.execute();
		} catch (BuildException e) {
			throw new Exception(e);		
		}
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public PackageSet<Webapp> list() throws Exception {
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void initialize() throws Exception {
		//not implemented
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	 public FileFilter getAcceptedFilesFilter() {
			return new FileFilter() {	    
			    public boolean accept(File file) {
			    	if (file.getAbsolutePath().endsWith(".war")) {
						if (file.exists()) 
							return true;
			    	}
					return false;
			    }
		};
	 }

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void shutdown() throws Exception {
		//not implemented
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean supportHotDeployment() {
		return true;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isAvailable() {
		try {
		      HttpURLConnection.setFollowRedirects(false);
		      HttpURLConnection con =
		         (HttpURLConnection) this.baseURL.openConnection();
		      con.setRequestMethod("HEAD");
		      return (con.getResponseCode() == HttpURLConnection.HTTP_OK);
		    }
		    catch (Exception e) {
		    	e.printStackTrace();
		    	System.err.println("failed to check if tomcat is available");
		       return false;
		    }
	}

}
