package org.gcube.portlets.user.geoexplorer.server;

import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.servlet.http.HttpSession;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.gcube.common.geoserverinterface.GeoCaller;
import org.gcube.common.geoserverinterface.GeoCaller.FILTER_TYPE;
import org.gcube.common.geoserverinterface.GeonetworkCommonResourceInterface.GeoserverMethodResearch;
import org.gcube.common.geoserverinterface.bean.CswLayersResult;
import org.gcube.common.geoserverinterface.bean.LayerCsw;
import org.gcube.portlets.user.geoexplorer.client.GeoExplorerService;
import org.gcube.portlets.user.geoexplorer.client.beans.CapabilitiesResult;
import org.gcube.portlets.user.geoexplorer.client.beans.GroupItem;
import org.gcube.portlets.user.geoexplorer.client.beans.LayerItem;
import org.gcube.portlets.user.geoexplorer.client.Constants;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import com.extjs.gxt.ui.client.Style.SortDir;
import com.extjs.gxt.ui.client.data.BasePagingLoadResult;
import com.extjs.gxt.ui.client.data.FilterConfig;
import com.extjs.gxt.ui.client.data.FilterPagingLoadConfig;
import com.extjs.gxt.ui.client.data.PagingLoadResult;
import com.extjs.gxt.ui.client.data.SortInfo;
import com.google.gwt.user.server.rpc.RemoteServiceServlet;

/**
 * The server side implementation of the RPC service.
 */

@SuppressWarnings("serial")
public abstract class GeoExplorerServiceImpl extends RemoteServiceServlet implements GeoExplorerService {

	protected abstract GeoExplorerServiceParameters getParameters(HttpSession httpSession) throws Exception;
	protected static final GeoserverMethodResearch researchMethod = GeoserverMethodResearch.MOSTUNLOAD;

	private static final String LAYERS_LOADED = "layersLoaded";
	private static final String FUTURE_LAYERS_RESULT = "futureCswLayersResult";
	private static final String LAYERS_RESULT = "layersResult";

	private static final Object TITLE_FIELD = "title";

	private GeoCaller geoCaller = null;
	//private List<LayerItem> layerItems; 

	private String referredWorkspace;
	
	protected GeoCaller getGeoCaller() throws Exception {
		HttpSession httpSession = this.getThreadLocalRequest().getSession();
		return getGeoCaller(httpSession);
	}

	protected GeoCaller getGeoCaller(HttpSession httpSession) throws Exception {
		
		if (geoCaller!=null) return geoCaller;
		try {

			GeoExplorerServiceParameters parameters = getParameters(httpSession);
			String geoserverUrl = parameters.getGeoServerUrl();
			String geonetworkUrl = parameters.getGeoNetworkUrl();
			String gnUser = parameters.getGeoNetworkUser();
			String gnPwd = parameters.getGeoNetworkPwd();
			String gsUser = parameters.getGeoServerUser();
			String gsPwd = parameters.getGeoServerPwd();

			geoCaller = new GeoCaller(geonetworkUrl, gnUser, gnPwd, geoserverUrl, gsUser, gsPwd, researchMethod);

			return geoCaller;
		} catch (Exception e) {
			throw new Exception("Error initializing the GeoCaller", e);
		}
	}


	/* (non-Javadoc)
	 * @see org.gcube.portlets.user.geoexplorer.client.GeoExplorerService#initGeoExplorer()
	 */
	@Override
	public String initGeoExplorer(final String referredWorkspace) {
		this.referredWorkspace = referredWorkspace;
		try {
			final HttpSession session = this.getThreadLocalRequest().getSession();
			//session.setAttribute(LAYERS_LOADED, new Boolean(false));

			// cancel eventual previous task
			Future<List<LayerItem>> previousTask = (Future<List<LayerItem>>)session.getAttribute(FUTURE_LAYERS_RESULT);
			if (previousTask!=null) {
				if (!Constants.reloadCswAtStartup)
					return "ok";
				else
					previousTask.cancel(true);
			}

			// define the executor
			ExecutorService executor = Executors.newSingleThreadExecutor();

			// create an asyncronous task 
			Callable<List<LayerItem>> task = new Callable<List<LayerItem>>() {
				@Override
				public List<LayerItem> call() throws Exception {
					try{
						System.out.println("Load layers from csw...");
						GeoCaller geoCaller = getGeoCaller(session);

						// this csw query loads all layers and require much time
						CswLayersResult result = geoCaller.getLayersFromCsw(referredWorkspace);
						List<LayerItem> layerItems = getLayerItemsFromLayersCsw(result.getLayers());
						System.out.println("Loading layers from csw terminated.");
						return layerItems;
					}catch (Exception e) {
						System.err.println("Error getting layers");
						e.printStackTrace();
						throw e;
					}
				}
			};

			// and wrap in a future object
			Future<List<LayerItem>> futureResult = executor.submit(task);
			session.setAttribute(FUTURE_LAYERS_RESULT, futureResult);

			//			// get result (block the execution)
			//			System.out.println("thread: loading layers...");
			//			CswLayersResult result = futureResult.get();			
			//			System.out.println("thread: layers loaded.");
			//			
			//			session.setAttribute(LAYERS_LOADED, new Boolean(true));
			//			session.setAttribute(LAYERS_RESULT, result);
			//			session.setAttribute(FUTURE_LAYERS_RESULT, null);
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
		return "ok";
	}


	public PagingLoadResult<LayerItem> getLayers(final FilterPagingLoadConfig config) {
		SortInfo sortInfo = config.getSortInfo();
		final String sortField = sortInfo.getSortField();
		SortDir sortDir = sortInfo.getSortDir();
		List<FilterConfig> filters = config.getFilterConfigs();

		//List<FilterConfig> filters = config.getFilterConfigs();

		HttpSession session = this.getThreadLocalRequest().getSession();
		List<LayerItem> layerItems=null;

		try {
			Future<List<LayerItem>> futureLayersResult = (Future<List<LayerItem>>)session.getAttribute(FUTURE_LAYERS_RESULT);
			if (futureLayersResult==null)
				initGeoExplorer("aquamaps"); // TODO

			// check if is possible to load data via csw query when  
			// if total results are not ready and 
			if (
					// check if total results are not ready 
					!futureLayersResult.isDone() 
					// check if sorting is with title or no sorting is used
					&& (sortField==null || sortField.equals(TITLE_FIELD))
					// check if filtering is with title or no filtering is used
					&& (filters.size()==0 || (filters.size()==1 && filters.get(0).getField().equals(TITLE_FIELD)))
			) {				
				// in this case it's not necessary get a csw request of all layers
				int start = config.getOffset()+1; // in csw query records start with 1
				int maxRecords = config.getLimit();
				boolean sortByTitle = sortField==null ? false : (sortField.equals(TITLE_FIELD));
				boolean sortAscendent = (sortDir!=SortDir.DESC);
				FILTER_TYPE filterType = (filters.size()==0 ? FILTER_TYPE.NO_FILTER : FILTER_TYPE.TITLE);
				String textToSearch = (filters.size()==0 ? null : (String)filters.get(0).getValue());

				GeoCaller geoCaller = getGeoCaller();

				CswLayersResult result = geoCaller.getLayersFromCsw(this.referredWorkspace, start, maxRecords,
						sortByTitle, sortAscendent, filterType, textToSearch);
				layerItems = getLayerItemsFromLayersCsw(result.getLayers());
				return new BasePagingLoadResult<LayerItem>(layerItems, config.getOffset(), result.getResultLayersCount());
			}

			// if total results are ready get them
			layerItems = futureLayersResult.get(60, TimeUnit.SECONDS);

		} catch (TimeoutException e) {
			e.printStackTrace();
			return null;
		} catch (InterruptedException e) {
			e.printStackTrace();
			return null;
		} catch (ExecutionException e) {
			e.printStackTrace();
			return null;
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}

		if (sortField != null) {
			Collections.sort(layerItems, config.getSortInfo().getSortDir().comparator(new Comparator<LayerItem>() {
				public int compare(LayerItem l1, LayerItem l2) {
					if (sortField.equals("title")) {
						return l1.getTitle().compareTo(l2.getTitle());
					} else if (sortField.equals("geoserverUrl")) {
						return l1.getGeoserverUrl().compareTo(l2.getGeoserverUrl());
					} else if (sortField.equals("name")) {
						return l1.getName().compareTo(l2.getName());
					}
					return 0;
				}
			}));
		}

		// apply filter
		ArrayList<LayerItem> layersToRemove = new ArrayList<LayerItem>();

		for (FilterConfig f : filters) {
			Object ov = f.getValue();
			String c = f.getComparison();
			for (LayerItem l : layerItems) {
				Object value = l.get(f.getField());
				if (f.isFiltered(l, ov, c, value))
					layersToRemove.add(l);
			}
		}

		List<LayerItem> filteredLayers = new ArrayList<LayerItem>();
		//		for (LayerItem l : layerItems)
		//			filteredLayers.add(l);
		filteredLayers.addAll(layerItems);
		for (LayerItem l : layersToRemove)
 			filteredLayers.remove(l);

		List<LayerItem> sublist = new ArrayList<LayerItem>();
		int start = config.getOffset();
		int limit = filteredLayers.size();
		if (config.getLimit() > 0) {
			limit = Math.min(start + config.getLimit(), limit);
		}
		for (int i = config.getOffset(); i < limit; i++)
			sublist.add(filteredLayers.get(i));

		return new BasePagingLoadResult<LayerItem>(sublist, config.getOffset(), filteredLayers.size());
	}


	public List<LayerItem> getLayerItemsFromLayersCsw(List<LayerCsw> layersCsw) {
		List<LayerItem> layerItems = new ArrayList<LayerItem>();
		for (LayerCsw l : layersCsw) {
			String layer = l.getName();
			layer = (layer==null)?"Unknown":layer;
			
			String[] splitName = layer.split(":");
			String name = (splitName.length==2 ? splitName[1] : layer);
			String geoserverUrl = l.getGeoserverUrl();

			// remove each string after "?"
			int index = geoserverUrl.indexOf("?");
			if (index!=-1)
				geoserverUrl = geoserverUrl.substring(0, geoserverUrl.indexOf("?"));
			// remove suffix "/wms" or "/wms/"
			geoserverUrl = geoserverUrl.replaceFirst("(/wms)$", "").replaceFirst("(/wms/)$", "");

			layerItems.add(new LayerItem(l.getUuid(), name, layer, l.getTitle(), geoserverUrl));
		}
		return layerItems;
	}


	//public PagingLoadResult<LayerInfo> getLayerInfos(final FilterPagingLoadConfig config) {		
	public List<LayerItem> getLayerInfos(final FilterPagingLoadConfig config) {		
		GeoCaller geoCaller=null;
		try {
			geoCaller = getGeoCaller();
			CswLayersResult result = geoCaller.getLayersFromCsw(this.referredWorkspace, 1, 10);

			List<LayerItem> layerItems = new ArrayList<LayerItem>();
			for (LayerCsw l : result.getLayers()) {
				String layer = l.getName();
				String[] splitName = layer.split(":");
				String name = (splitName.length==2 ? splitName[1] : layer);
				layerItems.add(new LayerItem(l.getUuid(), name, layer, l.getTitle(), l.getGeoserverUrl()));
			}

			return layerItems;
			//return new BasePagingLoadResult<LayerInfo>(layerInfos, config.getOffset(), 100);
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	public PagingLoadResult<LayerItem> getLayersWithGetCapabilities(final FilterPagingLoadConfig config) {
		List<LayerItem>layerItems = loadLayerItems(true);

		if (config.getSortInfo().getSortField() != null) {
			final String sortField = config.getSortInfo().getSortField();
			if (sortField != null) {
				Collections.sort(layerItems, config.getSortInfo().getSortDir().comparator(new Comparator<LayerItem>() {
					public int compare(LayerItem l1, LayerItem l2) {
						if (sortField.equals("title")) {
							return l1.getTitle().compareTo(l2.getTitle());
						} else if (sortField.equals("geoserverUrl")) {
							return l1.getGeoserverUrl().compareTo(l2.getGeoserverUrl());
						} else if (sortField.equals("name")) {
							return l1.getName().compareTo(l2.getName());
						}
						return 0;
					}
				}));
			}
		}

		// apply filter
		ArrayList<LayerItem> filteredLayers = new ArrayList<LayerItem>();
		for (LayerItem l : layerItems)
			filteredLayers.add(l);

		ArrayList<LayerItem> layersToRemove = new ArrayList<LayerItem>();

		List<FilterConfig> filters = config.getFilterConfigs();
		for (FilterConfig f : filters) {
			Object ov = f.getValue();
			String c = f.getComparison();
			for (LayerItem l : layerItems) {
				Object value = l.get(f.getField());
				if (f.isFiltered(l, ov, c, value))
					layersToRemove.add(l);
			}
		}

		for (LayerItem l : layersToRemove)
			filteredLayers.remove(l);


		List<LayerItem> sublist = new ArrayList<LayerItem>();
		int start = config.getOffset();
		int limit = filteredLayers.size();
		if (config.getLimit() > 0) {
			limit = Math.min(start + config.getLimit(), limit);
		}
		for (int i = config.getOffset(); i < limit; i++)
			sublist.add(filteredLayers.get(i));

		return new BasePagingLoadResult<LayerItem>(sublist, config.getOffset(), filteredLayers.size());
	}

	/**
	 * @param b 
	 * 
	 */
	@SuppressWarnings("unchecked")
	private List<LayerItem> loadLayerItems(boolean forceLoad) {
		if (!forceLoad) {
			HttpSession session = this.getThreadLocalRequest().getSession();
			List<LayerItem> layerItems = (List<LayerItem>)session.getAttribute("layerItems");
			if (layerItems!=null) {
				System.out.println("\nSESSION FOUND, NO LOAD\n");
				return layerItems;
			}
		}

		List<LayerItem> layerItems = new ArrayList<LayerItem>();
		GeoCaller geoCaller=null;
		try {
			geoCaller = getGeoCaller();
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}

		List<String> geoserverUrls = geoCaller.getWmsGeoserverList();

		for (String geoserverUrl: geoserverUrls) {
			CapabilitiesResult result = getCapabilitiesFromUrl(geoserverUrl);
			layerItems.addAll(result.getLayerItems());
		}

		if (!forceLoad) {
			// saving into session
			System.out.println("\nNEW SESSION, LAYERS SAVED IN SESSION\n");
			this.getThreadLocalRequest().getSession().setAttribute("layerItems", layerItems);
		}

		return layerItems;
	}


	public enum State {
		START,
		INSIDE_CAPABILITY,
		INSIDE_TOP_LAYER,
		INSIDE_LAYER,
		INSIDE_NAME,
		INSIDE_TITLE,
		INSIDE_BOUNDING_BOX,
		INSIDE_STYLE,
		INSIDE_STYLE_NAME,
		INSIDE_STYLE_TITLE
	};
	/**
	 * @param geoserverUrl
	 * @return
	 */
	private CapabilitiesResult getCapabilitiesFromUrl(final String geoserverUrl) {
		try {
			SAXParserFactory factory = SAXParserFactory.newInstance();
			SAXParser saxParser = factory.newSAXParser();

			final CapabilitiesResult result = new CapabilitiesResult();
			result.setTotalLayers(0);
			//			final List<LayerItem> layerItems = new ArrayList<LayerItem>();
			//			final List<GroupItem> groupItems = new ArrayList<GroupItem>();

			DefaultHandler handler = new DefaultHandler() {
				State state = State.START;
				String name;
				String layer;
				String title;
				boolean isLayer=true;
				boolean endTitle=false;
				boolean endName=false;
				boolean endBoundingBox=false;

				public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {

					switch (state) {
						case START:
							if (qName.equalsIgnoreCase("Capability"))
								state = State.INSIDE_CAPABILITY;
							//System.out.println("CAPABILITY: ");
							break;
						case INSIDE_CAPABILITY:
							if (qName.equalsIgnoreCase("Layer"))
								state = State.INSIDE_TOP_LAYER;
							//System.out.println("TOP LAYER: ");
							break;
						case INSIDE_TOP_LAYER:
							if (qName.equalsIgnoreCase("Layer"))
								state = State.INSIDE_LAYER;
							//layerItem = new LayerItem();
							//System.out.println("LAYER: ");
							break;
						case INSIDE_LAYER:
							if (qName.equalsIgnoreCase("name"))
								state = State.INSIDE_NAME;
							else if (qName.equalsIgnoreCase("title"))
								state = State.INSIDE_TITLE;
							else if (qName.equalsIgnoreCase("BoundingBox"))
								state = State.INSIDE_BOUNDING_BOX;						
							break;
					}
				}

				public void endElement(String uri, String localName, String qName) throws SAXException {
					switch (state) {
						case INSIDE_NAME:
							if (qName.equalsIgnoreCase("Name")) {
								state = State.INSIDE_LAYER;
								endName = true;
							}
							break;
						case INSIDE_TITLE:
							if (qName.equalsIgnoreCase("Title")) {
								state = State.INSIDE_LAYER;
								endTitle = true;
							}
							break;
						case INSIDE_BOUNDING_BOX:
							if (qName.equalsIgnoreCase("BoundingBox")) {
								state = State.INSIDE_LAYER;
								endBoundingBox = true;
							}
							break;
						case INSIDE_LAYER:
							if (qName.equalsIgnoreCase("Layer")) {
								state = State.INSIDE_TOP_LAYER;
								endName = endTitle = endBoundingBox = false;

								// object end, add to list
								if (isLayer)
									result.addLayer(new LayerItem(name, layer, title, geoserverUrl));
								else
									result.addGroup(new GroupItem(name, title));
							}
							break;
						case INSIDE_TOP_LAYER:
							if (qName.equalsIgnoreCase("Layer"))
								state = State.INSIDE_CAPABILITY;
							break;
						case INSIDE_CAPABILITY:
							if (qName.equalsIgnoreCase("Capability"))
								state = State.START;
							break;
					}
				}

				public void characters(char ch[], int start, int length) throws SAXException {
					switch (state) {
						case INSIDE_NAME:
							if (!endName)
								setName(new String(ch, start, length));
							//System.out.println("Title: " + new String(ch, start, length)+"\n");
							break;
						case INSIDE_TITLE:
							if (!endTitle)
								setTitle(new String(ch, start, length));
							//System.out.println("Name: " + new String(ch, start, length));
							break;
					}
				}

				private void setName(String _name) {
					try {
						String[] split = _name.split(":", 2);
						if (split.length==2) {
							isLayer=true;
							layer = _name;
							name = split[1];
						} else {
							isLayer=false;
							name = _name;
						}
					} catch (Exception e) {
						//e.printStackTrace();
					}
				}

				private void setTitle(String _title) {
					title = _title;
				}
			};

			String strUrl = geoserverUrl+"/wms?request=getCapabilities";
			URL url = new URL(strUrl);
			saxParser.parse(url.openConnection().getInputStream(), handler);
			return result;

		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}
}
