/**
 * 
 */
package org.gcube.portlets.user.tdtemplate.server.converter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import org.gcube.data.analysis.tabulardata.commons.templates.model.Template;
import org.gcube.data.analysis.tabulardata.commons.templates.model.TemplateCategory;
import org.gcube.data.analysis.tabulardata.commons.templates.model.columns.ColumnCategory;
import org.gcube.data.analysis.tabulardata.commons.templates.model.columns.ColumnDescription;
import org.gcube.data.analysis.tabulardata.commons.templates.model.columns.TemplateColumn;
import org.gcube.data.analysis.tabulardata.commons.utils.DimensionReference;
import org.gcube.data.analysis.tabulardata.commons.utils.LocaleReference;
import org.gcube.data.analysis.tabulardata.commons.utils.TimeDimensionReference;
import org.gcube.data.analysis.tabulardata.commons.webservice.types.OnRowErrorAction;
import org.gcube.data.analysis.tabulardata.commons.webservice.types.TemplateDescription;
import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.model.datatype.DataType;
import org.gcube.data.analysis.tabulardata.model.metadata.Locales;
import org.gcube.data.analysis.tabulardata.model.time.PeriodType;
import org.gcube.portlets.user.td.gwtservice.shared.tr.ColumnData;
import org.gcube.portlets.user.td.widgetcommonevent.shared.TRId;
import org.gcube.portlets.user.tdtemplate.server.TemplateUpdaterForDescription;
import org.gcube.portlets.user.tdtemplate.server.service.ExpressionEvaluatorInstance;
import org.gcube.portlets.user.tdtemplate.server.service.TemplateService;
import org.gcube.portlets.user.tdtemplate.server.session.CacheServerExpressions;
import org.gcube.portlets.user.tdtemplate.server.validator.ColumnCategoryConstraint;
import org.gcube.portlets.user.tdtemplate.server.validator.service.ColumnCategoryTemplateValidator;
import org.gcube.portlets.user.tdtemplate.shared.SPECIAL_CATEGORY_TYPE;
import org.gcube.portlets.user.tdtemplate.shared.TdColumnDefinition;
import org.gcube.portlets.user.tdtemplate.shared.TdTColumnCategory;
import org.gcube.portlets.user.tdtemplate.shared.TdTDataType;
import org.gcube.portlets.user.tdtemplate.shared.TdTTemplateType;
import org.gcube.portlets.user.tdtemplate.shared.TdTemplateDefinition;
import org.gcube.portlets.user.tdtemplate.shared.TdTemplateUpdater;
import org.gcube.portlets.user.tdtemplate.shared.TemplateExpression;
import org.gcube.portlets.user.tdtemplate.shared.util.CutStringUtil;
import org.gcube.portlets.user.tdtemplate.shared.validator.ViolationDescription;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * @author Francesco Mangiacrapa francesco.mangiacrapa@isti.cnr.it
 * @Jan 17, 2014
 *
 */
public class ConverterToTdTemplateModel {
	
	
	public static Logger logger = LoggerFactory.getLogger(ConverterToTdTemplateModel.class);
	
	/**
	 * 
	 * @return
	 */
	public static List<TdTTemplateType> getTdTTemplateTypeFromTemplateCategoryValues(){
		
		List<TdTTemplateType> templates = new ArrayList<TdTTemplateType>(TemplateCategory.values().length);
		for (TemplateCategory template : TemplateCategory.values()) {
			
			List<ViolationDescription> constraint = getTemplateCategoryConstraint(template);
			templates.add(new TdTTemplateType(template.name(), template.name(), constraint));
		}
		
		logger.info("Returning "+templates.size()+" template types");
		
		return templates;
		
	}
	
	public static List<ViolationDescription> getTemplateCategoryConstraint(TemplateCategory cat) {

		/*switch (cat) {
		
			case CODELIST: return "A Codelist must have one (and only one) Code column";
	
			case DATASET: return "A Dataset must have at least one dimension (reference to a CodeList) and at least one value column";
			
			case GENERIC: return "";
			
			default: return "";
		}*/
		
		List<ViolationDescription> violations  = new ArrayList<ViolationDescription>();
		
		ColumnCategoryTemplateValidator templateValidator = null;
		 
		switch (cat) {
			case CODELIST:
				templateValidator = new ColumnCategoryTemplateValidator(TemplateCategory.CODELIST);
				break;
			case DATASET:
				templateValidator = new ColumnCategoryTemplateValidator(TemplateCategory.DATASET);
				break;
			case GENERIC:
				templateValidator = new ColumnCategoryTemplateValidator(TemplateCategory.GENERIC);
				break;

			default:
				break;
		}

		if(templateValidator!=null){
			HashMap<String, ColumnCategoryConstraint> hash  = templateValidator.getTemplateColumnValidator();
			
			if(hash!=null){
				for (String columnType : hash.keySet()) {
					ColumnCategoryConstraint col = hash.get(columnType);
					violations.add(new ViolationDescription(columnType, col.getConstraintDescription()));
				}
			}
			
		}
		
		return violations;

	}
	
	
	/**
	 * 
	 * @return
	 */
	public static List<String> getOnErrorValues(){
		
		List<String> onErrors = new ArrayList<String>(OnRowErrorAction.values().length);
		for (OnRowErrorAction error : OnRowErrorAction.values()) {
			onErrors.add(error.toString());
		}
		
		logger.info("Returning "+onErrors.size()+" on errors types");
		return onErrors;
	}
	
	/**
	 * 
	 * @return
	 * @throws Exception 
	 */
	public static List<TdTColumnCategory> getTdTColumnCategoryFromTemplateCategory(TemplateCategory template) throws Exception{
		
		if(template==null)
			throw new Exception("TemplateCategory is null");
		
		List<ColumnDescription> listColumns = template.getAllowedColumn();
		if(listColumns==null)
			throw new Exception("ColumnCategory allowed columns is null for template "+template);
		
		
		List<TdTColumnCategory> columns = new ArrayList<TdTColumnCategory>(listColumns.size());
		for (ColumnDescription  col: listColumns) {
			List<Class<? extends DataType>> dataTypes = col.getColumnCategory().getAllowedClasses();
			TdTColumnCategory tdtCC = new TdTColumnCategory(col.getColumnCategory().name(), col.getColumnCategory().name());
			tdtCC.setTdtDataType(getTdTDataTypeFromListDataType(dataTypes));
			columns.add(tdtCC);
		}
		
		logger.info("Returning "+columns.size()+" ColumnCategory");
		
		return columns;
		
	}
	
	
	/**
	 * 
	 * @return
	 * @throws Exception 
	 */
	public static TdTColumnCategory getTdTColumnCategoryFromColumnCategory(ColumnCategory col) throws Exception{
		
		if(col==null)
			throw new Exception("ColumnCategory is null");
		
		return new TdTColumnCategory(col.name(), col.name());
		
	}
	
	public static List<TdTDataType> getTdTDataTypeFromListDataType(List<Class<? extends DataType>> dataTypes) throws Exception{
		
		List<TdTDataType> listTdTDataType  = new ArrayList<TdTDataType>();
		
		if(dataTypes==null)
			throw new Exception("List of data type is null");
		
		for (Class<? extends DataType> class1 : dataTypes) {
			try{
				listTdTDataType.add(getTdTDataTypeFromDataType(class1));
			}catch (Exception e) {
				logger.error("Errror on converting data type, skypping value",e);
			}
		}
		
		logger.info("Returning "+listTdTDataType.size()+" TdTDataType");
		
		return listTdTDataType;
	}
	
	
	public static TdTDataType getTdTDataTypeFromDataType(Class<? extends DataType> dataType) throws Exception{
		
		if(dataType==null)
			throw new Exception("Data type is null");
		
		String purgedSimpleName = CutStringUtil.stringPurgeSuffix(dataType.getSimpleName(), "Type");
		
		logger.info("\t \t DataType: "+dataType + " purged simple name: " +purgedSimpleName);
		
		return new TdTDataType(dataType.getName(), purgedSimpleName);

	}
	
	public List<String> getOnErrorActions(){
		//TODO
		return null;
	}
	
	public static TemplateUpdaterForDescription getTdTemplateUpdaterFromTemplateDescription(TemplateDescription sdescr, TemplateService service, CacheServerExpressions cache){
		
		if(sdescr==null)
			return null;
		
		logger.info("Converting TemplateDescription...");
		TdTemplateDefinition tdTempDefinition = new TdTemplateDefinition();
		tdTempDefinition.setServerId(sdescr.getId());
		tdTempDefinition.setAgency(sdescr.getAgency());
		tdTempDefinition.setTemplateDescription(sdescr.getDescription());
		tdTempDefinition.setTemplateName(sdescr.getName());
		
		Template stm = sdescr.getTemplate();
		
		logger.info("Converting Template...");
		List<TdColumnDefinition> listTdColumnDef = new ArrayList<TdColumnDefinition>();
		
		if(stm!=null){
			
			//ON ERROR
			if(stm.getOnRowErrorAction()!=null)
				tdTempDefinition.setOnError(stm.getOnRowErrorAction().toString());
			
			//CATEGORY
			TemplateCategory scat = stm.getCategory();
			if(scat!=null){
				tdTempDefinition.setTemplateType(scat.name());
				List<ViolationDescription> constraint = getTemplateCategoryConstraint(scat);
				tdTempDefinition.setTemplateType(new TdTTemplateType(scat.name(), scat.name(), constraint));
			}
			
			List<TemplateColumn<?>> sCol = stm.getColumns();
			
			if(sCol!=null){
				logger.info("Found column size: "+sCol.size());
				for (int i=0; i<sCol.size(); i++) {
					TemplateColumn<?> templateColumn = sCol.get(i);
					
					//ColumnCategory conversion
					ColumnCategory sColCat = templateColumn.getColumnType(); //SERVER 
					TdTColumnCategory tdtCC = new TdTColumnCategory(sColCat.name(), sColCat.name()); //WIDGET
					
					TdTDataType tdDataType = null;
					//Value type conversion
					Class<? extends DataType> sValueType = templateColumn.getValueType(); //SERVER
					try {
						tdDataType = getTdTDataTypeFromDataType(sValueType); //WIDGET
					}catch (Exception e) {
						logger.error("Errror on converting data type, skypping value",e);
					}
					
					
					//EXPRESSION TODO??
					List<Expression> sExpression = templateColumn.getExpressions();
					List<TemplateExpression> listExpressionExtend = null;
					
					if(sExpression!=null && sExpression.size()>0){
						listExpressionExtend = new ArrayList<TemplateExpression>(sExpression.size());
						
						ExpressionEvaluatorInstance evaluator = new ExpressionEvaluatorInstance(service);
						for (Expression expression : sExpression) {
							
							if(expression!=null){
								logger.info("Found expression : "+expression.toString());
								TemplateExpression cExpression = null;
								try{	
									cExpression = new TemplateExpression();
									cExpression.setServerExpression(expression.toString());
									cExpression.setHumanDescription(evaluator.evaluate(expression));
									listExpressionExtend.add(cExpression);
									logger.info("Updating expression cached: "+expression.toString());
									cache.addExpression(expression);
								}catch (Exception e) {
									logger.error("Expression evalutation exception: ",e);
								}
							}
	//						System.out.println(j +") " +expression.toString());
						}
					}

					TdColumnDefinition tdColumn = new TdColumnDefinition(i, tdtCC, tdDataType, SPECIAL_CATEGORY_TYPE.UNKNOWN);
					tdColumn = settingSpecificReference(templateColumn, tdColumn);
					
					if(listExpressionExtend!=null){
						logger.info("Adding expression description, expression size: "+listExpressionExtend.size());
						tdColumn.setRulesExtends(listExpressionExtend);
					}
					listTdColumnDef.add(tdColumn);
				}
			}
		}
		
		TdTemplateUpdater tdUpdater = new TdTemplateUpdater(tdTempDefinition, listTdColumnDef);
		TemplateUpdaterForDescription updaterDescription = new TemplateUpdaterForDescription(tdUpdater, cache);
		logger.info("Returning TemplateUpdaterForDescription "+updaterDescription);
		return updaterDescription;
	}
	
	/**
	 * 
	 * @param templateColumn
	 * @param tdColumn
	 * @return
	 */
	public static TdColumnDefinition settingSpecificReference(TemplateColumn<?> templateColumn, TdColumnDefinition tdColumn){
		
		if(templateColumn==null)
			return tdColumn;

		Object refence = templateColumn.getReference();
		
		//USED IN CODELIST
		if(refence instanceof LocaleReference){
			logger.info("Found column reference as LocaleReference..");
			LocaleReference locale = (LocaleReference) templateColumn.getReference();
			if(locale!=null){
				logger.info("Setting locale as "+locale.getLocale());
				tdColumn.setLocale(locale.getLocale());
//				tdColumn.setSpecialCategoryType(SPECIAL_CATEGORY_TYPE.CODENAME);
			}
		}else if(refence instanceof TimeDimensionReference){
			logger.info("Found column reference as TimeDimensionReference..");
			TimeDimensionReference timeDimension = (TimeDimensionReference) templateColumn.getReference();
			if(timeDimension!=null && timeDimension.getPeriod()!=null){
				logger.info("Setting timeDimension as "+timeDimension.getPeriod().name());
				tdColumn.setTimePeriod(timeDimension.getPeriod().name());
//				tdColumn.setSpecialCategoryType(SPECIAL_CATEGORY_TYPE.TIMEDIMENSION);
			}
		}else if(refence instanceof DimensionReference){
			logger.info("Found column reference as DimensionReference..");
			DimensionReference dimension = (DimensionReference) templateColumn.getReference();
			if(dimension!=null){
				
				String columnId = dimension.getColumnId()!=null?dimension.getColumnId().getValue():null;
				Long tableId = dimension.getTableId()!=null?dimension.getTableId().getValue():null;
				if(columnId!=null && tableId!=null){

					TRId trId = new TRId("", tableId+"");
					logger.info("Get column for Dimension with TRid "+trId +" and column id: "+columnId);
					ColumnData tempColData = new ColumnData();
					tempColData.setColumnId(columnId);
					tempColData.setTrId(trId);
					logger.info("Setting Column Data as "+tempColData);
					tdColumn.setColumnData(tempColData);
//					tdColumn.setSpecialCategoryType(SPECIAL_CATEGORY_TYPE.DIMENSION);
				}
			}
		}
		
		return tdColumn;
		
		
	}
	
	public static List<String> getAllowedLocales(){
		return new ArrayList<String>(Arrays.asList(Locales.ALLOWED_LOCALES));
	}

	/**
	 * @return
	 */
	public static List<String> getTimeDimensionPeriodTypes() {
		
		List<String> listDataTypes = new ArrayList<String>(PeriodType.values().length);
		for (PeriodType period : PeriodType.values())
			listDataTypes.add(period.name());
		
		return listDataTypes;
	}


}
