/**
 * 
 */
package org.gcube.portlets.user.tdwx.datasource.td;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.gcube.application.framework.core.session.ASLSession;
import org.gcube.data.analysis.tabulardata.commons.utils.AuthorizationProvider;
import org.gcube.data.analysis.tabulardata.commons.utils.AuthorizationToken;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.query.parameters.QueryOrder;
import org.gcube.data.analysis.tabulardata.query.parameters.QueryOrderDirection;
import org.gcube.data.analysis.tabulardata.query.parameters.QueryPage;
import org.gcube.data.analysis.tabulardata.service.TabularDataService;
import org.gcube.data.analysis.tabulardata.service.exception.NoSuchTableException;
import org.gcube.data.analysis.tabulardata.service.impl.TabularDataServiceFactory;
import org.gcube.portlets.user.tdwx.datasource.td.map.ColumnDefinitionBuilder;
import org.gcube.portlets.user.tdwx.server.datasource.DataSourceX;
import org.gcube.portlets.user.tdwx.server.datasource.DataSourceXException;
import org.gcube.portlets.user.tdwx.server.datasource.Direction;
import org.gcube.portlets.user.tdwx.server.datasource.util.TableJSonBuilder;
import org.gcube.portlets.user.tdwx.shared.model.ColumnDefinition;
import org.gcube.portlets.user.tdwx.shared.model.TableDefinition;
import org.gcube.portlets.user.tdwx.shared.model.TableId;
import org.gcube.portlets.user.tdwx.shared.model.ValueType;
import org.json.JSONArray;
import org.json.JSONException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * @author "Giancarlo Panichi" <a
 *         href="mailto:g.panichi@isti.cnr.it">g.panichi@isti.cnr.it</a>
 * 
 */
public class TDXDataSource implements DataSourceX {

	public static final String JSON_ROWS_FIELD = "ROWS";
	public static final String JSON_TOTAL_LENGTH_FIELD = "total";
	public static final String JSON_OFFSET_FIELD = "offset";

	protected Logger logger = LoggerFactory.getLogger(TDXDataSource.class);
	protected String dataSourceFactoryId;
	protected String tableName;
	protected TableDefinition tableDefinition;
	protected int tableSize = -1;
	protected ColumnDefinition autogeneratePrimaryColumn = null;
	protected TableJSonBuilder jsonBuilder;

	protected TabularDataService service;
	protected org.gcube.data.analysis.tabulardata.model.table.TableId serviceTableId;
	protected org.gcube.data.analysis.tabulardata.model.table.Table serviceTable;

	/**
	 * 
	 * @param dataSourceFactoryId
	 * @param tableName
	 * @return
	 */
	public static TDXDataSource createTDDataSource(String dataSourceFactoryId,
			ASLSession aslSession, String tableName)
			throws DataSourceXException {
		TDXDataSource dataSource = new TDXDataSource(dataSourceFactoryId,
				aslSession, tableName);
		return dataSource;
	}

	/**
	 * 
	 * @param dataSourceFactoryId
	 * @param tableName
	 * @throws DataSourceXException
	 */
	public TDXDataSource(String dataSourceFactoryId, ASLSession aslSession,
			String tableName) throws DataSourceXException {
		if (dataSourceFactoryId == null) {
			logger.error("An error occurred, dataSourceFactoryId is null");
			throw new DataSourceXException(
					"An error occurred, dataSourceFactoryId is null");
		}

		if (tableName == null) {
			logger.error("An error occurred, tableName is null");
			throw new DataSourceXException(
					"An error occurred, tableName is null");
		}

		this.dataSourceFactoryId = dataSourceFactoryId;
		this.tableName = tableName;

		AuthorizationProvider.instance.set(new AuthorizationToken(aslSession
				.getUsername()));
		service = TabularDataServiceFactory.getService();

		long tableId;

		try {
			tableId = Long.parseLong(tableName);
		} catch (NumberFormatException e) {
			logger.error("An error occurred, tableName is not a long", e);
			throw new DataSourceXException(
					"An error occurred, no tableName is not a long", e);
		}

		serviceTableId = new org.gcube.data.analysis.tabulardata.model.table.TableId(
				tableId);

		try {
			serviceTable = service.getTable(serviceTableId);
		} catch (NoSuchTableException e) {
			logger.error("An error occurred, no such table", e);
			throw new DataSourceXException("An error occurred, no such table",
					e);
		}

		logger.debug("Service Table: " + serviceTable);

	}

	public String getDataSourceFactoryId() {
		return dataSourceFactoryId;
	}

	public TableDefinition getTableDefinition() throws DataSourceXException {

		if (tableDefinition == null)
			tableDefinition = extractTableDefinition();
		logger.debug("Retrieved table definition");
		return tableDefinition;
	}

	/**
	 * 
	 * @return
	 * @throws DataSourceXException
	 */
	protected TableDefinition extractTableDefinition()
			throws DataSourceXException {
		List<ColumnDefinition> columns = getColumnDefinitions();
		
		logger.debug("Creating tableId...");
		TableId id = new TableId(dataSourceFactoryId, tableName);

		TableDefinition tableDefinition = new TableDefinition(id, tableName,
				JSON_ROWS_FIELD, JSON_TOTAL_LENGTH_FIELD, JSON_OFFSET_FIELD,
				columns);

		tableDefinition.setModelKeyColumnId("id");
		logger.debug("TableDefinition Created");
		return tableDefinition;

	}

	protected List<ColumnDefinition> getColumnDefinitions()
			throws DataSourceXException {
		logger.debug("Creating list of columns definition...");
		List<Column> serviceListColumn = serviceTable.getColumns();
		ArrayList<ColumnDefinition> columns = new ArrayList<ColumnDefinition>();

		int i = 0;
		Column serviceColumn;
		for (; i < serviceListColumn.size(); i++) {
			serviceColumn = serviceListColumn.get(i);
			ColumnDefinition column = getColumnDefinition(serviceColumn, i);
			columns.add(column);
		}
		logger.debug("List of columns definition created");
		return columns;
	}

	protected ColumnDefinition getColumnDefinition(Column serviceColumn,
			int ordinalPosition) {
		ColumnDefinitionBuilder columnDefinitionBuilder = new ColumnDefinitionBuilder(
				serviceColumn, ordinalPosition);
		ColumnDefinition columnDefinition = columnDefinitionBuilder.build();

		logger.debug("Column Definition:" + columnDefinition);

		return columnDefinition;
	}

	protected ColumnDefinition createPrimaryKeyColumn(
			List<ColumnDefinition> columns) {
		List<String> ids = new ArrayList<String>(columns.size());
		for (ColumnDefinition column : columns)
			ids.add(column.getId());

		String id = "idColumn";
		for (int i = 0; ids.contains(id); id = "idColumn" + i++)
			;

		return new ColumnDefinition(id, id, id, ValueType.INTEGER, -1, false,
				false,
				org.gcube.portlets.user.tdwx.shared.model.ColumnType.SYSTEM);
	}

	public String getDataAsJSon(int start, int limit, String sortingColumn,
			Direction direction) throws DataSourceXException {
		logger.trace("getDataAsJSon start: " + start + " limit: " + limit
				+ " sortingColumn: " + sortingColumn + " direction: "
				+ direction);
		tableSize = 0;
		try {
			tableSize = service.getQueryLenght(serviceTableId, null);
		} catch (NoSuchTableException e) {
			logger.error("An error occurred, tableSize is not recovered", e);
			throw new DataSourceXException(
					"An error occurred, tableSize is not recovered", e);
		}
		start = Math.max(0, start);
		start = Math.min(start, tableSize);
		if (start + limit > tableSize)
			limit = tableSize - start;
		logger.trace("checked bounds start: " + start + " limit: " + limit);

		TableDefinition tableDefinition = getTableDefinition();
		logger.debug("Creating queryOrder...");
		QueryOrder queryOrder = null;
		if (sortingColumn != null) {
			if (tableDefinition.getColumns().get(sortingColumn) == null) {
				logger.error("The specified sorting column \"" + sortingColumn
						+ "\" don't exists");

				throw new DataSourceXException(
						"The specified sorting column \"" + sortingColumn
								+ "\" don't exists");
			} else {
				ColumnDefinition columnDefinition = tableDefinition
						.getColumns().get(sortingColumn);
				Column column = serviceTable.getColumnByName(columnDefinition
						.getId());
				switch (direction) {
				case ASC:
					queryOrder = new QueryOrder(column.getLocalId(),
							QueryOrderDirection.ASCENDING);
					break;
				case DESC:
					queryOrder = new QueryOrder(column.getLocalId(),
							QueryOrderDirection.DESCENDING);
					break;
				default:
					break;
				}
			}
		} else {

		}

		String json = getJSon(start, queryOrder);
		logger.trace("Returning json");
		logger.debug(json);
		return json;

	}

	protected ArrayList<ColumnDefinition> sort(
			Collection<ColumnDefinition> columns) {

		ArrayList<ColumnDefinition> lcolumns = new ArrayList<ColumnDefinition>();
		for (ColumnDefinition column : columns) {
			lcolumns.add(column);
		}

		Collections.sort(lcolumns, new Comparator<ColumnDefinition>() {
			@Override
			public int compare(ColumnDefinition cd1, ColumnDefinition cd2) {
				int comp = 0;
				if (cd1.getPosition() == cd2.getPosition()) {
					comp = 0;
				} else {
					if (cd1.getPosition() > cd2.getPosition()) {
						comp = 1;
					} else {
						comp = -1;
					}
				}
				return comp;
			}
		});

		return lcolumns;

	}

	protected String getJSon(int start, QueryOrder queryOrder)
			throws DataSourceXException {
		logger.debug("Retrieving JSon");
		TableJSonBuilder json = getBuilder();

		TableDefinition tableDefinition = getTableDefinition();
		logger.debug("Retrieved tableDefinition");
		Collection<ColumnDefinition> columns = tableDefinition.getColumns()
				.values();
		logger.debug("Retrieved Columns");
		ArrayList<ColumnDefinition> lcolumns = sort(columns);

		logger.debug("ColumnDefinition:\n" + lcolumns.toString());

		json.startRows();

		int id = start;

		QueryPage queryPage = new QueryPage(start, 200);
		logger.debug("Created queryPage");
		String serviceJson = null;
		try {
			if (queryOrder == null) {
				serviceJson = service.queryAsJson(serviceTableId, queryPage);
			} else {
				serviceJson = service.queryAsJson(serviceTableId, queryPage,
						queryOrder);
			}
		} catch (NoSuchTableException e) {
			logger.error("An error occurred, no such table", e);
			throw new DataSourceXException("An error occurred, no such table",
					e);
		} catch (Throwable e) {
			logger.error("An error occurred", e);
			throw new DataSourceXException("An error occurred", e);
		}
		logger.debug("Created serviceJson");
		JSONArray currentRow = null;
		int i = -1;
		int j = -1;
		int totalRows = -1;

		try {
			org.json.JSONObject obj = new org.json.JSONObject(serviceJson);
			org.json.JSONArray rows = obj.getJSONArray("rows");

			totalRows = rows.length();
			logger.debug("Reading rows from json");
			for (i = 0; i < totalRows; i++) {

				json.startRow();
				currentRow = rows.getJSONArray(i);
				j = 0;
				for (ColumnDefinition column : lcolumns) {

					String columnId = column.getId();

					if (currentRow.isNull(j)) {
						json.addValue(columnId, "");
					} else {

						switch (column.getValueType()) {
						//TODO
						/*case DATE:
							String date = currentRow.getString(j);
							json.addValue(columnId, date);
							break;*/
						case BOOLEAN:
							Boolean b = currentRow.getBoolean(j);
							json.addValue(columnId, b);
							break;
						case DOUBLE:
							Double d = currentRow.getDouble(j);
							json.addValue(columnId, d);
							break;
						case FLOAT:
							Double f = currentRow.getDouble(j);
							json.addValue(columnId, f);
							break;
						case INTEGER:
							int integ = currentRow.getInt(j);
							json.addValue(columnId, integ);
							break;
						case LONG:
							Long l = currentRow.getLong(j);
							json.addValue(columnId, l);
							break;
						case STRING:
							String s = currentRow.getString(j);
							json.addValue(columnId, s);
							break;
						default:
							logger.warn("Unknow value type "
									+ column.getValueType());
						}
					}
					j++;
				}

				json.endRow();
			}
			id += i;

		} catch (JSONException e) {

			logger.error("An error occurred while parsing json document\n"
					+ "At Row " + i + ",Column " + j + "\nRow Content: "
					+ currentRow + "\nLenght rows " + totalRows, e);
			throw new DataSourceXException(
					"An error occurred,  while reading json of service", e);
		}

		json.endRows();

		json.setTotalLength(tableSize);

		json.setOffset(start);

		logger.trace("produced " + (id - start) + " rows");

		json.close();

		return json.toString();
	}

	protected TableJSonBuilder getBuilder() throws DataSourceXException {
		try {
			if (jsonBuilder == null) {
				TableDefinition tdef = getTableDefinition();
				logger.debug("Creating jsonBuilder...");
				if (tdef != null) {
					jsonBuilder = new TableJSonBuilder(tdef);
				} else {
					logger.error("table definition is null");
					throw new DataSourceXException("table definition is null");
				}
			} else {
				jsonBuilder.clean();
			}
			return jsonBuilder;
		} catch (Exception e) {
			logger.debug("Error Creating jsonBuilder: "+e.getMessage());
			throw new DataSourceXException("Error Creating jsonBuilder: "+e.getMessage());
		}
	}

	protected int getTableSize() throws DataSourceXException {
		return tableSize;
	}

	public void close() {
		// The service is stateless there is no need to close

	}

}
