package org.gcube.portlets.user.td.expressionwidget.server;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.expression.MultivaluedExpression;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.Equals;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.GreaterOrEquals;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.GreaterThan;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.LessOrEquals;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.LessThan;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.NotEquals;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.NotGreater;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.NotLess;
import org.gcube.data.analysis.tabulardata.expression.composite.text.TextBeginsWith;
import org.gcube.data.analysis.tabulardata.expression.composite.text.TextContains;
import org.gcube.data.analysis.tabulardata.expression.composite.text.TextEndsWith;
import org.gcube.data.analysis.tabulardata.expression.composite.text.TextMatchSQLRegexp;
import org.gcube.data.analysis.tabulardata.expression.leaf.ColumnReferencePlaceholder;
import org.gcube.data.analysis.tabulardata.expression.leaf.ConstantList;
import org.gcube.data.analysis.tabulardata.expression.leaf.Range;
import org.gcube.data.analysis.tabulardata.expression.leaf.TypedColumnReference;
import org.gcube.data.analysis.tabulardata.expression.logical.And;
import org.gcube.data.analysis.tabulardata.expression.logical.Between;
import org.gcube.data.analysis.tabulardata.expression.logical.IsNotNull;
import org.gcube.data.analysis.tabulardata.expression.logical.IsNull;
import org.gcube.data.analysis.tabulardata.expression.logical.Not;
import org.gcube.data.analysis.tabulardata.expression.logical.Or;
import org.gcube.data.analysis.tabulardata.expression.logical.ValueIsIn;
import org.gcube.data.analysis.tabulardata.model.column.ColumnLocalId;
import org.gcube.data.analysis.tabulardata.model.datatype.BooleanType;
import org.gcube.data.analysis.tabulardata.model.datatype.DataType;
import org.gcube.data.analysis.tabulardata.model.datatype.DateType;
import org.gcube.data.analysis.tabulardata.model.datatype.GeometryType;
import org.gcube.data.analysis.tabulardata.model.datatype.IntegerType;
import org.gcube.data.analysis.tabulardata.model.datatype.NumericType;
import org.gcube.data.analysis.tabulardata.model.datatype.TextType;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDBoolean;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDDate;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDInteger;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDNumeric;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDText;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDTypeValue;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.portlets.user.td.expressionwidget.shared.expression.C_MultivaluedExpression;
import org.gcube.portlets.user.td.expressionwidget.shared.model.composite.comparable.C_Equals;
import org.gcube.portlets.user.td.expressionwidget.shared.model.composite.comparable.C_GreaterOrEquals;
import org.gcube.portlets.user.td.expressionwidget.shared.model.composite.comparable.C_GreaterThan;
import org.gcube.portlets.user.td.expressionwidget.shared.model.composite.comparable.C_LessOrEquals;
import org.gcube.portlets.user.td.expressionwidget.shared.model.composite.comparable.C_LessThan;
import org.gcube.portlets.user.td.expressionwidget.shared.model.composite.comparable.C_NotEquals;
import org.gcube.portlets.user.td.expressionwidget.shared.model.composite.comparable.C_NotGreater;
import org.gcube.portlets.user.td.expressionwidget.shared.model.composite.comparable.C_NotLess;
import org.gcube.portlets.user.td.expressionwidget.shared.model.composite.text.C_TextBeginsWith;
import org.gcube.portlets.user.td.expressionwidget.shared.model.composite.text.C_TextContains;
import org.gcube.portlets.user.td.expressionwidget.shared.model.composite.text.C_TextEndsWith;
import org.gcube.portlets.user.td.expressionwidget.shared.model.composite.text.C_TextMatchSQLRegexp;
import org.gcube.portlets.user.td.expressionwidget.shared.model.leaf.C_ColumnReferencePlaceholder;
import org.gcube.portlets.user.td.expressionwidget.shared.model.leaf.C_ConstantList;
import org.gcube.portlets.user.td.expressionwidget.shared.model.leaf.C_Range;
import org.gcube.portlets.user.td.expressionwidget.shared.model.leaf.C_TypedColumnReference;
import org.gcube.portlets.user.td.expressionwidget.shared.model.leaf.TD_Value;
import org.gcube.portlets.user.td.expressionwidget.shared.model.logical.C_And;
import org.gcube.portlets.user.td.expressionwidget.shared.model.logical.C_Between;
import org.gcube.portlets.user.td.expressionwidget.shared.model.logical.C_IsNotNull;
import org.gcube.portlets.user.td.expressionwidget.shared.model.logical.C_IsNull;
import org.gcube.portlets.user.td.expressionwidget.shared.model.logical.C_Not;
import org.gcube.portlets.user.td.expressionwidget.shared.model.logical.C_Or;
import org.gcube.portlets.user.td.expressionwidget.shared.model.logical.C_ValueIsIn;
import org.gcube.portlets.user.td.widgetcommonevent.shared.expression.C_Expression;
import org.gcube.portlets.user.td.widgetcommonevent.shared.tr.column.ColumnDataType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class C_ExpressionParser {
	protected static Logger logger = LoggerFactory
			.getLogger(C_ExpressionParser.class);

	protected SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

	public Expression parse(C_Expression exp) {
		Expression ex = null;
		switch (exp.getId()) {
		case "ColumnReferencePlaceholder":
			ex = getColumnReferencePlaceholder(exp);
			break;
		case "ConstantList":
			ex = getConstantlist(exp);
			break;
		case "Range":
			ex = getRange(exp);
			break;
		case "TypedColumnReference":
			ex = getTypedColumnReference(exp);
			break;
		case "TD_Value":
			TD_Value value = (TD_Value) exp;
			ex = getExpressionValue(value);
			break;
		case "Equals":
			ex = getEquals(exp);
			break;
		case "GreaterOrEquals":
			ex = getGreaterOrEquals(exp);
			break;
		case "GreaterThan":
			ex = getGreaterThan(exp);
			break;
		case "LessOrEquals":
			ex = getLessOrEquals(exp);
			break;
		case "LessThan":
			ex = getLessThan(exp);
			break;
		case "NotEquals":
			ex = getNotEquals(exp);
			break;
		case "NotGreater":
			ex = getNotGreater(exp);
			break;
		case "NotLess":
			ex = getNotLess(exp);
			break;
		case "TextBeginsWith":
			ex = getTextBeginWith(exp);
			break;
		case "TextContains":
			ex = getTextContains(exp);
			break;
		case "TextEndsWith":
			ex = getTextEndWith(exp);
			break;
		case "TextMatchSQLRegexp":
			ex = getTextMatchSQLRegexp(exp);
			break;
		case "And":
			ex = getAnd(exp);
			break;
		case "Between":
			ex = getBetween(exp);
			break;
		case "IsNotNull":
			ex = getIsNotNull(exp);
			break;
		case "IsNull":
			ex = getIsNull(exp);
			break;
		case "Not":
			ex = getNot(exp);
			break;
		case "Or":
			ex = getOr(exp);
			break;
		case "ValueIsIn":
			ex = getValueIsIn(exp);
			break;
		default:
			break;
		}

		return ex;
	}

	public MultivaluedExpression parseMultivalued(C_MultivaluedExpression exp) {
		MultivaluedExpression ex = null;
		switch (exp.getIdMulti()) {
		case "ConstantList":
			ex = getConstantlist(exp);
			break;
		case "Range":
			ex = getRange(exp);
			break;
		case "ExternalReferenceExpression":
			break;
		default:
			break;
		}
		return ex;
	}

	private Expression getValueIsIn(C_Expression exp) {
		C_ValueIsIn v = (C_ValueIsIn) exp;
		MultivaluedExpression multivalued=parseMultivalued(v.getRightArgument());
		ValueIsIn valueIsIn = new ValueIsIn(parse(v.getLeftArgument()),multivalued
				);
		return valueIsIn;
	}

	private Expression getOr(C_Expression exp) {
		C_Or o = (C_Or) exp;
		List<C_Expression> listCExp = o.getArguments();
		List<Expression> listExp = new ArrayList<Expression>();
		Expression ex;
		for (C_Expression cexp : listCExp) {
			ex = parse(cexp);
			listExp.add(ex);
		}
		Or or = new Or(listExp);
		return or;
	}

	private Expression getNot(C_Expression exp) {
		C_Not n = (C_Not) exp;
		Not not = new Not(parse(n.getArgument()));
		return not;
	}

	private Expression getIsNull(C_Expression exp) {
		C_IsNull nullIs = (C_IsNull) exp;
		IsNull isNull = new IsNull(parse(nullIs.getArgument()));
		return isNull;
	}

	private Expression getIsNotNull(C_Expression exp) {
		C_IsNotNull nullNotIs = (C_IsNotNull) exp;
		IsNotNull isNotNull = new IsNotNull(parse(nullNotIs.getArgument()));
		return isNotNull;
	}

	private Expression getBetween(C_Expression exp) {
		C_Between bet = (C_Between) exp;
		C_Range crange = bet.getRightArgument();
		Range range = new Range(getExpressionValue(crange.getMinimum()),
				getExpressionValue(crange.getMaximum()));
		Between between = new Between(parse(bet.getLeftArgument()), range);

		return between;
	}

	private Expression getAnd(C_Expression exp) {
		C_And o = (C_And) exp;
		List<C_Expression> listCExp = o.getArguments();
		List<Expression> listExp = new ArrayList<Expression>();
		Expression ex;
		for (C_Expression cexp : listCExp) {
			ex = parse(cexp);
			listExp.add(ex);
		}
		And and = new And(listExp);
		return and;
	}

	private Expression getTextMatchSQLRegexp(C_Expression exp) {
		C_TextMatchSQLRegexp reg = (C_TextMatchSQLRegexp) exp;
		TextMatchSQLRegexp regExp = new TextMatchSQLRegexp(
				parse(reg.getLeftArgument()), parse(reg.getRightArgument()));
		return regExp;

	}

	private Expression getTextEndWith(C_Expression exp) {
		C_TextEndsWith textEnd = (C_TextEndsWith) exp;
		TextEndsWith textEndWith = new TextEndsWith(
				parse(textEnd.getLeftArgument()),
				parse(textEnd.getRightArgument()));
		return textEndWith;
	}

	private Expression getTextContains(C_Expression exp) {
		C_TextContains textContains = (C_TextContains) exp;
		TextContains textCont = new TextContains(
				parse(textContains.getLeftArgument()),
				parse(textContains.getRightArgument()));
		return textCont;
	}

	private Expression getTextBeginWith(C_Expression exp) {
		C_TextBeginsWith textB = (C_TextBeginsWith) exp;
		TextBeginsWith textBegins = new TextBeginsWith(
				parse(textB.getLeftArgument()), parse(textB.getRightArgument()));
		return textBegins;
	}

	private Expression getNotLess(C_Expression exp) {
		C_NotLess notL = (C_NotLess) exp;
		NotLess notLess = new NotLess(parse(notL.getLeftArgument()),
				parse(notL.getRightArgument()));
		return notLess;
	}

	private Expression getNotGreater(C_Expression exp) {
		C_NotGreater notG = (C_NotGreater) exp;
		NotGreater notGreater = new NotGreater(parse(notG.getLeftArgument()),
				parse(notG.getRightArgument()));
		return notGreater;
	}

	private Expression getNotEquals(C_Expression exp) {
		C_NotEquals notE = (C_NotEquals) exp;
		NotEquals notEquals = new NotEquals(parse(notE.getLeftArgument()),
				parse(notE.getRightArgument()));
		return notEquals;
	}

	private Expression getLessThan(C_Expression exp) {
		C_LessThan lessT = (C_LessThan) exp;
		LessThan lessThan = new LessThan(parse(lessT.getLeftArgument()),
				parse(lessT.getRightArgument()));
		return lessThan;
	}

	private Expression getLessOrEquals(C_Expression exp) {
		C_LessOrEquals lessOrE = (C_LessOrEquals) exp;
		LessOrEquals lessOrEquals = new LessOrEquals(
				parse(lessOrE.getLeftArgument()),
				parse(lessOrE.getRightArgument()));
		return lessOrEquals;
	}

	private Expression getGreaterThan(C_Expression exp) {
		C_GreaterThan greaterThan = (C_GreaterThan) exp;
		GreaterThan greater = new GreaterThan(
				parse(greaterThan.getLeftArgument()),
				parse(greaterThan.getRightArgument()));
		return greater;

	}

	private Expression getGreaterOrEquals(C_Expression exp) {
		C_GreaterOrEquals greaterOrEq = (C_GreaterOrEquals) exp;
		GreaterOrEquals greaterOrEquals = new GreaterOrEquals(
				parse(greaterOrEq.getLeftArgument()),
				parse(greaterOrEq.getRightArgument()));
		return greaterOrEquals;

	}

	protected TDTypeValue getExpressionValue(TD_Value value) {
		TDTypeValue ex = null;
		switch (value.getValueType()) {
		case Boolean:
			ex = new TDBoolean(Boolean.valueOf(value.getValue()));
			break;
		case Date:
			Date d = null;
			try {
				d = sdf.parse(value.getValue());
			} catch (ParseException e) {
				logger.error("Unparseable using " + sdf);
				return null;
			}
			ex = new TDDate(d);
			break;
		case Geometry:
			// ex = new TDGeometry(value.getValue());
			break;
		case Integer:
			ex = new TDInteger(Integer.valueOf(value.getValue()));
			break;
		case Numeric:
			ex = new TDNumeric(Float.valueOf(value.getValue()));
			break;
		case Text:
			ex = new TDText(value.getValue());
			break;
		default:
			break;
		}

		return ex;
	}

	protected DataType mapColumnDataType(ColumnDataType columnDataType) {
		if (columnDataType == ColumnDataType.Integer) {
			return new IntegerType();
		} else {
			if (columnDataType == ColumnDataType.Numeric) {
				return new NumericType();
			} else {
				if (columnDataType == ColumnDataType.Boolean) {
					return new BooleanType();
				} else {
					if (columnDataType == ColumnDataType.Geometry) {
						return new GeometryType();
					} else {
						if (columnDataType == ColumnDataType.Text) {
							return new TextType();
						} else {
							if (columnDataType == ColumnDataType.Date) {
								return new DateType();
							} else {
								return null;
							}
						}
					}
				}
			}
		}
	}

	protected Expression getConstantlist(C_Expression exp) {
		List<TDTypeValue> l = new ArrayList<TDTypeValue>();
		C_ConstantList c = (C_ConstantList) exp;
		List<TD_Value> arguments = c.getArguments();
		TDTypeValue tdv;
		for (TD_Value value : arguments) {
			tdv = getExpressionValue(value);
			l.add(tdv);
		}
		ConstantList constList = new ConstantList(l);
		return constList;
	}

	protected MultivaluedExpression getConstantlist(C_MultivaluedExpression exp) {
		List<TDTypeValue> l = new ArrayList<TDTypeValue>();
		C_ConstantList c = (C_ConstantList) exp;
		List<TD_Value> arguments = c.getArguments();
		TDTypeValue tdv;
		for (TD_Value value : arguments) {
			tdv = getExpressionValue(value);
			l.add(tdv);
		}
		ConstantList constList = new ConstantList(l);
		return constList;
	}

	protected Expression getColumnReferencePlaceholder(C_Expression exp) {
		C_ColumnReferencePlaceholder c = (C_ColumnReferencePlaceholder) exp;
		ColumnReferencePlaceholder col = new ColumnReferencePlaceholder(
				mapColumnDataType(c.getDataType()), c.getColumnId());
		return col;
	}

	protected Expression getRange(C_Expression exp) {
		C_Range c = (C_Range) exp;
		Range range = new Range(getExpressionValue(c.getMinimum()),
				getExpressionValue(c.getMaximum()));
		return range;
	}
	
	protected MultivaluedExpression getRange(C_MultivaluedExpression exp) {
		C_Range c = (C_Range) exp;
		Range range = new Range(getExpressionValue(c.getMinimum()),
				getExpressionValue(c.getMaximum()));
		return range;
	}

	protected Expression getTypedColumnReference(C_Expression exp) {
		C_TypedColumnReference c = (C_TypedColumnReference) exp;
		TableId tableId = new TableId(Long.valueOf(c.getTrId().getTableId()));
		ColumnLocalId columnId = new ColumnLocalId(c.getColumnId());

		TypedColumnReference ref = new TypedColumnReference(tableId, columnId,
				mapColumnDataType(c.getDataType()));
		return ref;
	}

	protected Expression getEquals(C_Expression exp) {
		C_Equals e = (C_Equals) exp;
		Expression left = parse(e.getLeftArgument());
		Expression right = parse(e.getRightArgument());
		Equals eq = new Equals(left, right);
		return eq;
	}

}
