/**
 * Join VRE Portlet JavaScript Functions
 * Migrated from Liferay 6.2 to 7.4
 */

// Namespace for Join VRE portlet functions
window.JoinVRE = window.JoinVRE || {};

/**
 * Request membership for a VRE
 * @param {number} vreId - The ID of the VRE to request membership for
 * @param {HTMLElement} buttonEl - The button element triggering the request
 */
function requestMembership(vreId, buttonEl) {
    console.log('Requesting membership for VRE ID:', vreId);

    // Show loading state
    const button = buttonEl;
    const originalText = button.textContent;
    button.textContent = 'Processing...';
    button.disabled = true;

    // Get portlet namespace and action URL generated server-side (includes CSRF token)
    const portletNamespace = (window.JoinVREConfig && window.JoinVREConfig.portletNamespace)
        ? window.JoinVREConfig.portletNamespace
        : (window.portletNamespace || '_org_gcube_portlets_user_joinvre_portlet_JoinVREPortlet_');
    const actionURL = (window.JoinVREConfig && window.JoinVREConfig.actionURL)
        ? window.JoinVREConfig.actionURL
        : null;

    if (!actionURL) {
        console.error('Action URL not available. Ensure <portlet:actionURL /> is set in view.jsp');
        showNotification('Error sending membership request. Action URL missing.', 'error');
        button.textContent = originalText;
        button.disabled = false;
        return;
    }

    // Create form data for the request (namespaced parameters)
    const formData = new FormData();
    formData.append(portletNamespace + 'vreId', vreId);

    // Make AJAX request to portlet action URL (contains p_auth automatically)
    fetch(actionURL, {
        method: 'POST',
        body: formData,
        credentials: 'same-origin',
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'text/html'
        }
    })
    .then(response => {
        if (response.ok) {
            return response.text();
        }
        throw new Error('Network response was not ok');
    })
    .then(data => {
        // Show success message
        showNotification('Membership request sent successfully!', 'success');

        // Update button state
        button.textContent = 'Request Sent';
        button.classList.remove('btn-secondary');
        button.classList.add('btn-warning');

        // Optionally reload the portlet content
        setTimeout(() => {
            window.location.reload();
        }, 1500);
    })
    .catch(error => {
        console.error('Error requesting membership:', error);
        showNotification('Error sending membership request. Please try again. ' + (error && error.message ? '(' + error.message + ')' : ''), 'error');

        // Restore button state
        button.textContent = originalText;
        button.disabled = false;
    });
}

/**
 * Show notification message to user
 * @param {string} message - The message to display
 * @param {string} type - The type of notification (success, error, warning, info)
 */
function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `alert alert-${type === 'error' ? 'danger' : type} alert-dismissible fade show`;
    notification.style.position = 'fixed';
    notification.style.top = '20px';
    notification.style.right = '20px';
    notification.style.zIndex = '9999';
    notification.style.minWidth = '300px';
    
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    `;
    
    // Add to page
    document.body.appendChild(notification);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 5000);
}

/**
 * Handle VRE link clicks with CSP considerations
 * @param {string} url - The URL to navigate to
 * @param {boolean} isPublic - Whether this is a public VRE
 */
function handleVRELink(url, isPublic = false) {
    if (isPublic) {
        // For public VREs, try to open in a new tab to avoid CSP issues
        const newWindow = window.open(url, '_blank', 'noopener,noreferrer');
        if (!newWindow) {
            // If popup blocked, show alternative
            showNotification('Please allow popups for this site or copy this URL: ' + url, 'warning');
        }
    } else {
        // For member VREs, navigate directly
        window.location.href = url;
    }
}

/**
 * Initialize portlet when DOM is ready
 */
document.addEventListener('DOMContentLoaded', function() {
    console.log('Join VRE Portlet initialized');
    
    // Set up global portlet namespace if available
    if (window.JoinVREConfig && window.JoinVREConfig.portletNamespace) {
        window.portletNamespace = window.JoinVREConfig.portletNamespace;
    } else {
        const portletContainer = document.querySelector('[id*="joinvre"]');
        if (portletContainer) {
            const portletId = portletContainer.id;
            window.portletNamespace = '_' + portletId.replace(/^p_p_id_|_$/, '') + '_';
        }
    }
    
    // Handle public VRE links to avoid CSP issues
    const publicLinks = document.querySelectorAll('a[href*="d4science.org"]');
    publicLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            handleVRELink(this.href, true);
        });
    });

    // Detect very wide logos and adjust object-fit
    const logoImages = document.querySelectorAll('.vre-logo');
    logoImages.forEach(img => {
        const applyClassIfWide = () => {
            const nw = img.naturalWidth;
            const nh = img.naturalHeight;
            if (nw > 0 && nh > 0) {
                const ratio = nw / nh;
                // threshold: mark as wide if aspect ratio exceeds 2.0
                if (ratio >= 2.0) {
                    img.classList.add('vre-logo--wide');
                } else {
                    img.classList.remove('vre-logo--wide');
                }
            }
        };
        if (img.complete) {
            // already loaded
            applyClassIfWide();
        } else {
            img.addEventListener('load', applyClassIfWide);
            img.addEventListener('error', applyClassIfWide); // fallback logo also handled
        }
    });
});

// Export functions for global access
window.requestMembership = requestMembership;
window.showNotification = showNotification;
window.handleVRELink = handleVRELink;