package org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.commons.lang.Validate;
import org.apache.jackrabbit.util.Text;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.FolderItemType;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.Document;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.DocumentMetadata;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.link.DocumentAlternativeLink;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.link.DocumentPartLink;
import org.gcube.portlets.user.homelibrary.jcr.repository.JCRRepository;
import org.gcube.portlets.user.homelibrary.jcr.workspace.JCRWorkspace;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.ContentType;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRFile;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRWorkspaceFolderItem;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.link.JCRDocumentAlternativeLink;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.link.JCRDocumentPartLink;


public class JCRDocument extends JCRWorkspaceFolderItem implements Document {

	private static final String METADATA   		= "hl:metadata"; 
	private static final String ANNOTATIONS 	= "hl:annotations";
	
	private static final String ALTERNATIVES 	= "hl:alternatives";
	
	private static final String PARENT_URI		= "hl:parentUri";
	private static final String URI				= "hl:uri";
	private static final String NAME			=  "hl:name";
	private static final String MIME_TYPE		=  "hl:mimeType";
	
	private static final String NT_ALTERNATIVE	= "nthl:documentAlternativeLink";
	private static final String NT_PART			= "nthl:documentPartLink";
	
	private static final String PARTS			= "hl:parts";
	private static final String COLLECTION_NAME = "hl:collectionName";
	private static final String OID				= "hl:oid";
	
	private static final String NT_CONTENT 		= "nthl:documentItemContent";
	
	private final String oid;
	protected JCRFile documentContent;
	
	public JCRDocument(JCRWorkspace workspace, Node node) throws RepositoryException  {
		this(workspace,node,ContentType.GENERAL);
	}
	
	public JCRDocument(JCRWorkspace workspace, Node node, String name,
			String description, String oid, String mimeType, InputStream documentData, 
			Map<String, String> metadata, Map<String,String> annotations,
			String collectionName) throws RepositoryException, InternalErrorException   {
		this(workspace,node,name,description,oid,mimeType,ContentType.GENERAL,documentData,
				metadata,annotations,collectionName);
	}
	
	protected JCRDocument(JCRWorkspace workspace, Node node, ContentType contentType) throws RepositoryException  {
		super(workspace, node);
	
		Node contentNode = node.getNode(CONTENT);
		this.oid = contentNode.getProperty(OID).getString();
		this.documentContent = workspace.getGCUBEDocumentContent(node.getSession(), oid,contentType);
				
	}
	
	protected JCRDocument(JCRWorkspace workspace, Node node, String name,
			String description, String oid, String mimeType, ContentType contentType, InputStream documentData, 
			Map<String, String> metadata, Map<String,String> annotations,
			String collectionName) throws RepositoryException, InternalErrorException  {
		super(workspace,node,name,description); 
		
		Validate.notNull(oid, "Oid must be not null");
		Validate.notNull(mimeType, "MimeType must be not null");
		Validate.notNull(metadata, "Metadata must be not null");
		Validate.notNull(annotations, "Annotations must be not null");
		Validate.notNull(collectionName, "CollectionName must be not null");
		
		Node contentNode = node.addNode(CONTENT,NT_CONTENT);
		node.setProperty(FOLDER_ITEM_TYPE, FolderItemType.DOCUMENT.toString());

		contentNode.setProperty(OID, oid);
		contentNode.setProperty(COLLECTION_NAME, collectionName);

		this.oid = oid;
		
		try {
			this.documentContent = (workspace).getGCUBEDocumentContent(node.getSession(), oid,contentType);
		} catch (RepositoryException e) {
			try {
				if(documentData != null) {
					this.documentContent = (workspace).setGCUBEDocumentContent(node.getSession(), oid,documentData,
							mimeType, contentType);
				} else 
					throw new IllegalArgumentException("Document content is not present end data stream is null");
			} catch (IOException e1) {
				throw new InternalErrorException(e1);
			} 
		}
				
	}
	
	private Map<String,DocumentMetadata> serializeMetadata( Node contentNode, Map<String,String> metadata) throws RepositoryException {
		
		Node metadataNode = contentNode.getNode(METADATA);
		HashMap<String,DocumentMetadata> map = new HashMap<String,DocumentMetadata>();
		for (Entry<String, String> entry : metadata.entrySet()) {
			metadataNode.setProperty(entry.getKey(), entry.getValue());
			map.put(entry.getKey(), new JCRDocumentMetadata(entry.getKey(),
					entry.getValue()));
		}
		return map;
	}
	
	private Map<String,DocumentMetadata> deserializeMetadata(Node contentNode) throws RepositoryException {
		
		Node metadataNode = contentNode.getNode(METADATA);
		Map<String,DocumentMetadata> map = new HashMap<String,DocumentMetadata>();
		
		for (PropertyIterator iterator =  metadataNode.getProperties(); iterator.hasNext();) {
			Property property = iterator.nextProperty();
			if(!property.getName().startsWith("jcr:")) {
				DocumentMetadata metadata = new JCRDocumentMetadata(property.getName(), property.getValue().getString());
				map.put(property.getName(),metadata);
			}
		}
		return map;
	}
	
	private Map<String, String> serializeAnnotations(Node contentNode, Map<String, String> annotations) throws RepositoryException {
		
		Node annotationsNode = contentNode.getNode(ANNOTATIONS);
		for (Entry<String,String> entry : annotations.entrySet()) 
			annotationsNode.setProperty(entry.getKey(), entry.getValue());
		
		return annotations;
	}
	
	private Map<String, String> deserializeAnnotations(Node contentNode) throws RepositoryException {
		
		Map<String,String> map = new HashMap<String,String>();
		Node annotationsNode = contentNode.getNode(ANNOTATIONS);
		
		for (PropertyIterator iterator =  annotationsNode.getProperties(); iterator.hasNext();) {
			Property property = iterator.nextProperty();
			if(!property.getName().startsWith("jcr:")) 
				map.put(property.getName(), property.getValue().getString());
		}
		return map;
	}
	
	public void setAlternatives(Session session, List<DocumentAlternativeLink> alternatives) {
				
		try {
			Node itemNode = session.getNodeByIdentifier(identifier);
			Node contentNode = itemNode.getNode(CONTENT);
			
			Node alternativesNode = contentNode.getNode(ALTERNATIVES);
			int index = 0;
			for (DocumentAlternativeLink documentAlternativeLink : alternatives) {
				
				Node alternative = alternativesNode.addNode("" + index, NT_ALTERNATIVE);
			
				logger.debug("\nINFOS ALTERNATIVES ---------------------------------- : " + documentAlternativeLink.getParentURI() +
						documentAlternativeLink.getURI() +
						 documentAlternativeLink.getName() +
						 documentAlternativeLink.getMimeType());
				
				alternative.setProperty(PARENT_URI, (documentAlternativeLink.getParentURI()==null)?"":documentAlternativeLink.getParentURI());
				alternative.setProperty(URI, (documentAlternativeLink.getURI()== null)?"":documentAlternativeLink.getURI());
				alternative.setProperty(NAME, (documentAlternativeLink.getName()==null)?"":documentAlternativeLink.getName());
				alternative.setProperty(MIME_TYPE, (documentAlternativeLink.getMimeType()==null)?"":documentAlternativeLink.getMimeType());
				index ++;
			}
		
		} catch (RepositoryException e) {
			logger.error("Set DocumentAlternativeLink failed",e);
		}
		
		
	}
	
	private List<DocumentAlternativeLink> getAlternativesFromNode(Node node) {
		List<DocumentAlternativeLink> list = new LinkedList<DocumentAlternativeLink>();
		
		try {
			Node contentNode = node.getNode(CONTENT);
			Node alternativesNode = contentNode.getNode(ALTERNATIVES);
			for (NodeIterator iterator = alternativesNode.getNodes(); iterator.hasNext();) {
				
				Node alternative = iterator.nextNode();
				if (!alternative.getPrimaryNodeType().getName().equals(NT_ALTERNATIVE))	
					continue;
				
				logger.debug(alternative.getPrimaryNodeType().getName());
				String parentURI = alternative.getProperty(PARENT_URI).getString();
				String uri = alternative.getProperty(URI).getString();
				String name = alternative.getProperty(NAME).getString();
				String mimeType = alternative.getProperty(MIME_TYPE).getString();
				DocumentAlternativeLink documentAlternativeLink = new JCRDocumentAlternativeLink(parentURI, uri, name, mimeType);
				list.add(documentAlternativeLink);
			}
		
		} catch (RepositoryException e) {
			logger.debug("getDocumentAlternativeLink failed",e);
		}
		return list;
		
	}
	
	
	public void setParts(Session session, List<DocumentPartLink> parts) {
		
		try {
			Node node = session.getNodeByIdentifier(identifier);
			Node contentNode = node.getNode(CONTENT);
			
			Node partsNode = contentNode.getNode(PARTS);
			int index = 0;
			for (DocumentPartLink documentPartLink : parts) {
				Node part = partsNode.addNode("" + index, NT_ALTERNATIVE);
				logger.debug("\nINFOS PARTS ------------------------------------ : " + documentPartLink.getParentURI() +
						documentPartLink.getURI() +
						 documentPartLink.getName() +
						 documentPartLink.getMimeType());
				
				part.setProperty(PARENT_URI, (documentPartLink.getParentURI()==null)?"":documentPartLink.getParentURI());
				part.setProperty(URI, (documentPartLink.getURI()==null)?"":documentPartLink.getURI());
				part.setProperty(NAME, (documentPartLink.getName()==null)?"":documentPartLink.getName());
				part.setProperty(MIME_TYPE, (documentPartLink.getMimeType()==null)?"":documentPartLink.getMimeType());
				index ++;
			}
		
		} catch (RepositoryException e) {
			logger.error("setPart failed",e);
		}
	}
	
	private List<DocumentPartLink> getPartsFromNode(Node node) {
		List<DocumentPartLink> list = new LinkedList<DocumentPartLink>();
		
		try {
			Node contentNode = node.getNode(CONTENT);
			Node partsNode = contentNode.getNode(PARTS);
			for (NodeIterator iterator = partsNode.getNodes(); iterator.hasNext();) {
				Node part = iterator.nextNode();
				if (!part.getPrimaryNodeType().getName().equals(NT_PART))
					continue;
				
				
				logger.debug(part.getPrimaryNodeType().getName());
				String parentURI = part.getProperty(PARENT_URI).getString();
				String uri = part.getProperty(URI).getString();
				String name = part.getProperty(NAME).getString();
				String mimeType = part.getProperty(MIME_TYPE).getString();
				DocumentPartLink documentAlternativeLink = new JCRDocumentPartLink(parentURI, uri, name, mimeType);
				list.add(documentAlternativeLink);
			}
		
		} catch (RepositoryException e) {
			logger.debug("getPartsFromNode failed");
		}
		return list;
	}

	@Override
	public InputStream getData() throws InternalErrorException {
		return documentContent.getData();
	}
	
	@Override
	public long getLength() throws InternalErrorException {
		return documentContent.getLength();
	}
	
	@Override
	public String getMimeType() {
		return documentContent.getMimeType();
	}

	@Override
	public Map<String, DocumentMetadata> getMetadata() {
		
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node node = session.getNodeByIdentifier(identifier);
			Node contentNode = node.getNode(CONTENT);
			return deserializeMetadata(contentNode);
		} catch (Exception e) {
			throw new RuntimeException(e) ;
		} finally {
			if(session != null)
				session.logout();
		}
	}

	@Override
	public Map<String, String> getAnnotation() {
		
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node node = session.getNodeByIdentifier(identifier);
			Node contentNode = node.getNode(CONTENT);
			return deserializeAnnotations(contentNode);
		} catch (Exception e) {
			throw new RuntimeException(e) ;
		} finally {
			if(session != null)
				session.logout();
		}	
		
	}

	@Override
	public List<DocumentAlternativeLink> getAlternatives() {
		
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node node = session.getNodeByIdentifier(identifier);
			List<DocumentAlternativeLink> alternatives = getAlternativesFromNode(node);
			return (alternatives == null)?new LinkedList<DocumentAlternativeLink>():alternatives;
		} catch (Exception e) {
			throw new RuntimeException(e) ;
		} finally {
			if(session != null)
				session.logout();
		}	
	}

	@Override
	public List<DocumentPartLink> getParts() {
		
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node node = session.getNodeByIdentifier(identifier);
			List<DocumentPartLink> parts = getPartsFromNode(node);
			return (parts == null)? new LinkedList<DocumentPartLink>():parts;
		} catch (Exception e) {
			throw new RuntimeException(e) ;
		} finally {
			if(session != null)
				session.logout();
		}	
		
	}

	@Override
	public String getCollectionName() {
		
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node node = session.getNodeByIdentifier(identifier);
			Node contentNode = node.getNode(CONTENT);
			return contentNode.getProperty(COLLECTION_NAME).getString();
		} catch (Exception e) {
			throw new RuntimeException(e) ;
		} finally {
			if(session != null)
				session.logout();
		}	
		
	}

	@Override
	public FolderItemType getFolderItemType() {
		return FolderItemType.DOCUMENT;
	}

	@Override
	public String getURI() {
		
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node node = session.getNodeByIdentifier(identifier);
			Node contentNode = node.getNode(CONTENT);
			return contentNode.getProperty(OID).getString();
		} catch (Exception e) {
			throw new RuntimeException(e) ;
		} finally {
			if(session != null)
				session.logout();
		}	
		
	}

	@Override
	public void saveContent(Node node) throws RepositoryException {
		
		Node fileNode = JCRRepository.getGCubeRoot(node.getSession())
		.getNode(Text.escapeIllegalJcrChars(oid)).getNode(CONTENT);
		this.documentContent.save(fileNode);
	}

}
