package org.gcube.portlets.user.homelibrary.jcr.workspace;

import java.util.Calendar;


import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import org.apache.commons.lang.Validate;
import org.apache.jackrabbit.util.Text;
import org.gcube.common.core.utils.logging.GCUBEClientLog;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.workspace.Properties;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItemAction;
import org.gcube.portlets.user.homelibrary.home.workspace.acl.Capabilities;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WrongDestinationException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WrongItemTypeException;
import org.gcube.portlets.user.homelibrary.jcr.content.JCRRepository;

public abstract class JCRWorkspaceItem implements WorkspaceItem {
	

	private static final String LAST_MODIFIED_BY 	= "jcr:lastModifiedBy";
	private static final String DESCRIPTION 		= "jcr:description";
	private static final String TITLE 				= "jcr:title"; 
	private static final String CREATED 			= "jcr:created";
	private static final String LAST_ACTION 		= "hl:lastAction";
	private static final String LAST_MODIFIED 		= "jcr:lastModified";
	
	protected final JCRWorkspace workspace;
	protected String identifier;
	protected Calendar creationDate;

	
	protected static GCUBEClientLog logger = new GCUBEClientLog(JCRWorkspaceItem.class);

	public JCRWorkspaceItem(JCRWorkspace workspace, Node node) throws RepositoryException {

		this.workspace = workspace;
		this.identifier = node.getIdentifier();
		this.creationDate = node.getProperty(CREATED).getDate();

	}
	
	public JCRWorkspaceItem(JCRWorkspace workspace, Node node, String name,
			String description) throws  RepositoryException  {			

		Validate.notNull(name, "Name must be not null");
		Validate.notNull(description, "Description must be not null");
				
		this.workspace = workspace;
		
		node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
		node.setProperty(DESCRIPTION, description);
		node.setProperty(TITLE, name);
		node.setProperty(LAST_ACTION,WorkspaceItemAction.CREATED.toString());

	}
	
	public void save(Node node) throws RepositoryException {
		
		node.getSession().save();		
		this.identifier = node.getIdentifier();
		this.creationDate = node.getProperty(CREATED).getDate();

	}
	
	@Override
	public String getId() throws InternalErrorException {
		return identifier;
	}

	@Override
	public String getName() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			String name = node.getProperty(TITLE).getString();
			return name;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public String getDescription() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return node.getProperty(DESCRIPTION).getString();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void setDescription(String description)
			throws InternalErrorException {	
		
		Session session = JCRRepository.getSession();
		try {
			internalDescription(session.getNodeByIdentifier(identifier), description);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void rename(String name) throws InternalErrorException,
			InsufficientPrivilegesException, ItemAlreadyExistException {
		
		if (!workspace.isValidName(name))
			throw new IllegalArgumentException("Invalid item name");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			internalRename(node, name);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public Calendar getCreationTime() throws InternalErrorException {
		
		return creationDate;
	}

	@Override
	public Calendar getLastModificationTime() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return node.getProperty(LAST_MODIFIED).getDate();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkspaceItemAction getLastAction() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return WorkspaceItemAction.valueOf(node.getProperty(LAST_ACTION).getString());
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public Capabilities getCapabilities() {
		return null;
	}

	@Override
	public Properties getProperties() throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return new JCRProperties(node);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkspaceFolder getParent() throws InternalErrorException {
	
		Session session = JCRRepository.getSession();
		try {
			return workspace.getParent(session.
					getNodeByIdentifier(identifier));
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}
	
	private WorkspaceFolder getParent(Node node) throws InternalErrorException {
		try {
			return workspace.getParent(node);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} 
	}
	
	@Override
	public String getPath() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			return getPath(session.getNodeByIdentifier(identifier));
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
	}
	
	public String getPath(Node node) throws RepositoryException, InternalErrorException {
		
		if (isRoot(node)) return workspace.getPathSeparator() + node.getProperty(TITLE).getString();
		return ((JCRWorkspaceFolder)getParent(node)).getPath(node.getParent()) + workspace.getPathSeparator() 
		+ node.getProperty(TITLE).getString();		
	}

	@Override
	public boolean isRoot() throws InternalErrorException { 
		return getParent() == null;
	}
	
	public boolean isRoot(Node node) throws RepositoryException, InternalErrorException {
		return workspace.getParent(node) == null;
	}

	@Override
	public void remove() throws InternalErrorException,
			InsufficientPrivilegesException {
		try {
			workspace.removeItem(getId());
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} 
		
	}

	@Override
	public void move(WorkspaceFolder destination)
			throws InternalErrorException, WrongDestinationException,
			InsufficientPrivilegesException, ItemAlreadyExistException {
		
		Validate.notNull(destination, "Destination must be not null");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(destination.getId());
			internalMove(node);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			if (session != null)
				session.logout();
		}
	}

	@Override
	public WorkspaceItem cloneItem(String cloneName)
			throws InternalErrorException, InsufficientPrivilegesException,
			ItemAlreadyExistException {
		
		Validate.notNull(cloneName, "Clone name must be not null");
		if(!workspace.isValidName(cloneName))
			throw new IllegalArgumentException("Clone name is not valid");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return (workspace).getWorkspaceItem(internalCopy(node.getParent(),cloneName));
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		} finally {
			if(session != null)
				session.logout();
		}
	}
	
	
	public Node internalCopy(Node nodeFolder, String newName) throws InternalErrorException,
	ItemAlreadyExistException, WrongDestinationException, RepositoryException{
		
		Node node = nodeFolder.getSession().getNodeByIdentifier(identifier);
		
		String pathNewNode = nodeFolder.getPath()
		+ "/" + Text.escapeIllegalJcrChars(newName);
		
		try {
			
			if(node.getSession().getNode(pathNewNode) != null)
				throw new ItemAlreadyExistException(newName + " already exist");
		} catch(RepositoryException e) {
			
		}
		
		try {
			node.getSession().getWorkspace().copy(node.getPath(),pathNewNode);
			Node newNode = node.getSession().getNode(pathNewNode);
			newNode.setProperty(LAST_MODIFIED,Calendar.getInstance());
			newNode.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			newNode.setProperty(TITLE, newName);
			newNode.setProperty(LAST_ACTION,WorkspaceItemAction.CLONED.toString());
			newNode.getSession().save();
			return newNode;
		} catch (ItemExistsException e) {
			throw new ItemAlreadyExistException(e.getMessage());
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		}
		
	}
	
	public void internalMove(Node destinationFolderNode) throws ItemAlreadyExistException,
	InternalErrorException, RepositoryException {
			
		try {
			logger.debug("Start internal move item with id " 
					+ getId() + " to destination item with id " + destinationFolderNode.getIdentifier());
			
		
			Node node = destinationFolderNode.getSession().getNodeByIdentifier(identifier);
			
			if (workspace.exists(node.getName(), destinationFolderNode.getIdentifier())) {
				logger.error("Item with name " + getName() + " exists");
				throw new ItemAlreadyExistException("Item " + node.getName() + " already exists");
			}
			node.setProperty(LAST_MODIFIED,Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			node.setProperty(LAST_ACTION,WorkspaceItemAction.MOVED.toString());
			node.getSession().save();
			
			node.getSession().getWorkspace().move(node.getPath(), destinationFolderNode.getPath() 
					+ "/" + node.getName());
		} catch (RepositoryException e) {
			logger.error("Repository exception thrown by move operation",e);
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			logger.fatal("Unhandled Exception ");
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			logger.fatal("Unhandled Exception ");
			throw new InternalErrorException(e);
		} 
	}

	public void internalRename(Node node, String newName) throws ItemAlreadyExistException, InternalErrorException {
		
		String nodeNewName = Text.escapeIllegalJcrChars(newName);
		try {
			
			logger.debug("Internal rename item with id " 
					+ getId() + " to destination item with id " + node.getParent().getIdentifier());
			if (workspace.exists(nodeNewName, node.getParent().getIdentifier())) {
				logger.error("Item with name " + nodeNewName + " exists");
				throw new ItemAlreadyExistException("Item " + nodeNewName + " already exists");
			}
			String newPath = node.getParent().getPath() 
			+ workspace.getPathSeparator() + nodeNewName;
			
			node.setProperty(LAST_MODIFIED,Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			node.setProperty(LAST_ACTION,WorkspaceItemAction.RENAMED.toString());
			node.setProperty(TITLE,newName);
			node.getSession().save();
			
			String path = node.getPath();
			node.getSession().getWorkspace().move(path, newPath);
			
		} catch (RepositoryException e) {
			logger.error("Repository exception thrown by move operation",e);
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			logger.fatal("Unhandled Exception ");
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			logger.fatal("Unhandled Exception ");
			throw new InternalErrorException(e);
		}
		
	}

	public void internalDescription(Node node, String newDescription) throws InternalErrorException {
		
		Validate.notNull(newDescription, "Description must be not null");
		try {
			node.setProperty(DESCRIPTION, newDescription);
			node.setProperty(LAST_MODIFIED, Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY, workspace.getOwner().getPortalLogin());
			node.getSession().save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 
	}
	
}
