package org.gcube.portlets.user.homelibrary.jcr.workspace;


import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.jackrabbit.util.Text;
import org.gcube.portlets.user.homelibrary.home.Home;
import org.gcube.portlets.user.homelibrary.home.HomeManager;
import org.gcube.portlets.user.homelibrary.home.User;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItemAction;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItemType;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceSharedFolder;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WorkspaceFolderNotFoundException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WrongDestinationException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WrongItemTypeException;
import org.gcube.portlets.user.homelibrary.jcr.content.JCRRepository;
import org.gcube.portlets.user.homelibrary.jcr.home.JCRHome;
import org.gcube.portlets.user.homelibrary.util.WorkspaceUtil;

public class JCRWorkspaceSharedFolder extends JCRAbstractWorkspaceFolder implements WorkspaceSharedFolder {

	private static final String USERS =	"hl:users";
	
	private String destinationFolderId;
	private List<String> users;
	
	public JCRWorkspaceSharedFolder(JCRWorkspace workspace, Node node) throws RepositoryException {
		super(workspace,node);	
	}
	
	public JCRWorkspaceSharedFolder(JCRWorkspace workspace, Node node,
			String name, String description, String destinationFolderId, List<String> users) throws RepositoryException, InternalErrorException {		
		super(workspace,node,name,description);
		
		super.save(node);
		
		this.destinationFolderId = destinationFolderId;
		
		// TODO
		this.users = users;
		
	}
		
	public void save(Node node) throws RepositoryException {
		
		// TODO
		try {
			
			addUser(node,workspace.getOwner().getPortalLogin(),destinationFolderId);
			
			for (String user : users) {
				HomeManager homeManager = workspace.getHome().getHomeManager();
			
				Home home = homeManager.getHome(user);
				addUser(node, user, home.getWorkspace().getRoot().getId());
			}
		} catch (Exception e) {
			throw new RepositoryException(e);
		}
	}
	
	private Node getUserNode(Node node) throws RepositoryException,
	InternalErrorException {
		
		Node usersNode = node.getNode(USERS);
		String value = usersNode.getProperty(
				workspace.getOwner().getPortalLogin()).getString();
		
		
		String[] values = value.split(workspace.getPathSeparator());
		if (values.length < 2)
			throw new InternalErrorException("Path node corrupt");
		
		String parentId = values[0];
		String nodeName = values[1];
		Node parentNode = node.getSession().getNodeByIdentifier(parentId);
				
		return node.getSession().getNode(parentNode.getPath() + 
				workspace.getPathSeparator() + nodeName);
	}
	
	private String getNodeName(Node node) throws RepositoryException,
	InternalErrorException {
		
		String[] names = node.getPath().split(
				workspace.getPathSeparator());
		return names[names.length - 1];
	}
	
	@Override
	public String getName() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node sharedNode = session.getNodeByIdentifier(getId());
			Node userNode = getUserNode(sharedNode);
			return getNodeName(userNode);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}	
	}
		
	public void internalRename(Node node, String newName) throws ItemAlreadyExistException, InternalErrorException {
		
		String nodeNewName = Text.escapeIllegalJcrChars(newName);
		try {
			
			Node userNode = getUserNode(node);
			
			if (workspace.exists(nodeNewName, userNode.getParent().getIdentifier())) {
				logger.error("Item with name " + nodeNewName + " exists");
				throw new ItemAlreadyExistException("Item " + nodeNewName + " already exists");
			}
			String newPath = userNode.getParent().getPath() 
			+ workspace.getPathSeparator() + nodeNewName;
			
			node.setProperty(LAST_MODIFIED,Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			node.setProperty(LAST_ACTION,WorkspaceItemAction.RENAMED.toString());
			node.getSession().save();
			
			String path = userNode.getPath();
			node.getSession().getWorkspace().move(path, newPath);

			Node usersNode = node.getNode(USERS);
			String value = userNode.getParent().getIdentifier() + workspace.getPathSeparator() + newName;
			usersNode.setProperty(getOwner().getPortalLogin(), value);
			

			node.getSession().save();
			
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			throw new InternalErrorException(e);
		} finally {
			
		}
		
	}
	
	public void internalMove(Node destinationFolderNode) throws ItemAlreadyExistException,
	InternalErrorException, RepositoryException {
			
		try {
			logger.debug("Start internal move item with id " 
					+ getId() + " to destination item with id " + destinationFolderNode.getIdentifier());
			
			Session session = destinationFolderNode.getSession();
			
			Node node = session.getNodeByIdentifier(identifier);
			
			if (workspace.exists(node.getName(), destinationFolderNode.getIdentifier())) {
				logger.error("Item with name " + getName() + " exists");
				throw new ItemAlreadyExistException("Item " + node.getName() + " already exists");
			}
			node.setProperty(LAST_MODIFIED,Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			node.setProperty(LAST_ACTION,WorkspaceItemAction.MOVED.toString());
			node.getSession().save();
			
			Node userNode = getUserNode(node);
			String userNodeName = getNodeName(userNode);
			
			String newPath = destinationFolderNode.getPath() 
					+ workspace.getPathSeparator() + userNodeName;
			
			String value = destinationFolderNode.getIdentifier() +
					workspace.getPathSeparator() + userNodeName;
			
			session.getWorkspace().clone(session.getWorkspace().getName(),
					node.getPath(), newPath, false);
			
			
			Node usersNode = node.getNode(USERS);
			usersNode.setProperty(getOwner().getPortalLogin(), value);

			session.removeItem(userNode.getPath());
			
			session.save();
		} catch (ItemExistsException e) {
			throw new ItemAlreadyExistException(e.getMessage());
		} catch (RepositoryException e) {
			logger.error("Repository exception thrown by move operation",e);
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			logger.fatal("Unhandled Exception ");
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			logger.fatal("Unhandled Exception ");
			throw new InternalErrorException(e);
		} 
	}
	
	public Node unShareNode(Node node, Node parentNode) throws RepositoryException,
	InternalErrorException, InsufficientPrivilegesException, ItemAlreadyExistException, WorkspaceFolderNotFoundException, WrongDestinationException, ItemNotFoundException {
		
		Session session = node.getSession();
		Node userNode = getUserNode(node);
		
		String folderName = getNodeName(userNode);
		String description = getDescription();
		if (parentNode.getIdentifier().equals(userNode.getParent().getIdentifier())) {
			// Remove user
			Node usersNode = node.getNode(USERS);
			usersNode.getProperty(getOwner().getPortalLogin()).remove();

			remove();
			session.save();
		}

		String unSharedFolderId = workspace.createFolder(folderName, description, parentNode.getIdentifier()).getId();
		Node nodeFolder = session.getNodeByIdentifier(unSharedFolderId);
		
		for (NodeIterator iterator = node.getNodes(); iterator.hasNext();) {
			Node child = (Node) iterator.next();
			// FIXME
			if (!child.getName().startsWith("hl:") && !child.getName().startsWith("jcr:"))			
				session.getWorkspace().copy(child.getPath(), nodeFolder.getPath() 
						+ workspace.getPathSeparator() + child.getName());
		}
		
		session.save();
		return nodeFolder;
	}
	
	
	
	public Node internalCopy(Node nodeFolder, String newName) throws InternalErrorException,
	ItemAlreadyExistException, WrongDestinationException, RepositoryException{
		
		Session session = nodeFolder.getSession();
		
		Node node = session.getNodeByIdentifier(identifier);
		
		String pathNewNode = nodeFolder.getPath()
		+ workspace.getPathSeparator() + Text.escapeIllegalJcrChars(newName);
		
		try {
			
			if(session.getNode(pathNewNode) != null)
				throw new ItemAlreadyExistException(newName + " already exist");
		} catch(RepositoryException e) {
			
		}
		
		try {
			
			Node newNode = unShareNode(node, nodeFolder);
			
			newNode.setProperty(LAST_MODIFIED,Calendar.getInstance());
			newNode.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			newNode.setProperty(TITLE, newName);
			newNode.setProperty(LAST_ACTION,WorkspaceItemAction.CLONED.toString());
			newNode.getSession().save();
			return newNode;
		} catch (ItemExistsException e) {
			throw new ItemAlreadyExistException(e.getMessage());
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 
	}
	
	@Override
	public WorkspaceFolder getParent() throws InternalErrorException {
	
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return workspace.getParent(getUserNode(node));
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	public String getPath(Node node) throws RepositoryException, InternalErrorException {
		
		Node userNode =  getUserNode(node);
		return ((JCRWorkspaceFolder)getParent(userNode)).getPath(userNode.getParent())
				+ workspace.getPathSeparator() + getNodeName(userNode);		
	}
	
	
	@Override
	public void remove() throws InternalErrorException,
			InsufficientPrivilegesException {
		
		
		Session session = JCRRepository.getSession();
		try {
			
			Node sharedNode = session.getNodeByIdentifier(getId());
			Node userNode = getUserNode(sharedNode);
				
			session.removeItem(userNode.getPath());
			
			// Remove user in sharingSet
			Node usersNode = sharedNode.getNode(USERS);
			usersNode.getProperty(workspace.getOwner().getPortalLogin()).remove();
			
			session.save();
			
			
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public List<String> getUsers() throws InternalErrorException {
		
		ArrayList<String> list = new ArrayList<String>();
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(getId());
			Node usersNode = node.getNode(USERS);
			for (PropertyIterator iterator = usersNode.getProperties(); iterator.hasNext();) {
				Property property  = iterator.nextProperty();
				String name = property.getName();
				if (!name.startsWith("jcr:") && !name.startsWith("hl:"))
					list.add(name);
			} 
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		return list;
	}

	private void addUser(Node sharedNode, String user, String destinationFolderId) throws InternalErrorException {
		
		try {
			Session	session = sharedNode.getSession();

			HomeManager homeManager = workspace.getHome().getHomeManager();
			WorkspaceFolder userRoot = (WorkspaceFolder)homeManager.getHome(user).getWorkspace().getItem(destinationFolderId);
			
			Node rootNode =  session.getNodeByIdentifier(userRoot.getId());
			
			String sharedFolderName = WorkspaceUtil.getUniqueName(
					sharedNode.getProperty(TITLE).getString(),
						userRoot);

			String pathUser = rootNode.getPath() + workspace.getPathSeparator() 
					+ sharedFolderName;
			
			Node usersNode = sharedNode.getNode(USERS);
			
			try {
				if (usersNode.getProperty(user) != null)
					return;
			} catch (PathNotFoundException e) {
				logger.debug("User is not present");
			}
			
			session.getWorkspace().clone(session.getWorkspace().getName(),
					sharedNode.getPath(),pathUser, false);
			
			String value = userRoot.getId() + workspace.getPathSeparator() 
			+ sharedFolderName;
			usersNode.setProperty(user, value);
			session.save();
	
			
		} catch (Exception e) {
			throw new InternalErrorException(e);
		}
		
	}
	
	@Override
	public void addUser(String user) throws InsufficientPrivilegesException,
			InternalErrorException {
		
		 Session session = JCRRepository.getSession(); 
		 try {	 
			 Node sharedNode = session.getNodeByIdentifier(getId());
			 Node usersNode = sharedNode.getNode(USERS);
			 
			 try {
				 if (usersNode.getProperty(user) != null)
					 return;
			 } catch (PathNotFoundException e) {
				 logger.debug("User "+ user + " is not present");
			 }

			 HomeManager homeManager = workspace.getHome().getHomeManager();
			 
			 Home home = homeManager.getHome(user);
		
			
			 
			 addUser(sharedNode,user, home.getWorkspace().getRoot().getId());
			 
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			 session.logout();
		}
	}

	@Override
	public WorkspaceItemType getType() {
		return WorkspaceItemType.SHARED_FOLDER;
	}

	@Override
	public WorkspaceFolder unShare() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			
			Node userNode = getUserNode(node);
			
			Node unsharedNode = unShareNode(node, userNode.getParent());
			
			return new JCRWorkspaceFolder(workspace, unsharedNode);
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
	}

	@Override
	public WorkspaceSharedFolder share(List<String> users) throws InsufficientPrivilegesException,
	WrongDestinationException, InternalErrorException {
	
		for (String user : users) {
			addUser(user);
		}
		return this;
	}
	
}
