/****************************************************************************
 *  This software is part of the gCube Project.
 *  Site: http://www.gcube-system.org/
 ****************************************************************************
 * The gCube/gCore software is licensed as Free Open Source software
 * conveying to the EUPL (http://ec.europa.eu/idabc/eupl).
 * The software and documentation is provided by its authors/distributors
 * "as is" and no expressed or
 * implied warranty is given for its use, quality or fitness for a
 * particular case.
 ****************************************************************************
 * Filename: UpgradeListener.java
 ****************************************************************************
 * @author <a href="mailto:daniele.strollo@isti.cnr.it">Daniele Strollo</a>
 ***************************************************************************/

package org.gcube.resourcemanagement.support.managers.services;

import org.gcube.resourcemanagement.support.utils.ServerConsole;

/**
 * @author Daniele Strollo (ISTI-CNR)
 *
 */

/**
 * To an upgrade process can be attached an handler to perform
 * additional operations at the happening of events.
 * @author Daniele Strollo (ISTI-CNR)
 *
 */
public abstract class UpgradeListener {
	protected static final String LOG_PREFIX = "[SW-UPGR-LISTENER]";
	private SWRepositoryUpgrader upgrader =  null;

	public UpgradeListener() {
		this.onInit();
	}

	/**
	 * Method invoked at construction of listener.
	 */
	public abstract void onInit();

	/**
	 * Automatically invoked when the listener is attached to an upgrader.
	 *
	 * @param upgrader
	 */
	public final void setUpgrader(final SWRepositoryUpgrader upgrader) {
		this.upgrader = upgrader;
	}

	public final SWRepositoryUpgrader getUpgrader() {
		return this.upgrader;
	}

	/**
	 * Invoked each time the update progresses.
	 * @param progress
	 */
	public abstract void onProgress(final double progress);
	/**
	 * Invoked in case of failure during update operation.
	 * @param failure
	 */
	public final void onError(final Exception failure) {
		ServerConsole.error(LOG_PREFIX, failure);
	}
	/**
	 * Invoked in case of failure during update operation.
	 * @param failure
	 */
	public final void onError(final String message, final Exception failure) {
		ServerConsole.error(LOG_PREFIX, message, failure);
	}

	/**
	 * In successful update, the resulting report files will be
	 * provided in input.
	 */
	public abstract void onFinish(
			final String deployURL,
			final String reportFileName,
			final String distributionFileName);

	/**
	 * Invoked once the lock has been acquired.
	 */
	public final void onReserve() {
		ServerConsole.info(LOG_PREFIX, "Software repository lock request [DONE]");
	}

	/**
	 * Once finished the lock is released and this event raised.
	 * The release is called both if the process successfully finish
	 * or an abort is required.
	 */
	public void onRelease() {
		ServerConsole.info(LOG_PREFIX, "Software repository lock release [DONE]");
	}
}
