/**
 * 
 */
package org.gcube.portlets.user.warmanagementwidget.server.management;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.Collections;
import java.util.zip.GZIPOutputStream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.compress.archivers.tar.TarEntry;
import org.apache.commons.compress.archivers.tar.TarOutputStream;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.SimpleLayout;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.resources.GCUBEService;
import org.gcube.common.core.resources.common.PlatformDescription;
import org.gcube.common.core.resources.service.Software;
import org.gcube.common.core.resources.service.Package.GHNRequirement;
import org.gcube.common.core.resources.service.Package.ScopeLevel;
import org.gcube.common.core.resources.service.Package.GHNRequirement.Category;
import org.gcube.common.core.resources.service.Package.GHNRequirement.OpType;
import org.gcube.common.core.resources.service.Software.Type;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.security.GCUBESecurityManager;
import org.gcube.portlets.user.warmanagementwidget.client.data.WarProfile;
import org.gcube.portlets.user.warmanagementwidget.client.progress.OperationProgress;
import org.gcube.portlets.user.warmanagementwidget.client.progress.OperationState;
import org.gcube.portlets.user.warmanagementwidget.client.upload.progress.CreationStates;
import org.gcube.portlets.user.warmanagementwidget.server.accesslog.AccessLogUtil;
import org.gcube.portlets.user.warmanagementwidget.server.util.Util;
import org.gcube.vremanagement.softwarerepository.stubs.SoftwareRepositoryPortType;
import org.gcube.vremanagement.softwarerepository.stubs.Store;
import org.gcube.vremanagement.softwarerepository.stubs.StoreItem;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * @author Federico De Faveri defaveri@isti.cnr.it
 *
 */
public class GCubeWarUploader implements Runnable{

	protected static final String SVN_PATH_TXT = "https://svn.research-infrastructures.eu/d4science/gcube/trunk/";

	protected static final String SERVICE_VERSION = "1.0.0";

	protected static final String SERVICE_CLASS = "WebApp";

	protected Logger logger = Logger.getLogger(GCubeWarUploader.class);

	protected OperationProgress operationProgress;

	protected String user;
	protected String sessionId;
	protected File warFile;
	protected WarProfile profile;
	protected String baseUrl;
	protected GCUBEScope scope;
	protected GCUBESecurityManager securityManager;

	/**
	 * @param operationProgress
	 * @param sessionId
	 * @param warFile
	 * @param profile
	 * @param baseUrl
	 */
	public GCubeWarUploader(OperationProgress operationProgress,
			String user, GCUBEScope scope, GCUBESecurityManager securityManager,
			String sessionId, File warFile, WarProfile profile, String baseUrl) {

		this.operationProgress = operationProgress;
		this.user = user;
		this.sessionId = sessionId;
		this.warFile = warFile;
		this.profile = profile;
		this.baseUrl = baseUrl;
		this.scope = scope;
		this.securityManager = securityManager;
		
		ConsoleAppender ca = new ConsoleAppender(new SimpleLayout());
		ca.setThreshold(Level.ALL);
		ca.activateOptions();
		logger.addAppender(ca);
		logger.setLevel(Level.ALL);
	}

	public GCubeWarUploader() {
	}

	@Override
	public void run() {
		
		operationProgress.setElaboratedLenght(CreationStates.CREATE_STATE_SA_CREATION);

		logger.trace("creating SA");
		File saFile;
		try {
			saFile = createServiceArchive(profile, warFile);
		} catch (Exception e) {
			logger.error("An error occured creating the SA", e);
			operationProgress.setFailed("An error occured creating the Service Archive", Util.exceptionDetailMessage(e));
			return ;
		}

		logger.trace("publishing the SA file locally");
		SAPublisher.getInstance().publishSA(sessionId, saFile);

		operationProgress.setElaboratedLenght(CreationStates.CREATE_STATE_STORE);

		//workaround for SR which require the file name in the sa URL and no https
		if (baseUrl.startsWith("https")) baseUrl = "http"+baseUrl.substring(5);
		//if (baseUrl.startsWith("http")) baseUrl = "https"+baseUrl.substring(4);
		String saUrl = baseUrl+"SAServlet?saId="+sessionId+"/"+profile.getApplicationName()+".tar.gz";
		logger.trace("SA url: "+saUrl);
		
		
		logger.trace("retrieving Software Repository Service in scope "+scope);
		SoftwareRepositoryPortType sr;
		try {
			sr = ISUtil.getSoftwareRepositoryPortType(securityManager, scope);
		} catch (Exception e) {
			logger.error("An error occured retrieving the SR epr", e);
			operationProgress.setFailed("An error occured retrieving the Software Repository instance", Util.exceptionDetailMessage(e));
			return;
		}
		
		logger.trace("storing the SA on the SR");
		String storeResult;
		try {
			storeResult = storeServiceArchive(sr, saUrl);
		} catch (Exception e) {
			logger.error("An error occured storing the SA", e);
			operationProgress.setFailed("An error occured storing the SA", Util.exceptionDetailMessage(e));
			return;
		}
		
		logger.trace("parsing result: "+storeResult);
		SoftwareRepositoryResult result;
		try {
			result = parseSoftwareRepositoryResult(storeResult);
		} catch (Exception e) {
			logger.error("An error occured parsing the SR result", e);
			operationProgress.setFailed("An error occured parsing the SR report result", Util.reportDetailMessage(e, storeResult));
			return;
		}
		
		if (result.getStatus() == STATUS.ERROR) {
			logger.error("Store failed. Report: "+result.getReport());
			operationProgress.setFailed("Store failed", Util.reportDetailMessage(storeResult));
			return;
		}

		
		logger.trace("Approving the Software Archive");
		
		operationProgress.setElaboratedLenght(CreationStates.CREATE_STATE_APPROVE);
		
		try {
			approveSoftwareArchive(sr, result.getId());
		} catch (Exception e) {
			logger.error("Approve failed", e);
			operationProgress.setFailed("Approve failed", Util.reportDetailMessage(e, storeResult));
			return;
		}
		
		operationProgress.setState(OperationState.COMPLETED);
		AccessLogUtil.logWarUpload(user, scope, profile);
	}
	
	protected void approveSoftwareArchive(SoftwareRepositoryPortType sr, String id) throws GCUBEFault, RemoteException
	{
		sr.approve(id);
	}
	

	public SoftwareRepositoryResult parseSoftwareRepositoryResult(String result) throws ParserConfigurationException, SAXException, IOException, XPathExpressionException
	{

		DocumentBuilderFactory builderFactory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = builderFactory.newDocumentBuilder();
		Document document = builder.parse(new ByteArrayInputStream(result.getBytes()));

		XPathFactory xPathFactory = XPathFactory.newInstance();
		XPath xPath = xPathFactory.newXPath();
		String expression = "//Status";
		XPathExpression xPathExpression = xPath.compile(expression);
		String status = xPathExpression.evaluate(document);

		logger.trace("status: "+status);

		if (status.equalsIgnoreCase("OK") || status.equalsIgnoreCase("WARNING")) {

			logger.trace("Retrieving id");
			
			expression = "//ID";
			xPathExpression = xPath.compile(expression);
			String id = xPathExpression.evaluate(document);
			logger.trace("id: "+id);
			
			return new SoftwareRepositoryResult(id);
		}

		if (status.equalsIgnoreCase("ERROR")) {

			logger.trace("Retrieving report url");
			expression = "//ReportURL";
			xPathExpression = xPath.compile(expression);
			String reportUrl = xPathExpression.evaluate(document);
			logger.trace("Report url: "+reportUrl);

			String report = result;
			
			if (reportUrl!= null && reportUrl.length()>0) {
				try{
					URL url = new URL(reportUrl);
					InputStream reportIs = url.openStream();
					StringWriter reportWriter = new StringWriter();
					IOUtils.copy(reportIs, reportWriter);
					report = reportWriter.toString();
					logger.trace("report: "+report);
				}catch(Exception e)
				{
					logger.warn("An error occured retrieving the report", e);
				}
			}
			
			return new SoftwareRepositoryResult(STATUS.ERROR, report);
		}

		return new SoftwareRepositoryResult(STATUS.ERROR, result);
	}


	public String storeServiceArchive(SoftwareRepositoryPortType sr, String saUrl) throws Exception
	{
		logger.trace("SoftwareRepository epr: "+sr);

		StoreItem storeItem = new StoreItem();
		storeItem.setURL(saUrl);
		storeItem.setServiceClass(SERVICE_CLASS);
		storeItem.setServiceVersion(SERVICE_VERSION);
		storeItem.setServiceName(profile.getCategoryName());
		storeItem.setDescription(profile.getCategoryDescription());
		storeItem.setScopes(new String[]{scope.toString()});

		Store store = new Store(new StoreItem[]{storeItem});
		logger.trace("calling the service");
		String result = sr.store(store);
		logger.trace("result: "+result);
		return result;
	}

	public File createServiceArchive(WarProfile profile, File warFile) throws Exception
	{
		logger.trace("generating the XML profile");
		String xmlProfile = null;
		try {
			xmlProfile = generateXMLProfile(profile);
		} catch (Exception e) {
			logger.error("Error generating the XML profile", e);
			throw new Exception("Error generating the XML profile", e);
		}

		File saFile = File.createTempFile("serviceArchive", "tmp.tar.gz");
		logger.trace("saving SA in "+saFile.getAbsolutePath());

		BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(saFile));
		TarOutputStream tarOutput = new TarOutputStream(new GZIPOutputStream(bos));
		tarOutput.setLongFileMode(TarOutputStream.LONGFILE_GNU);

		//profile.xml
		logger.trace("adding profile...");
		TarEntry profileEntry = new TarEntry("profile.xml");
		profileEntry.setSize(xmlProfile.length());
		tarOutput.putNextEntry(profileEntry);
		tarOutput.write(xmlProfile.getBytes());
		tarOutput.closeEntry();


		//war file in application directory
		logger.trace("adding war file...");
		String warDir = profile.getApplicationName()+"/";
		TarEntry warFolderEntry = new TarEntry(warDir);
		tarOutput.putNextEntry(warFolderEntry);
		tarOutput.closeEntry();

		TarEntry warEntry = new TarEntry(warDir+profile.getWarFileName());
		warEntry.setSize(warFile.length());
		tarOutput.putNextEntry(warEntry);
		IOUtils.copy(new FileInputStream(warFile), tarOutput);
		tarOutput.closeEntry();

		//svnpath.txt
		TarEntry svnPathEntry = new TarEntry(warDir+"svnpath.txt");
		svnPathEntry.setSize(SVN_PATH_TXT.length());
		tarOutput.putNextEntry(svnPathEntry);
		tarOutput.write(SVN_PATH_TXT.getBytes());
		tarOutput.closeEntry();

		tarOutput.close();
		logger.trace("SA complete");

		return saFile;
	}

	public static String generateXMLProfile(WarProfile profile) throws Exception
	{

		GCUBEService gcubeService = GHNContext.getImplementation(GCUBEService.class);

		/*	<Description>**DESCRIPTION**</Description>
		<Class>WebApp</Class>
		<Name>**APP CATEGORY**</Name>
		<Version>1.0.0</Version>
		 */
		gcubeService.setServiceName(profile.getCategoryName());
		gcubeService.setDescription(profile.getCategoryDescription());
		gcubeService.setServiceClass(SERVICE_CLASS);
		gcubeService.setVersion(SERVICE_VERSION);

		/*	
		<Packages>
			<Software>
				<Description>**DESCRIPTION**</Description>
				<Name>**APP NAME**</Name>
				<Version>**VERSION**</Version>
		 */
		Software warPackage = new Software();
		warPackage.setName(profile.getApplicationName());
		warPackage.setDescription(profile.getApplicationDescription());

		warPackage.setVersion(Util.getVersion(profile));

		/*	
		<TargetPlatform>
			<Name>Tomcat</Name>
			<Version>6</Version>
			<MinorVersion>0</MinorVersion>
		</TargetPlatform>
		 */
		PlatformDescription targetPlatform = new PlatformDescription();
		targetPlatform.setName("Tomcat");
		targetPlatform.setVersion((short) 6);
		targetPlatform.setMinorVersion((short) 0);
		warPackage.setTargetPlatform(targetPlatform);

		/*
		 *					<MultiVersion value="true" />
					<Mandatory level="NONE" />
					<Shareable level="NONE" />
					<GHNRequirements>
						<Requirement category="Site" operator="ge" requirement="string" value="java1.6" />
					</GHNRequirements>
					<SpecificData/>
					<Type>webapplication</Type>
		 */

		warPackage.setMultiVersion(true);
		warPackage.setMandatoryLevel(ScopeLevel.NONE);
		warPackage.setSharingLevel(ScopeLevel.NONE);

		GHNRequirement requirement = new GHNRequirement();
		requirement.setCategory(Category.SITE_LOCATION);
		requirement.setOperator(OpType.GE);
		requirement.setRequirement("string");
		requirement.setValue("java1.6");
		warPackage.setGHNRequirements(Collections.singletonList(requirement));
		warPackage.setType(Type.webapplication);

		/*
		<EntryPoint>**APPLICATION ENTRY POINT 1**</EntryPoint>	
		<EntryPoint>**APPLICATION ENTRY POINT N**</EntryPoint>
					<Files>
							<File>**NAME OF THE WAR**</File>
						</Files>
					</Software>
				</Packages>
			</Profile>
		</Resource>*/

		warPackage.getEntrypoints().addAll(profile.getEntryPoints());
		warPackage.getFiles().add(profile.getWarFileName());
		gcubeService.getPackages().add(warPackage);

		StringWriter xml = new StringWriter();
		gcubeService.store(xml);

		return xml.toString();
	}

	public static enum STATUS {ERROR, OK};

	protected class SoftwareRepositoryResult {
		protected STATUS status;
		protected String report;
		protected String id;
		
		

		/**
		 * @param id
		 */
		public SoftwareRepositoryResult(String id) {
			this.id = id;
			status = STATUS.OK;
		}

		/**
		 * @param status
		 * @param report
		 */
		public SoftwareRepositoryResult(STATUS status, String report) {
			this.status = status;
			this.report = report;
		}

		/**
		 * @return the status
		 */
		public STATUS getStatus() {
			return status;
		}

		/**
		 * @param status the status to set
		 */
		public void setStatus(STATUS status) {
			this.status = status;
		}

		/**
		 * @return the report
		 */
		public String getReport() {
			return report;
		}

		/**
		 * @param report the report to set
		 */
		public void setReport(String report) {
			this.report = report;
		}

		/**
		 * @return the id
		 */
		public String getId() {
			return id;
		}

		/**
		 * @param id the id to set
		 */
		public void setId(String id) {
			this.id = id;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public String toString() {
			return "SoftwareRepositoryResult [status=" + status + ", report="
					+ report + ", id=" + id + "]";
		}
	}
}
