package org.gcube.portal.social.networking.orchestrator;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.json.simple.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.ObjectMapper;

public class OrchestratorManager {

    private static final Logger logger = LoggerFactory.getLogger(OrchestratorManager.class);

    private OrchestratorAPICredentials credentials;

    /**
     * Private constructor to prevent instantiation.
     * It initializes the credentials for the Orchestrator.
     */
    private OrchestratorManager() {
        try {
            this.credentials = OrchestratorAPICredentials.getInstance();
        } catch (Exception e) {
            logger.error("Failed to initialize OrchestratorManager", e);
            throw new RuntimeException("Failed to initialize OrchestratorManager", e);
        }
    }

    /**
     * Holder class for lazy-loaded singleton instance.
     * This is a thread-safe way to implement a singleton.
     */
    private static class Holder {
        private static final OrchestratorManager INSTANCE = new OrchestratorManager();
    }

    /**
     * Returns the singleton instance of the OrchestratorManager.
     *
     * @return the singleton instance
     */
    public static OrchestratorManager getInstance() {
        return Holder.INSTANCE;
    }

    /**
     * Invokes a workflow on the Conductor server. This method is a generic
     * client for the Conductor API's workflow endpoint.
     *
     * @param workflowName The name of the workflow to invoke.
     * @param workflowInput A map of input parameters for the workflow.
     * @return The response body from the Conductor server.
     * @throws Exception if the workflow invocation fails.
     */
    private String invokeConductorWorkflow(String workflowName, Map<String, String> workflowInput) throws Exception {
        String url = credentials.getServerURL();
        String auth = credentials.getUsername() + ":" + credentials.getPassword();
        
        String encodedAuth = Base64.getEncoder().encodeToString(auth.getBytes());
        String authHeader = "Basic " + encodedAuth;

        JSONObject inputJson = new JSONObject(workflowInput);

        JSONObject payload = new JSONObject();
        payload.put("name", workflowName);
        payload.put("input", inputJson);

        try (CloseableHttpClient client = HttpClients.createDefault()) {
            HttpPost httpPost = new HttpPost(url);

            httpPost.setHeader("Content-Type", "application/json");
            httpPost.setHeader("Authorization", authHeader);
            httpPost.setEntity(new StringEntity(payload.toJSONString()));

            HttpResponse response = client.execute(httpPost);
            String responseBody = EntityUtils.toString(response.getEntity());
            
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode < 200 || statusCode >= 300) {
                logger.error("Failed to invoke conductor workflow '{}'. Status: {}, Response: {}", workflowName, statusCode, responseBody);
                throw new Exception("Failed to invoke conductor workflow '" + workflowName + "'. Status: " + statusCode + ", Response: " + responseBody);
            }
            
            logger.info("Successfully invoked conductor workflow '{}'. Task ID: {}", workflowName, responseBody);
            return responseBody;
        }
    }

    /**
     * Schedules the asynchronous addition of a user to a group by invoking the 'user-group_created' workflow.
     * The success of this method indicates that the operation has been successfully scheduled, not that it has been executed.
     *
     * @param user The username to add.
     * @param group The group to add the user to.
     * @throws Exception if scheduling the operation fails.
     */
    public String addUserToGroup(String user, String group) throws Exception {
        Map<String, String> input = new HashMap<>();
        input.put("user", user);
        input.put("group", group);
        return invokeConductorWorkflow("user-group_created", input);
    }

    /**
     * Schedules the asynchronous removal of a user from a group by invoking the 'user-group_deleted' workflow.
     * The success of this method indicates that the operation has been successfully scheduled, not that it has been executed.
     *
     * @param user The username to remove.
     * @param group The group to remove the user from.
     * @throws Exception if scheduling the operation fails.
     */
    public String removeUserFromGroup(String user, String group) throws Exception {
        Map<String, String> input = new HashMap<>();
        input.put("user", user);
        input.put("group", group);
        return invokeConductorWorkflow("user-group_deleted", input);
    }
    /**
     * Retrieves the status of a specific workflow from the Conductor server.
     *
     * @param workflowId The ID of the workflow to retrieve.
     * @return A map representing the JSON response from the Conductor server.
     * @throws Exception if the API call fails.
     */
    public Map<String, Object> getWorkflowStatus(String workflowId) throws Exception {
        String url = credentials.getServerURL();
        url += "/" + URLEncoder.encode(workflowId, StandardCharsets.UTF_8.name());

        String auth = credentials.getUsername() + ":" + credentials.getPassword();
        String encodedAuth = Base64.getEncoder().encodeToString(auth.getBytes());
        String authHeader = "Basic " + encodedAuth;

        try (CloseableHttpClient client = HttpClients.createDefault()) {
            HttpGet httpGet = new HttpGet(url);
            httpGet.setHeader("Authorization", authHeader);

            HttpResponse response = client.execute(httpGet);
            String responseBody = EntityUtils.toString(response.getEntity());

            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode < 200 || statusCode >= 300) {
                logger.error("Failed to get workflow status for '{}'. Status: {}, Response: {}", workflowId, statusCode, responseBody);
                throw new Exception("Failed to get workflow status for '" + workflowId + "'. Status: " + statusCode + ", Response: " + responseBody);
            }

            logger.info("Successfully retrieved workflow status for '{}'.", workflowId);
            ObjectMapper mapper = new ObjectMapper();
            return mapper.readValue(responseBody, Map.class);
        }
    }
}
