package org.gcube.portal.social.networking.orchestrator;

import static org.gcube.resources.discovery.icclient.ICFactory.clientFor;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;

import java.util.Iterator;
import java.util.List;

import org.gcube.common.encryption.encrypter.StringEncrypter;
import org.gcube.common.resources.gcore.ServiceEndpoint;
import org.gcube.common.resources.gcore.ServiceEndpoint.AccessPoint;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.api.SimpleQuery;
import org.gcube.smartgears.ContextProvider;
import org.gcube.smartgears.context.application.ApplicationContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A singleton class responsible for retrieving and holding the API credentials
 * for the Orchestrator service. It fetches the service endpoint details from the
 * Information System (IS) and provides them to the application.
 * The singleton is implemented using a thread-safe, lazy initialization pattern
 * with double-checked locking.
 */
public class OrchestratorAPICredentials {

    private static final Logger logger = LoggerFactory.getLogger(OrchestratorAPICredentials.class);


    /**
     * The volatile keyword ensures that multiple threads handle the instance variable correctly
     * when it is being initialized to the singleton instance.
     * In particular, it guarantees that the write to the instance variable happens-before
     * every subsequent read of that same variable.
     */
    private static volatile OrchestratorAPICredentials instance;

    // properties that it contains
    private String orchestrator_url;
    private String username;
    private String password;

    // Service endpoint properties
    private final static String RUNTIME_RESOURCE_NAME = "Orchestrator";
    private final static String IS_AUTH_CATEGORY = "Service";
    private final static String ENDPOINT = "d4science.event-broker-endpoint";


    /**
     * Private constructor
     */
    private OrchestratorAPICredentials() {
        logger.debug("Building OrchestratorAOICredentials object");

        lookupPropertiesFromIs();
        logger.debug("OrchestratorAOICredentials object built");
    }

    /**
     * Read the properties from the infrastructure
     */
    private void lookupPropertiesFromIs() {

        logger.debug("Starting creating OrchestratorAOICredentials");

        String oldContext = ScopeProvider.instance.get();
        ApplicationContext ctx = ContextProvider.get(); // get this info from SmartGears
        ScopeProvider.instance.set("/" + ctx.container().configuration().infrastructure());
        logger.debug("Discovering orchestrator's credentials in context "
                + ctx.container().configuration().infrastructure());

        try {
            List<ServiceEndpoint> resources = getConfigurationFromIS();
            if (resources.size() == 0) {
                logger.error("There is no Runtime Resource having name " + RUNTIME_RESOURCE_NAME + " and Category "
                        + IS_AUTH_CATEGORY + " in this scope.");
                throw new Exception("There is no Runtime Resource having name " + RUNTIME_RESOURCE_NAME
                        + " and Category " + IS_AUTH_CATEGORY + " in this scope.");
            } else {
                for (ServiceEndpoint res : resources) {
                    Iterator<AccessPoint> accessPointIterator = res.profile().accessPoints().iterator();
                    while (accessPointIterator.hasNext()) {
                        ServiceEndpoint.AccessPoint accessPoint = (ServiceEndpoint.AccessPoint) accessPointIterator
                                .next();

                        if (accessPoint.name().equals(ENDPOINT)){
                            orchestrator_url = accessPoint.address();
                            username = accessPoint.username();
                            password = StringEncrypter.getEncrypter().decrypt(accessPoint.password());
                            logger.info("Found accesspoint URL = " + orchestrator_url);
                        }
                    }
                }
            }
        } catch (Exception e) {
            logger.error("Unable to retrieve such service endpoint information!", e);
            return;
        } finally {
            if (oldContext != null)
                ScopeProvider.instance.set(oldContext);
        }

        logger.debug("Bean built " + toString());
    }

    /**
     * Retrieve endpoints information from IS for DB
     * 
     * @return list of endpoints for ckan database
     * @throws Exception
     */
    private List<ServiceEndpoint> getConfigurationFromIS() throws Exception {

        SimpleQuery query = queryFor(ServiceEndpoint.class);
        query.addCondition("$resource/Profile/Name/text() eq '" + RUNTIME_RESOURCE_NAME + "'");
        query.addCondition("$resource/Profile/Category/text() eq '" + IS_AUTH_CATEGORY + "'");
        DiscoveryClient<ServiceEndpoint> client = clientFor(ServiceEndpoint.class);
        List<ServiceEndpoint> toReturn = client.submit(query);
        return toReturn;

    }

    /**
     * Returns the singleton instance of the OrchestratorAPICredentials.
     * It uses double-checked locking for thread-safe lazy initialization.
     *
     * @return the singleton instance
     */
    public static OrchestratorAPICredentials getInstance() {
        if (instance == null) {
            synchronized (OrchestratorAPICredentials.class) {
                if (instance == null) {
                    instance = new OrchestratorAPICredentials();
                }
            }
        }
        return instance;
    }

    /**
     * Gets the Orchestrator service URL.
     *
     * @return The server URL.
     */
    public String getServerURL() {
        String url = orchestrator_url;
        if (url.endsWith("/")) {
            url = url.substring(0, url.length() - 1);
        }

        return url;
    }

    /**
     * Gets the username for the Orchestrator service.
     *
     * @return The username.
     */
    public String getUsername() {
        return username;
    }

    /**
     * Gets the decrypted password for the Orchestrator service.
     *
     * @return The password.
     */
    public String getPassword() {
        return password;
    }

    @Override
    public String toString() {
        return "OrchestratorAPICredentials [orchestrator_url=" + orchestrator_url + ", username=" + username + ", password=****]";
    }

}
