package org.gcube.portal.social.networking.ws.methods.v2;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import javax.validation.ValidationException;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import org.gcube.applicationsupportlayer.social.ApplicationNotificationsManager;
import org.gcube.applicationsupportlayer.social.NotificationsManager;
import org.gcube.applicationsupportlayer.social.shared.SocialNetworkingSite;
import org.gcube.applicationsupportlayer.social.shared.SocialNetworkingUser;
import org.gcube.common.authorization.library.provider.AuthorizationProvider;
import org.gcube.common.authorization.library.provider.SecurityTokenProvider;
import org.gcube.common.authorization.library.utils.Caller;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.common.storagehub.client.plugins.AbstractPlugin;
import org.gcube.common.storagehub.client.proxies.MessageManagerClient;
import org.gcube.common.storagehub.model.messages.Message;
import org.gcube.portal.notifications.bean.GenericItemBean;
import org.gcube.portal.notifications.thread.MessageNotificationsThread;
import org.gcube.portal.social.networking.caches.SocialNetworkingSiteFinder;
import org.gcube.portal.social.networking.liferay.ws.LiferayJSONWsCredentials;
import org.gcube.portal.social.networking.liferay.ws.UserManagerWSBuilder;
import org.gcube.portal.social.networking.ws.inputs.MessageInputBean;
import org.gcube.portal.social.networking.ws.inputs.Recipient;
import org.gcube.portal.social.networking.ws.outputs.ResponseBean;
import org.gcube.portal.social.networking.ws.utils.ErrorMessages;
import org.gcube.portal.social.networking.ws.utils.TokensUtils;
import org.gcube.vomanagement.usermanagement.exception.UserManagementSystemException;
import org.gcube.vomanagement.usermanagement.exception.UserRetrievalFault;
import org.gcube.vomanagement.usermanagement.model.GCubeUser;
import org.slf4j.LoggerFactory;

import com.webcohesion.enunciate.metadata.rs.RequestHeader;
import com.webcohesion.enunciate.metadata.rs.RequestHeaders;
import com.webcohesion.enunciate.metadata.rs.ResponseCode;
import com.webcohesion.enunciate.metadata.rs.StatusCodes;

/**
 * Messages services REST interface
 * @author Costantino Perciante at ISTI-CNR 
 * (costantino.perciante@isti.cnr.it)
 */
@Path("2/messages")
@RequestHeaders ({
	  @RequestHeader( name = "Authorization", description = "Bearer token, see https://dev.d4science.org/how-to-access-resources"),
	  @RequestHeader( name = "Content-Type", description = "application/json")	  
	})
public class Messages {

	// Logger
	private static final org.slf4j.Logger logger = LoggerFactory.getLogger(Messages.class);

	/**
	 * Write a message to another user. The sender is the token's owner by default
	 * @responseExample application/json {"success": true, "message": null, "result": "556142e3-d6f5-4550-b2fa-abe5626625d3"}
	 * @param input The message to write"
	 * @param httpServletRequest
	 * @return see response example
	 * @throws ValidationException
	 * @throws UserManagementSystemException
	 * @throws UserRetrievalFault
	 */
	@POST
	@Path("write-message/")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	@StatusCodes ({
		@ResponseCode ( code = 200, condition = "Successful write a message. Its id is reported in the 'result' field of the returned object"),
		@ResponseCode ( code = 500, condition = ErrorMessages.ERROR_IN_API_RESULT)
	})
	public Response writeMessage(
			@NotNull(message="Message to send is missing") 
			@Valid 
			MessageInputBean input,
			@Context HttpServletRequest httpServletRequest) throws ValidationException, UserManagementSystemException, UserRetrievalFault{

		logger.debug("Incoming message bean is " + input);

		Caller caller = AuthorizationProvider.instance.get();

		// check if the token belongs to an application token. In this case use J.A.R.V.I.S (the username used to communicate with Liferay)
		String senderId = null;
		if(!TokensUtils.isUserToken(caller)){
			GCubeUser jarvis = UserManagerWSBuilder.getInstance().getUserManager().getUserByEmail(LiferayJSONWsCredentials.getSingleton().getUser());
			SecurityTokenProvider.instance.set(LiferayJSONWsCredentials.getSingleton().getNotifierUserToken());
			senderId = jarvis.getUsername();
		}else{
			senderId = caller.getClient().getId();
		}
		String scope = ScopeProvider.instance.get();
		ResponseBean responseBean = new ResponseBean();
		Status status = Status.CREATED;
		String body = input.getBody();
		String subject = input.getSubject();
		List<Recipient> recipientsIds = input.getRecipients(); // "recipients":[{"recipient":"id recipient"}, ......]
		logger.info("Sender is going to be the token's owner [" + senderId  + "]");

		// get the recipients ids (simple check, trim)
		List<String> recipientsListFiltered = new ArrayList<String>();
		List<GenericItemBean> recipientsBeans = new ArrayList<GenericItemBean>();
		for (Recipient recipientId : recipientsIds) {
			try{
				String tempId = recipientId.getId().trim();
				if(tempId.isEmpty())
					continue;
				GCubeUser userRecipient = UserManagerWSBuilder.getInstance().getUserManager().getUserByUsername(tempId);				
				if(userRecipient == null)
					userRecipient = UserManagerWSBuilder.getInstance().getUserManager().getUserByEmail(tempId);
				if(userRecipient != null){
					GenericItemBean beanUser = new GenericItemBean(userRecipient.getUsername(), userRecipient.getUsername(), userRecipient.getFullname(), userRecipient.getUserAvatarURL());
					recipientsBeans.add(beanUser);
					recipientsListFiltered.add(userRecipient.getUsername());
				}
			}catch(Exception e){
				logger.error("Unable to retrieve recipient information for recipient with id " + recipientId, e);
			}
		}

		if(recipientsListFiltered.isEmpty()){
			logger.error("Missing/wrong request parameters");
			status = Status.BAD_REQUEST;
			responseBean.setMessage(ErrorMessages.MISSING_PARAMETERS);
			return Response.status(status).entity(responseBean).build();
		}

		try{

			logger.debug("Trying to send message with body " + body + " subject " + subject + " to users " + recipientsIds + " from " + senderId);

			// sender info
			GCubeUser senderUser = UserManagerWSBuilder.getInstance().getUserManager().getUserByUsername(senderId);
			MessageManagerClient client = AbstractPlugin.messages().build();

			// send message
			logger.debug("Sending message to " + recipientsListFiltered);
			String messageId = client.sendMessage(recipientsListFiltered, subject, body, null);

			// send notification
			logger.debug("Message sent to " + recipientsIds + ". Sending message notification to: " + recipientsIds);
			SocialNetworkingSite site = SocialNetworkingSiteFinder.getSocialNetworkingSiteFromScope(scope);
			SocialNetworkingUser user = new SocialNetworkingUser(
					senderUser.getUsername(), senderUser.getEmail(),
					senderUser.getFullname(), senderUser.getUserAvatarURL());

			NotificationsManager nm = new ApplicationNotificationsManager(UserManagerWSBuilder.getInstance().getUserManager(), site, ScopeProvider.instance.get(), user);
			new Thread(new MessageNotificationsThread(recipientsBeans, messageId, subject, body, nm)).start();
			responseBean.setSuccess(true);
			responseBean.setResult(messageId);

		}catch(Exception e){
			logger.error("Unable to send message.", e);
			status = Status.INTERNAL_SERVER_ERROR;
			responseBean.setMessage(e.toString());
		}
		return Response.status(status).entity(responseBean).build();
	}

	@GET
	@Path("get-sent-messages")
	@Produces(MediaType.APPLICATION_JSON)
	@StatusCodes ({
		@ResponseCode ( code = 200, condition = "Successful read of the sent messages, reported in the 'result' field of the returned object"),
		@ResponseCode ( code = 500, condition = ErrorMessages.ERROR_IN_API_RESULT)
	})
	/**
	 * 
	 * @return the list of sent messages of the user (the token's owner)
	 */
	public Response getSentMessages(){

		Caller caller = AuthorizationProvider.instance.get();
		String username = caller.getClient().getId();
		ResponseBean responseBean = new ResponseBean();
		Status status = Status.OK;

		logger.info("Request for retrieving sent messages by " + username);

		try{
			MessageManagerClient client = AbstractPlugin.messages().build();
			List<Message> sentMessages = client.getSentMessages();
			Collections.reverse(sentMessages);
			responseBean.setSuccess(true);
			logger.debug("Result is " + sentMessages);
			responseBean.setResult(sentMessages);
		}catch(Exception e){
			logger.error("Unable to retrieve sent messages", e);
			responseBean.setMessage(e.getMessage());
			status = Status.INTERNAL_SERVER_ERROR;
		}

		return Response.status(status).entity(responseBean).build();
	}

	@GET
	@Path("get-received-messages")
	@Produces(MediaType.APPLICATION_JSON)
	@StatusCodes ({
		@ResponseCode ( code = 200, condition = "Successful read of the received messages, reported in the 'result' field of the returned object"),
		@ResponseCode ( code = 500, condition = ErrorMessages.ERROR_IN_API_RESULT)
	})
	/**
	 * 
	 * @return the list of received messages of the user (the token's owner)
	 */
	public Response getReceivedMessages(){

		Caller caller = AuthorizationProvider.instance.get();
		String username = caller.getClient().getId();
		ResponseBean responseBean = new ResponseBean();
		Status status = Status.OK;

		logger.info("Request for retrieving received messages by " + username);
		try{
			MessageManagerClient client = AbstractPlugin.messages().build();
			List<Message> getMessages =client.getReceivedMessages();
			Collections.reverse(getMessages);
			responseBean.setSuccess(true);
			responseBean.setResult(getMessages);

		}catch(Exception e){
			logger.error("Unable to retrieve sent messages", e);
			responseBean.setMessage(e.getMessage());
			status = Status.INTERNAL_SERVER_ERROR;
		}

		return Response.status(status).entity(responseBean).build();
	}
}
