package org.gcube.portal.social.networking.ws.methods.v2;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.validation.ValidationException;
import javax.validation.constraints.NotNull;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import org.gcube.common.authorization.library.provider.AuthorizationProvider;
import org.gcube.common.authorization.library.utils.Caller;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.common.scope.impl.ScopeBean;
import org.gcube.common.scope.impl.ScopeBean.Type;
import org.gcube.portal.social.networking.caches.UsersCache;
import org.gcube.portal.social.networking.liferay.ws.GroupManagerWSBuilder;
import org.gcube.portal.social.networking.liferay.ws.RoleManagerWSBuilder;
import org.gcube.portal.social.networking.liferay.ws.UserManagerWSBuilder;
import org.gcube.portal.social.networking.ws.utils.TokensUtils;
import org.gcube.portal.social.networking.ws.utils.UserProfileExtendedWithVerifiedEmail;
import org.gcube.social_networking.socialnetworking.model.beans.UserProfileExtended;
import org.gcube.social_networking.socialnetworking.model.output.ResponseBean;
import org.gcube.vomanagement.usermanagement.GroupManager;
import org.gcube.vomanagement.usermanagement.RoleManager;
import org.gcube.vomanagement.usermanagement.UserManager;
import org.gcube.vomanagement.usermanagement.model.GCubeRole;
import org.gcube.vomanagement.usermanagement.model.GCubeUser;
import org.slf4j.LoggerFactory;

/**
 * REST interface for the social networking library (users).
 * @author Costantino Perciante at ISTI-CNR
 */
@Path("2/users")
public class Users {

	// Logger
	private static final org.slf4j.Logger logger = LoggerFactory.getLogger(Users.class);
	private static final String NOT_USER_TOKEN_CONTEXT_USED = "User's information can only be retrieved through a user token (not qualified)";
	private static final List<String> GLOBAL_ROLES_ALLOWED_BY_LOCAL_CALL_METHOD = Arrays.asList("DataMiner-Manager","Quota-Manager");

	@GET
	@Path("get-custom-attribute/")
	@Produces(MediaType.APPLICATION_JSON)
	public Response readCustomAttr(
			@QueryParam("attribute") 
			@NotNull(message="attribute name is missing") 
			String attributeKey
			) throws ValidationException {

		Caller caller = AuthorizationProvider.instance.get();
		String username = caller.getClient().getId();
		ResponseBean<String> responseBean = new ResponseBean<String>();
		Status status = Status.OK;
		if(!TokensUtils.isUserTokenDefault(caller)){
			status = Status.FORBIDDEN;
			responseBean.setMessage(NOT_USER_TOKEN_CONTEXT_USED);
			logger.warn("Trying to access users method via a token different than USER is not allowed");
		}else{
			UserManager userManager = UserManagerWSBuilder.getInstance().getUserManager();
			try{

				GCubeUser user = userManager.getUserByUsername(username);
				String toReturn = (String)userManager.readCustomAttr(user.getUserId(), attributeKey);
				responseBean.setSuccess(true);
				responseBean.setResult(toReturn);

			}catch(Exception e){

				logger.error("Unable to retrieve attribute for user.", e);
				responseBean.setMessage(e.toString());
				responseBean.setSuccess(false);
				status = Status.NOT_FOUND;

			}
		}
		return Response.status(status).entity(responseBean).build();
	}

	@GET
	@Path("get-fullname")
	@Produces(MediaType.APPLICATION_JSON)
	public Response getUserFullname(){

		Caller caller = AuthorizationProvider.instance.get();
		String username = caller.getClient().getId();
		String fullName = null;
		ResponseBean<String> responseBean = new ResponseBean<String>();
		Status status = Status.OK;

		if(!TokensUtils.isUserTokenDefault(caller)){
			status = Status.FORBIDDEN;
			responseBean.setMessage(NOT_USER_TOKEN_CONTEXT_USED);
			logger.warn("Trying to access users method via a token different than USER is not allowed");
		}else{
			UserManager userManager = UserManagerWSBuilder.getInstance().getUserManager();
			try{

				GCubeUser user = userManager.getUserByUsername(username);
				fullName = user.getFullname();
				logger.info("Found fullname " + fullName + " for user " +  username);
				responseBean.setResult(fullName);
				responseBean.setSuccess(true);

			}catch(Exception e){

				logger.error("Unable to retrieve attribute for user.", e);
				status = Status.INTERNAL_SERVER_ERROR;

			}
		}
		return Response.status(status).entity(responseBean).build();
	}

	@GET
	@Path("get-email")
	@Produces(MediaType.APPLICATION_JSON)
	public Response getUserEmail(){

		Caller caller = AuthorizationProvider.instance.get();
		String username = caller.getClient().getId();
		String email = null;
		ResponseBean<String> responseBean = new ResponseBean<String>();
		Status status = Status.OK;
		if(!TokensUtils.isUserTokenDefault(caller)){
			status = Status.FORBIDDEN;
			responseBean.setMessage(NOT_USER_TOKEN_CONTEXT_USED);
			logger.warn("Trying to access users method via a token different than USER is not allowed");
		}else{
			try{
				UserManager userManager = UserManagerWSBuilder.getInstance().getUserManager();
				GCubeUser user = userManager.getUserByUsername(username);
				email = user.getEmail();
				logger.info("Found email " + email + " for user " +  username);
				responseBean.setResult(email);
				responseBean.setSuccess(true);

			}catch(Exception e){

				logger.error("Unable to retrieve attribute for user.", e);
				status = Status.INTERNAL_SERVER_ERROR;

			}
		}
		return Response.status(status).entity(responseBean).build();
	}

	@GET
	@Path("get-profile")
	@Produces(MediaType.APPLICATION_JSON)
	public Response getUserProfile(){

		Caller caller = AuthorizationProvider.instance.get();
		String username = caller.getClient().getId();
		String scope = ScopeProvider.instance.get();
		GCubeUser user = null;
		ResponseBean<UserProfileExtended> responseBean = new ResponseBean<UserProfileExtended>();
		Status status = Status.OK;

		if(!TokensUtils.isUserTokenDefault(caller)){
			status = Status.FORBIDDEN;
			responseBean.setMessage(NOT_USER_TOKEN_CONTEXT_USED);
			logger.warn("Trying to access users method via a token different than USER is not allowed");
		}else{
			try{
				UserManager userManager = UserManagerWSBuilder.getInstance().getUserManager();
				RoleManager roleManager = RoleManagerWSBuilder.getInstance().getRoleManager();
				GroupManager groupManager = GroupManagerWSBuilder.getInstance().getGroupManager();
				user = userManager.getUserByUsername(username);
				UserProfileExtended extendedProfile = GCUBE_TO_EXTENDED_PROFILE_MAP.apply(user);
				List<GCubeRole> roles = roleManager.listRolesByUserAndGroup(user.getUserId(), groupManager.getGroupIdFromInfrastructureScope(scope));
				List<String> rolesNames = new ArrayList<String>();
				for (GCubeRole gCubeRole : roles) {
					rolesNames.add(gCubeRole.getRoleName());
				}
				extendedProfile.setRoles(rolesNames);
				responseBean.setResult(extendedProfile);
				responseBean.setSuccess(true);
			}catch(Exception e){
				logger.error("Unable to retrieve user's profile", e);
				responseBean.setMessage(e.getMessage());
				status = Status.INTERNAL_SERVER_ERROR;
			}
		}
		return Response.status(status).entity(responseBean).build();
	}

	@GET
	@Path("get-oauth-profile")
	@Produces(MediaType.APPLICATION_JSON)
	public Response getUserOAuthProfile(){
		Caller caller = AuthorizationProvider.instance.get();
		String username = caller.getClient().getId();
		String scope = ScopeProvider.instance.get();
		GCubeUser user = null;
		ResponseBean<UserProfileExtended> responseBean = new ResponseBean<UserProfileExtended>();
		Status status = Status.OK;
		UserProfileExtendedWithVerifiedEmail userWithEmailVerified = null;
		if(! (TokensUtils.isUserTokenDefault(caller) || TokensUtils.isUserTokenQualified(caller))){
			status = Status.FORBIDDEN;
			responseBean.setMessage(NOT_USER_TOKEN_CONTEXT_USED);
			logger.warn("Trying to access users method via a token different than USER is not allowed");
		}else{
			try{
				UserManager userManager = UserManagerWSBuilder.getInstance().getUserManager();
				RoleManager roleManager = RoleManagerWSBuilder.getInstance().getRoleManager();
				GroupManager groupManager = GroupManagerWSBuilder.getInstance().getGroupManager();
				user = userManager.getUserByUsername(username);
				userWithEmailVerified = GCUBE_TO_EXTENDED_PROFILE_MAP_WITH_VERIFIED_EMAIL.apply(user);
				List<GCubeRole> roles = roleManager.listRolesByUserAndGroup(user.getUserId(), groupManager.getGroupIdFromInfrastructureScope(scope));
				List<String> rolesNames = new ArrayList<String>();
				for (GCubeRole gCubeRole : roles) {
					rolesNames.add(gCubeRole.getRoleName());
				}
				userWithEmailVerified.setRoles(rolesNames);

				//responseBean.setResult(userWithEmailVerified);
				responseBean.setSuccess(true);
			}catch(Exception e){
				logger.error("Unable to retrieve user's profile", e);
				responseBean.setMessage(e.getMessage());
				status = Status.INTERNAL_SERVER_ERROR;
			}
		}
		logger.debug("returning: "+userWithEmailVerified.toString());
		return Response.status(status).entity(userWithEmailVerified).build();
	}


	private static final Function<GCubeUser, UserProfileExtended> GCUBE_TO_EXTENDED_PROFILE_MAP
	= new Function<GCubeUser, UserProfileExtended>() {

		@Override
		public UserProfileExtended apply(GCubeUser t) {

			if(t == null)
				return null;

			UserProfileExtended profile = new UserProfileExtended(t.getUsername(), null, t.getUserAvatarURL(), t.getFullname());
			profile.setEmail(t.getEmail());
			profile.setFirstName(t.getFirstName());
			profile.setJobTitle(t.getJobTitle());
			profile.setLastName(t.getLastName());
			profile.setLocationIndustry(t.getLocation_industry());
			profile.setMale(t.isMale());
			profile.setRegistrationDate(t.getRegistrationDate());
			profile.setMiddleName(t.getMiddleName());
			profile.setUserId(t.getUserId());
			profile.setEmailAddresses(t.getEmailAddresses().stream().map((mail) -> {return mail.getEmail();}).collect(Collectors.<String> toList()));
			return profile;
		}
	};

	private static final Function<GCubeUser, UserProfileExtendedWithVerifiedEmail> GCUBE_TO_EXTENDED_PROFILE_MAP_WITH_VERIFIED_EMAIL
	= new Function<GCubeUser, UserProfileExtendedWithVerifiedEmail>() {

		@Override
		public UserProfileExtendedWithVerifiedEmail apply(GCubeUser t) {

			if(t == null)
				return null;

			UserProfileExtendedWithVerifiedEmail profile = new UserProfileExtendedWithVerifiedEmail(t.getUsername(), null, t.getUserAvatarURL(), t.getFullname());
			profile.setEmail(t.getEmail());
			profile.setFirstName(t.getFirstName());
			profile.setJobTitle(t.getJobTitle());
			profile.setLastName(t.getLastName());
			profile.setLocationIndustry(t.getLocation_industry());
			profile.setMale(t.isMale());
			profile.setMiddleName(t.getMiddleName());
			profile.setVerifiedEmail(true);
			return profile;
		}
	};


	@GET
	@Path("get-all-usernames")
	@Produces(MediaType.APPLICATION_JSON)
	public Response getAllUserNames(){

		ResponseBean<List<String>> responseBean = new ResponseBean<List<String>>();
		Status status = Status.OK;

		List<String> usernames = new ArrayList<String>();
		try{

			GroupManager groupManager = GroupManagerWSBuilder.getInstance().getGroupManager();
			UserManager userManager = UserManagerWSBuilder.getInstance().getUserManager();
			long groupId = groupManager.getGroupIdFromInfrastructureScope(ScopeProvider.instance.get());

			// first retrieve ids
			List<Long> userIds = userManager.getUserIdsByGroup(groupId);

			// check info in cache when available
			UsersCache cache = UsersCache.getSingleton();

			for (Long userId : userIds) {
				if(cache.getUser(userId) == null){
					GCubeUser user = userManager.getUserById(userId);
					if(user != null){
						usernames.add(user.getUsername());
						cache.pushEntry(userId, user);
					}
				}else
					usernames.add(cache.getUser(userId).getUsername());
			}

			responseBean.setResult(usernames);
			responseBean.setSuccess(true);
		}catch(Exception e){
			logger.error("Unable to retrieve user's usernames", e);
			responseBean.setMessage(e.getMessage()); 
			status = Status.INTERNAL_SERVER_ERROR;
		}

		return Response.status(status).entity(responseBean).build();
	}

	@GET
	@Path("get-all-fullnames-and-usernames")
	@Produces(MediaType.APPLICATION_JSON)
	public Response getFullnamesAndUsernames(){

		ResponseBean<Map<String, String>> responseBean = new ResponseBean<Map<String, String>>();
		Status status = Status.OK;

		Map<String, String> usernamesAndFullnames = new HashMap<String, String>();
		try{

			GroupManager groupManager = GroupManagerWSBuilder.getInstance().getGroupManager();
			UserManager userManager = UserManagerWSBuilder.getInstance().getUserManager();
			long groupId = groupManager.getGroupIdFromInfrastructureScope(ScopeProvider.instance.get());

			// first retrieve ids
			List<Long> userIds = userManager.getUserIdsByGroup(groupId);

			// check info in cache when available
			UsersCache cache = UsersCache.getSingleton();

			for (Long userId : userIds) {
				if(cache.getUser(userId) == null){
					GCubeUser user = userManager.getUserById(userId);
					if(user != null){
						usernamesAndFullnames.put(user.getUsername(), user.getFullname());
						cache.pushEntry(userId, user);
					}
				}else
					usernamesAndFullnames.put(cache.getUser(userId).getUsername(), cache.getUser(userId).getFullname());
			}

			responseBean.setResult(usernamesAndFullnames);
			responseBean.setSuccess(true);
		}catch(Exception e){
			logger.error("Unable to retrieve user's usernames", e);
			responseBean.setMessage(e.getMessage()); 
			status = Status.INTERNAL_SERVER_ERROR;
		}

		return Response.status(status).entity(responseBean).build();
	}

	@GET
	@Path("get-usernames-by-global-role")
	@Produces(MediaType.APPLICATION_JSON)
	public Response getUsernamesByGlobalRole(
			@QueryParam("role-name") String roleName){

		ResponseBean<List<String>> responseBean = new ResponseBean<List<String>>();
		Status status = Status.OK;

		// this method can only be called from IS scope (except for roles in GLOBAL_ROLES_ALLOWED_BY_LOCAL)
		ScopeBean scopeInfo = new ScopeBean(ScopeProvider.instance.get());

		if(!scopeInfo.is(Type.INFRASTRUCTURE)){
			status = Status.BAD_REQUEST;
			responseBean.setMessage("This method can only be called with an infrastructure token");
		}else{

			List<String> usernames = new ArrayList<String>();
			try{
				RoleManager roleManager = RoleManagerWSBuilder.getInstance().getRoleManager();
				long globalRoleId = roleManager.getRoleIdByName(roleName);
				if(globalRoleId > 0){
					UserManager userManager = UserManagerWSBuilder.getInstance().getUserManager();
					List<GCubeUser> users = userManager.listUsersByGlobalRole(globalRoleId);
					if(users != null){
						for (GCubeUser gCubeUser : users) {
							usernames.add(gCubeUser.getUsername());
						}
					}
					responseBean.setResult(usernames);
					responseBean.setSuccess(true);
				}else{
					responseBean.setMessage("No global role exists whit such a name");
					status = Status.BAD_REQUEST;
				}
			}catch(Exception e){
				logger.error("Unable to retrieve user's usernames", e);
				responseBean.setMessage(e.getMessage()); 
				status = Status.INTERNAL_SERVER_ERROR;
			}
		}
		return Response.status(status).entity(responseBean).build();
	}

	@GET
	@Path("get-usernames-by-role")
	@Produces(MediaType.APPLICATION_JSON)
	public Response getUsernamesByRole(
			@QueryParam("role-name") String roleName){

		ResponseBean<List<String>> responseBean = new ResponseBean<List<String>>();
		Status status = Status.OK;
		String context = ScopeProvider.instance.get();
		List<String> usernames = new ArrayList<String>();
		try{
			GroupManager groupManager = GroupManagerWSBuilder.getInstance().getGroupManager();
			RoleManager roleManager = RoleManagerWSBuilder.getInstance().getRoleManager();
			long roleId = roleManager.getRoleIdByName(roleName);
			if(roleId > 0){
				UserManager userManager = UserManagerWSBuilder.getInstance().getUserManager();
				List<GCubeUser> users = null;
				long groupId = groupManager.getGroupIdFromInfrastructureScope(context);
				// first check if for any reason this is a global role, then (if result is null or exception arises) check for site role
				// Global role's users are retrieved much faster
				try{
					if(GLOBAL_ROLES_ALLOWED_BY_LOCAL_CALL_METHOD.contains(roleName)){	
						// TODO inconsistent value can be returned
						users = userManager.listUsersByGlobalRole(roleId);
					}
				}catch(Exception globalExp){
					logger.warn("Failed while checking for global role... trying with local one", globalExp);
				}

				if(users == null || users.isEmpty()){
					logger.debug("User list is still null/empty, checking for local information");
					users = userManager.listUsersByGroupAndRole(groupId, roleId);
				}

				if(users != null){
					for (GCubeUser gCubeUser : users) {
						usernames.add(gCubeUser.getUsername());
					}
				}
				responseBean.setResult(usernames);
				responseBean.setSuccess(true);
			}else{
				responseBean.setMessage("No role exists whit such a name");
				status = Status.BAD_REQUEST;
			}
		}catch(Exception e){
			logger.error("Unable to retrieve user's usernames", e);
			responseBean.setMessage(e.getMessage()); 
			status = Status.INTERNAL_SERVER_ERROR;
		}

		return Response.status(status).entity(responseBean).build();
	}

	@GET
	@Path("user-exists")
	@Produces(MediaType.APPLICATION_JSON)
	public Response existUser(@QueryParam("username") String username){

		ResponseBean<Boolean> responseBean = new ResponseBean<Boolean>();
		String messageOnError = "This method can be invoked only by using an application token bound to the root context";
		Status status = Status.BAD_REQUEST;
		responseBean.setMessage(messageOnError);
		responseBean.setSuccess(false);
		Caller caller = AuthorizationProvider.instance.get();

		if(!TokensUtils.isApplicationToken(caller))
			return Response.status(status).entity(responseBean).build();

		ScopeBean scopeInfo = new ScopeBean(ScopeProvider.instance.get());

		if(!scopeInfo.is(Type.INFRASTRUCTURE))
			return Response.status(status).entity(responseBean).build();

		try{

			UserManager userManager = UserManagerWSBuilder.getInstance().getUserManager();
			GCubeUser user = userManager.getUserByUsername(username);
			responseBean.setSuccess(true);
			responseBean.setMessage(null);
			responseBean.setResult(user != null);
			status = Status.OK;

		}catch(Exception e){
			logger.error("Unable to retrieve such information", e);
			responseBean.setMessage(e.getMessage()); 
			status = Status.INTERNAL_SERVER_ERROR;
		}

		return Response.status(status).entity(responseBean).build();
	}

	
}
