/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.astyanax.recipes.queue;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Queues;
import com.google.common.collect.Sets;
import com.netflix.astyanax.ColumnListMutation;
import com.netflix.astyanax.Execution;
import com.netflix.astyanax.Keyspace;
import com.netflix.astyanax.MutationBatch;
import com.netflix.astyanax.connectionpool.exceptions.ConnectionException;
import com.netflix.astyanax.connectionpool.exceptions.NotFoundException;
import com.netflix.astyanax.model.Column;
import com.netflix.astyanax.model.ColumnFamily;
import com.netflix.astyanax.model.ColumnList;
import com.netflix.astyanax.model.ConsistencyLevel;
import com.netflix.astyanax.model.Equality;
import com.netflix.astyanax.model.Row;
import com.netflix.astyanax.model.Rows;
import com.netflix.astyanax.recipes.locks.BusyLockException;
import com.netflix.astyanax.recipes.queue.CountingQueueStats;
import com.netflix.astyanax.recipes.queue.KeyExistsException;
import com.netflix.astyanax.recipes.queue.Message;
import com.netflix.astyanax.recipes.queue.MessageConsumer;
import com.netflix.astyanax.recipes.queue.MessageContext;
import com.netflix.astyanax.recipes.queue.MessageHistory;
import com.netflix.astyanax.recipes.queue.MessageMetadataEntry;
import com.netflix.astyanax.recipes.queue.MessageMetadataEntryType;
import com.netflix.astyanax.recipes.queue.MessageProducer;
import com.netflix.astyanax.recipes.queue.MessageQueue;
import com.netflix.astyanax.recipes.queue.MessageQueueEntry;
import com.netflix.astyanax.recipes.queue.MessageQueueEntryState;
import com.netflix.astyanax.recipes.queue.MessageQueueEntryType;
import com.netflix.astyanax.recipes.queue.MessageQueueException;
import com.netflix.astyanax.recipes.queue.MessageQueueHooks;
import com.netflix.astyanax.recipes.queue.MessageQueueSettings;
import com.netflix.astyanax.recipes.queue.MessageQueueShard;
import com.netflix.astyanax.recipes.queue.MessageQueueStats;
import com.netflix.astyanax.recipes.queue.MessageStatus;
import com.netflix.astyanax.recipes.queue.SendMessageResponse;
import com.netflix.astyanax.recipes.queue.triggers.Trigger;
import com.netflix.astyanax.retry.RetryPolicy;
import com.netflix.astyanax.retry.RunOnce;
import com.netflix.astyanax.serializers.AnnotatedCompositeSerializer;
import com.netflix.astyanax.serializers.StringSerializer;
import com.netflix.astyanax.serializers.TimeUUIDSerializer;
import com.netflix.astyanax.util.RangeBuilder;
import com.netflix.astyanax.util.TimeUUIDUtils;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nullable;
import org.apache.commons.lang.StringUtils;
import org.codehaus.jackson.JsonGenerationException;
import org.codehaus.jackson.JsonParseException;
import org.codehaus.jackson.map.JsonMappingException;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.annotate.JsonSerialize;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ShardedDistributedMessageQueue
implements MessageQueue {
    private static final Logger LOG = LoggerFactory.getLogger(ShardedDistributedMessageQueue.class);
    public static final String DEFAULT_COLUMN_FAMILY_NAME = "Queues";
    public static final String DEFAULT_QUEUE_NAME = "Queue";
    public static final ConsistencyLevel DEFAULT_CONSISTENCY_LEVEL = ConsistencyLevel.CL_LOCAL_QUORUM;
    public static final RetryPolicy DEFAULT_RETRY_POLICY = RunOnce.get();
    public static final long DEFAULT_LOCK_TIMEOUT = TimeUnit.MICROSECONDS.convert(30L, TimeUnit.SECONDS);
    public static final Integer DEFAULT_LOCK_TTL = (int)TimeUnit.SECONDS.convert(2L, TimeUnit.MINUTES);
    public static final long DEFAULT_POLL_WAIT = TimeUnit.MILLISECONDS.convert(100L, TimeUnit.MILLISECONDS);
    public static final Boolean DEFAULT_POISON_QUEUE_ENABLED = false;
    public static final String DEFAULT_QUEUE_SUFFIX = "_queue";
    public static final String DEFAULT_METADATA_SUFFIX = "_metadata";
    public static final String DEFAULT_HISTORY_SUFFIX = "_history";
    public static final long SCHEMA_CHANGE_DELAY = 3000L;
    private static final AnnotatedCompositeSerializer<MessageQueueEntry> entrySerializer = new AnnotatedCompositeSerializer<MessageQueueEntry>(MessageQueueEntry.class);
    private static final AnnotatedCompositeSerializer<MessageMetadataEntry> metadataSerializer = new AnnotatedCompositeSerializer<MessageMetadataEntry>(MessageMetadataEntry.class);
    private static final ObjectMapper mapper = new ObjectMapper();
    private ColumnFamily<String, MessageQueueEntry> queueColumnFamily;
    private ColumnFamily<String, MessageMetadataEntry> keyIndexColumnFamily;
    private ColumnFamily<String, UUID> historyColumnFamily;
    private Keyspace keyspace;
    private ConsistencyLevel consistencyLevel;
    private long lockTimeout;
    private int lockTtl;
    private long pollInterval;
    private MessageQueueStats stats;
    private String queueName;
    private AtomicLong counter;
    private Collection<MessageQueueHooks> hooks;
    private MessageQueueSettings settings;
    private List<MessageQueueShard> partitions;
    private Boolean bPoisonQueueEnabled;
    private Function<String, Message> invalidMessageHandler;

    public ShardedDistributedMessageQueue() {
        mapper.getSerializationConfig().setSerializationInclusion(JsonSerialize.Inclusion.NON_NULL);
        mapper.enableDefaultTyping();
        this.consistencyLevel = DEFAULT_CONSISTENCY_LEVEL;
        this.lockTimeout = DEFAULT_LOCK_TIMEOUT;
        this.lockTtl = DEFAULT_LOCK_TTL;
        this.pollInterval = DEFAULT_POLL_WAIT;
        this.stats = new CountingQueueStats();
        this.queueName = DEFAULT_QUEUE_NAME;
        this.counter = new AtomicLong(new Random().nextInt(1000));
        this.hooks = Lists.newArrayList();
        this.settings = new MessageQueueSettings();
        this.bPoisonQueueEnabled = DEFAULT_POISON_QUEUE_ENABLED;
        this.invalidMessageHandler = new Function<String, Message>(){

            public Message apply(@Nullable String input) {
                LOG.warn("Invalid message: " + input);
                return null;
            }
        };
    }

    private void initialize() {
        Preconditions.checkArgument((TimeUnit.SECONDS.convert(this.lockTimeout, TimeUnit.MICROSECONDS) < (long)this.lockTtl ? 1 : 0) != 0, (Object)("Timeout " + this.lockTtl + " seconds must be less than TTL " + TimeUnit.SECONDS.convert(this.lockTtl, TimeUnit.MICROSECONDS) + " seconds"));
        Preconditions.checkNotNull((Object)this.keyspace, (Object)"Must specify keyspace");
        try {
            Column column = (Column)this.keyspace.prepareQuery(this.queueColumnFamily).getRow(this.queueName).getColumn(MessageQueueEntry.newMetadataEntry()).execute().getResult();
            ByteArrayInputStream bais = new ByteArrayInputStream(column.getByteArrayValue());
            this.settings = (MessageQueueSettings)mapper.readValue((InputStream)bais, MessageQueueSettings.class);
        }
        catch (Exception e) {
            // empty catch block
        }
        this.partitions = Lists.newArrayList();
        for (int i = 0; i < this.settings.getPartitionCount(); ++i) {
            for (int j = 0; j < this.settings.getShardCount(); ++j) {
                this.partitions.add(new MessageQueueShard(this.queueName + ":" + i + ":" + j, i, j));
            }
        }
    }

    protected MessageQueueEntry getBusyEntry(Message message) {
        return MessageQueueEntry.newMessageEntry(message.getPriority(), message.getToken(), MessageQueueEntryState.Busy);
    }

    protected String getShardKey(MessageQueueEntry message) {
        return this.getShardKey(TimeUUIDUtils.getMicrosTimeFromUUID(message.getTimestamp()));
    }

    private String getShardKey(long messageTime) {
        long timePartition = this.settings.getPartitionDuration() != null ? messageTime / this.settings.getPartitionDuration() % (long)this.settings.getPartitionCount() : 0L;
        long shard = messageTime % (long)this.settings.getShardCount();
        return this.queueName + ":" + timePartition + ":" + shard;
    }

    private String getCompositeKey(String name, String key) {
        return name + "$" + key;
    }

    private static String[] splitCompositeKey(String key) throws MessageQueueException {
        String[] parts = StringUtils.split((String)key, (String)"$");
        if (parts.length != 2) {
            throw new MessageQueueException("Invalid key '" + key + "'.  Expected format <queue|shard>$<name>. ");
        }
        return parts;
    }

    private <T> String serializeToString(T trigger) throws JsonGenerationException, JsonMappingException, IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        mapper.writeValue((OutputStream)baos, trigger);
        baos.flush();
        return baos.toString();
    }

    private <T> T deserializeString(String data, Class<T> clazz) throws JsonParseException, JsonMappingException, IOException {
        return (T)mapper.readValue((InputStream)new ByteArrayInputStream(data.getBytes()), clazz);
    }

    private <T> T deserializeString(String data, String className) throws JsonParseException, JsonMappingException, IOException, ClassNotFoundException {
        return (T)mapper.readValue((InputStream)new ByteArrayInputStream(data.getBytes()), Class.forName(className));
    }

    @Override
    public String getName() {
        return this.queueName;
    }

    @Override
    public long getMessageCount() throws MessageQueueException {
        Map<String, Integer> counts = this.getShardCounts();
        long count = 0L;
        for (Integer value : counts.values()) {
            count += (long)value.intValue();
        }
        return count;
    }

    @Override
    public Map<String, Integer> getShardCounts() throws MessageQueueException {
        try {
            ArrayList keys = Lists.newArrayList();
            for (int i = 0; i < this.settings.getPartitionCount(); ++i) {
                for (int j = 0; j < this.settings.getShardCount(); ++j) {
                    keys.add(this.queueName + ":" + i + ":" + j);
                }
            }
            TreeMap result = Maps.newTreeMap();
            result.putAll((Map)this.keyspace.prepareQuery(this.queueColumnFamily).getKeySlice(keys).getColumnCounts().execute().getResult());
            return result;
        }
        catch (ConnectionException e) {
            throw new MessageQueueException("Failed to get counts", e);
        }
    }

    public String getCurrentPartition() {
        long timePartition = this.settings.getPartitionDuration() != null ? TimeUnit.MICROSECONDS.convert(System.currentTimeMillis(), TimeUnit.MILLISECONDS) / this.settings.getPartitionDuration() : 0L;
        return this.queueName + ":" + (timePartition + (long)this.settings.getPartitionCount()) % (long)this.settings.getPartitionCount();
    }

    @Override
    public void clearMessages() throws MessageQueueException {
        LOG.info("Clearing messages from '" + this.queueName + "'");
        MutationBatch mb = this.keyspace.prepareMutationBatch().setConsistencyLevel(this.consistencyLevel);
        for (MessageQueueShard partition : this.partitions) {
            mb.withRow(this.queueColumnFamily, partition.getName()).delete();
        }
        try {
            mb.execute();
        }
        catch (ConnectionException e) {
            throw new MessageQueueException("Failed to clear messages from queue " + this.queueName, e);
        }
    }

    @Override
    public void deleteQueue() throws MessageQueueException {
        LOG.info("Deleting queue '" + this.queueName + "'");
        MutationBatch mb = this.keyspace.prepareMutationBatch().setConsistencyLevel(this.consistencyLevel);
        for (MessageQueueShard partition : this.partitions) {
            mb.withRow(this.queueColumnFamily, partition.getName()).delete();
        }
        mb.withRow(this.queueColumnFamily, this.queueName);
        try {
            mb.execute();
        }
        catch (ConnectionException e) {
            throw new MessageQueueException("Failed to clear messages from queue " + this.queueName, e);
        }
    }

    @Override
    public Message peekMessage(String messageId) throws MessageQueueException {
        String[] parts = ShardedDistributedMessageQueue.splitCompositeKey(messageId);
        String shardKey = parts[0];
        MessageQueueEntry entry = new MessageQueueEntry(parts[1]);
        try {
            Column column = (Column)this.keyspace.prepareQuery(this.queueColumnFamily).setConsistencyLevel(this.consistencyLevel).getKey(shardKey).getColumn(entry).execute().getResult();
            try {
                ByteArrayInputStream bais = new ByteArrayInputStream(column.getByteArrayValue());
                return (Message)mapper.readValue((InputStream)bais, Message.class);
            }
            catch (Exception e) {
                try {
                    return (Message)this.invalidMessageHandler.apply((Object)column.getStringValue());
                }
                catch (Exception e2) {
                    throw new MessageQueueException("Error parsing message " + messageId);
                }
            }
        }
        catch (NotFoundException e) {
            return null;
        }
        catch (ConnectionException e) {
            throw new MessageQueueException("Error getting message " + messageId, e);
        }
    }

    @Override
    public Message peekMessageByKey(String key) throws MessageQueueException {
        String groupRowKey = this.getCompositeKey(this.queueName, key);
        try {
            ColumnList columns = (ColumnList)this.keyspace.prepareQuery(this.keyIndexColumnFamily).getRow(groupRowKey).withColumnRange(metadataSerializer.buildRange().greaterThanEquals((byte)MessageMetadataEntryType.MessageId.ordinal()).lessThanEquals((byte)MessageMetadataEntryType.MessageId.ordinal()).build()).execute().getResult();
            Iterator i$ = columns.iterator();
            if (i$.hasNext()) {
                Column entry = (Column)i$.next();
                return this.peekMessage(((MessageMetadataEntry)entry.getName()).getName());
            }
            return null;
        }
        catch (NotFoundException e) {
            return null;
        }
        catch (ConnectionException e) {
            throw new MessageQueueException("Error fetching row " + groupRowKey, e);
        }
    }

    @Override
    public boolean deleteMessageByKey(String key) throws MessageQueueException {
        MutationBatch mb = this.keyspace.prepareMutationBatch();
        String groupRowKey = this.getCompositeKey(this.queueName, key);
        try {
            ColumnList columns = (ColumnList)this.keyspace.prepareQuery(this.keyIndexColumnFamily).getRow(groupRowKey).withColumnRange(metadataSerializer.buildRange().greaterThanEquals((byte)MessageMetadataEntryType.MessageId.ordinal()).lessThanEquals((byte)MessageMetadataEntryType.MessageId.ordinal()).build()).execute().getResult();
            for (Column entry : columns) {
                String[] parts = ShardedDistributedMessageQueue.splitCompositeKey(((MessageMetadataEntry)entry.getName()).getName());
                String shardKey = parts[0];
                MessageQueueEntry queueEntry = new MessageQueueEntry(parts[1]);
                mb.withRow(this.queueColumnFamily, shardKey).deleteColumn(queueEntry);
            }
            mb.withRow(this.keyIndexColumnFamily, groupRowKey).delete();
        }
        catch (NotFoundException e) {
            return false;
        }
        catch (ConnectionException e) {
            throw new MessageQueueException("Error fetching row " + groupRowKey, e);
        }
        try {
            mb.execute();
        }
        catch (ConnectionException e) {
            throw new MessageQueueException("Error deleting queue item " + groupRowKey, e);
        }
        return true;
    }

    @Override
    public void deleteMessage(String messageId) throws MessageQueueException {
        String[] parts = ShardedDistributedMessageQueue.splitCompositeKey(messageId);
        String shardKey = parts[0];
        MessageQueueEntry entry = new MessageQueueEntry(parts[1]);
        try {
            this.keyspace.prepareColumnMutation(this.queueColumnFamily, shardKey, entry).setConsistencyLevel(this.consistencyLevel).deleteColumn().execute();
        }
        catch (ConnectionException e) {
            throw new MessageQueueException("Error deleting message " + messageId, e);
        }
    }

    @Override
    public void deleteMessages(Collection<String> messageIds) throws MessageQueueException {
        MutationBatch mb = this.keyspace.prepareMutationBatch().setConsistencyLevel(this.consistencyLevel);
        for (String messageId : messageIds) {
            String[] parts = ShardedDistributedMessageQueue.splitCompositeKey(messageId);
            String shardKey = parts[0];
            MessageQueueEntry entry = new MessageQueueEntry(parts[1]);
            mb.withRow(this.queueColumnFamily, shardKey).deleteColumn(entry);
        }
        try {
            mb.execute();
        }
        catch (ConnectionException e) {
            throw new MessageQueueException("Error deleting messages " + messageIds, e);
        }
    }

    @Override
    public void createStorage() throws MessageQueueException {
        block14: {
            block13: {
                block12: {
                    try {
                        this.keyspace.createColumnFamily(this.queueColumnFamily, (Map<String, Object>)ImmutableMap.builder().put((Object)"key_validation_class", (Object)"UTF8Type").put((Object)"comparator_type", (Object)"CompositeType(BytesType, BytesType(reversed=true), TimeUUIDType, BytesType)").put((Object)"read_repair_chance", (Object)1.0).put((Object)"gc_grace_period", (Object)0).put((Object)"compaction_strategy", (Object)"LeveledCompactionStrategy").put((Object)"min_compaction_threshold", (Object)2).put((Object)"max_compaction_threshold", (Object)4).build());
                        try {
                            Thread.sleep(3000L);
                        }
                        catch (InterruptedException e) {}
                    }
                    catch (ConnectionException e) {
                        if (e.getMessage().contains("already exist")) break block12;
                        throw new MessageQueueException("Failed to create column family for " + this.queueColumnFamily.getName(), e);
                    }
                }
                try {
                    this.keyspace.createColumnFamily(this.keyIndexColumnFamily, (Map<String, Object>)ImmutableMap.builder().put((Object)"key_validation_class", (Object)"UTF8Type").put((Object)"comparator_type", (Object)"CompositeType(BytesType, UTF8Type)").put((Object)"read_repair_chance", (Object)1.0).put((Object)"gc_grace_period", (Object)0).put((Object)"compaction_strategy", (Object)"LeveledCompactionStrategy").put((Object)"min_compaction_threshold", (Object)2).put((Object)"max_compaction_threshold", (Object)4).build());
                    try {
                        Thread.sleep(3000L);
                    }
                    catch (InterruptedException e) {}
                }
                catch (ConnectionException e) {
                    if (e.getMessage().contains("already exist")) break block13;
                    throw new MessageQueueException("Failed to create column family for " + this.keyIndexColumnFamily.getName(), e);
                }
            }
            try {
                this.keyspace.createColumnFamily(this.historyColumnFamily, (Map<String, Object>)ImmutableMap.builder().put((Object)"read_repair_chance", (Object)1.0).put((Object)"gc_grace_period", (Object)0).put((Object)"default_validation_class", (Object)"UTF8Type").put((Object)"compaction_strategy", (Object)"LeveledCompactionStrategy").put((Object)"min_compaction_threshold", (Object)2).put((Object)"max_compaction_threshold", (Object)4).build());
                try {
                    Thread.sleep(3000L);
                }
                catch (InterruptedException e) {}
            }
            catch (ConnectionException e) {
                if (e.getMessage().contains("already exist")) break block14;
                throw new MessageQueueException("Failed to create column family for " + this.historyColumnFamily.getName(), e);
            }
        }
    }

    @Override
    public void dropStorage() throws MessageQueueException {
        block9: {
            block8: {
                try {
                    this.keyspace.dropColumnFamily(this.queueColumnFamily);
                    try {
                        Thread.sleep(3000L);
                    }
                    catch (InterruptedException e) {}
                }
                catch (ConnectionException e) {
                    if (e.getMessage().contains("already exist")) break block8;
                    throw new MessageQueueException("Failed to create column family for " + this.queueColumnFamily.getName(), e);
                }
            }
            try {
                this.keyspace.dropColumnFamily(this.keyIndexColumnFamily);
                try {
                    Thread.sleep(3000L);
                }
                catch (InterruptedException e) {}
            }
            catch (ConnectionException e) {
                if (e.getMessage().contains("already exist")) break block9;
                throw new MessageQueueException("Failed to create column family for " + this.queueColumnFamily.getName(), e);
            }
        }
    }

    @Override
    public void createQueue() throws MessageQueueException {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            mapper.writeValue((OutputStream)baos, (Object)this.settings);
            baos.flush();
            this.keyspace.prepareColumnMutation(this.queueColumnFamily, this.queueName, MessageQueueEntry.newMetadataEntry()).putValue(baos.toByteArray(), null).execute();
        }
        catch (ConnectionException e) {
            throw new MessageQueueException("Failed to create column family for " + this.queueColumnFamily.getName(), e);
        }
        catch (Exception e) {
            throw new MessageQueueException("Error serializing queue settings " + this.queueColumnFamily.getName(), e);
        }
    }

    @Override
    public MessageConsumer createConsumer() {
        return new MessageConsumer(){
            private LinkedBlockingQueue<MessageQueueShard> workQueue = Queues.newLinkedBlockingQueue();
            private LinkedBlockingQueue<MessageQueueShard> idleQueue = Queues.newLinkedBlockingQueue();
            private long currentPartition = -1L;
            {
                ArrayList randomized = Lists.newArrayList((Iterable)ShardedDistributedMessageQueue.this.partitions);
                Collections.shuffle(randomized);
                this.idleQueue.addAll(randomized);
            }

            @Override
            public Collection<MessageContext> readMessages(int itemsToPop) throws MessageQueueException, BusyLockException, InterruptedException {
                return this.readMessages(itemsToPop, 0L, null);
            }

            @Override
            public Collection<MessageContext> readMessages(int itemsToPop, long timeout, TimeUnit units) throws MessageQueueException, BusyLockException, InterruptedException {
                long timeoutTime;
                long l = timeoutTime = timeout == 0L ? 0L : System.currentTimeMillis() + TimeUnit.MILLISECONDS.convert(timeout, units);
                if (ShardedDistributedMessageQueue.this.settings.getPartitionDuration() != null) {
                    long partitionIndex = TimeUnit.MICROSECONDS.convert(System.currentTimeMillis(), TimeUnit.MILLISECONDS) / ShardedDistributedMessageQueue.this.settings.getPartitionDuration() % (long)ShardedDistributedMessageQueue.this.settings.getPartitionCount();
                    if (partitionIndex != this.currentPartition) {
                        this.currentPartition = partitionIndex;
                        ArrayList partitions = Lists.newArrayList();
                        this.idleQueue.drainTo(partitions);
                        for (MessageQueueShard partition : partitions) {
                            if ((long)partition.getPartition() == this.currentPartition) {
                                this.workQueue.add(partition);
                                continue;
                            }
                            this.idleQueue.add(partition);
                        }
                    }
                } else {
                    this.currentPartition = 0L;
                }
                Collection<MessageContext> messages = null;
                while (true) {
                    if (!this.workQueue.isEmpty()) {
                        MessageQueueShard partition = (MessageQueueShard)this.workQueue.remove();
                        messages = this.readAndReturnShard(partition, itemsToPop);
                        if (messages != null && !messages.isEmpty()) {
                            return messages;
                        }
                        if ((long)partition.getPartition() != this.currentPartition) continue;
                    }
                    if (!this.idleQueue.isEmpty() && (messages = this.readAndReturnShard((MessageQueueShard)this.idleQueue.remove(), itemsToPop)) != null && !messages.isEmpty()) {
                        return messages;
                    }
                    if (timeoutTime != 0L && System.currentTimeMillis() > timeoutTime) {
                        return Lists.newLinkedList();
                    }
                    Thread.sleep(ShardedDistributedMessageQueue.this.pollInterval);
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private Collection<MessageContext> readAndReturnShard(MessageQueueShard partition, int itemsToPop) throws MessageQueueException, BusyLockException, InterruptedException {
                Collection<MessageContext> messages = null;
                if (partition != null) {
                    try {
                        if (partition.getLastCount() == 0 && !this.hasMessages(partition.getName())) {
                            Collection<MessageContext> collection = null;
                            return collection;
                        }
                        messages = this.internalReadMessages(partition.getName(), itemsToPop);
                        if (!messages.isEmpty()) {
                            Collection<MessageContext> collection = messages;
                            return collection;
                        }
                        ShardedDistributedMessageQueue.this.stats.incEmptyPartitionCount();
                    }
                    catch (BusyLockException e) {
                    }
                    finally {
                        if ((long)partition.getPartition() == this.currentPartition || messages != null && !messages.isEmpty()) {
                            this.workQueue.add(partition);
                        } else {
                            this.idleQueue.add(partition);
                        }
                    }
                }
                return messages;
            }

            @Override
            public Collection<Message> peekMessages(int itemsToPop) throws MessageQueueException {
                long partitionIndex = ShardedDistributedMessageQueue.this.settings.getPartitionDuration() != null ? TimeUnit.MICROSECONDS.convert(System.currentTimeMillis(), TimeUnit.MILLISECONDS) / ShardedDistributedMessageQueue.this.settings.getPartitionDuration() % (long)ShardedDistributedMessageQueue.this.settings.getPartitionCount() : 0L;
                ArrayList messages = Lists.newArrayList();
                for (int i = 0; i < ShardedDistributedMessageQueue.this.settings.getPartitionCount(); ++i) {
                    for (int j = 0; j < ShardedDistributedMessageQueue.this.settings.getShardCount(); ++j) {
                        int index = (int)(((long)i + partitionIndex) % (long)ShardedDistributedMessageQueue.this.settings.getPartitionCount() * (long)ShardedDistributedMessageQueue.this.settings.getShardCount() + (long)j);
                        messages.addAll(this.peekMessages(((MessageQueueShard)ShardedDistributedMessageQueue.this.partitions.get(index)).getName(), itemsToPop - messages.size()));
                        if (messages.size() != itemsToPop) continue;
                        return messages;
                    }
                }
                return messages;
            }

            private Collection<Message> peekMessages(String shardName, int itemsToPop) throws MessageQueueException {
                long curTimeMicros = TimeUnit.MICROSECONDS.convert(System.currentTimeMillis(), TimeUnit.MILLISECONDS);
                try {
                    ColumnList result = (ColumnList)ShardedDistributedMessageQueue.this.keyspace.prepareQuery(ShardedDistributedMessageQueue.this.queueColumnFamily).setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel).getKey(shardName).withColumnRange(new RangeBuilder().setLimit(itemsToPop).setEnd(entrySerializer.makeEndpoint((byte)MessageQueueEntryType.Message.ordinal(), Equality.EQUAL).append((byte)0, Equality.EQUAL).append(TimeUUIDUtils.getMicrosTimeUUID(curTimeMicros), Equality.LESS_THAN_EQUALS).toBytes()).build()).execute().getResult();
                    ArrayList messages = Lists.newArrayListWithCapacity((int)result.size());
                    for (Column column : result) {
                        Message message = this.extractMessageFromColumn(column);
                        if (message == null) continue;
                        messages.add(message);
                    }
                    return messages;
                }
                catch (ConnectionException e) {
                    throw new MessageQueueException("Error peeking for messages from shard " + shardName, e);
                }
            }

            private boolean hasMessages(String shardName) throws MessageQueueException {
                UUID currentTime = TimeUUIDUtils.getUniqueTimeUUIDinMicros();
                try {
                    ColumnList result = (ColumnList)ShardedDistributedMessageQueue.this.keyspace.prepareQuery(ShardedDistributedMessageQueue.this.queueColumnFamily).setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel).getKey(shardName).withColumnRange(new RangeBuilder().setLimit(1).setStart(entrySerializer.makeEndpoint((byte)MessageQueueEntryType.Message.ordinal(), Equality.EQUAL).toBytes()).setEnd(entrySerializer.makeEndpoint((byte)MessageQueueEntryType.Message.ordinal(), Equality.EQUAL).append((byte)0, Equality.EQUAL).append(currentTime, Equality.LESS_THAN_EQUALS).toBytes()).build()).execute().getResult();
                    return !result.isEmpty();
                }
                catch (ConnectionException e) {
                    throw new MessageQueueException("Error checking shard for messages. " + shardName, e);
                }
            }

            private Collection<MessageContext> internalReadMessages(String shardName, int itemsToPop) throws MessageQueueException, BusyLockException {
                ColumnList result;
                long curTimeMicros;
                ArrayList entries = Lists.newArrayList();
                Execution m = null;
                MessageQueueEntry lockColumn = null;
                ColumnListMutation<MessageQueueEntry> rowMutation = null;
                int lockColumnCount = 0;
                try {
                    lockColumn = MessageQueueEntry.newLockEntry(MessageQueueEntryState.None);
                    curTimeMicros = TimeUUIDUtils.getTimeFromUUID(lockColumn.getTimestamp());
                    m = ShardedDistributedMessageQueue.this.keyspace.prepareMutationBatch().setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel);
                    m.withRow(ShardedDistributedMessageQueue.this.queueColumnFamily, shardName).putColumn(lockColumn, curTimeMicros + ShardedDistributedMessageQueue.this.lockTimeout, (Integer)ShardedDistributedMessageQueue.this.lockTtl);
                    m.execute();
                    result = (ColumnList)ShardedDistributedMessageQueue.this.keyspace.prepareQuery(ShardedDistributedMessageQueue.this.queueColumnFamily).setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel).getKey(shardName).withColumnRange(entrySerializer.buildRange().greaterThanEquals((byte)MessageQueueEntryType.Lock.ordinal()).lessThanEquals((byte)MessageQueueEntryType.Lock.ordinal()).build()).execute().getResult();
                    m = ShardedDistributedMessageQueue.this.keyspace.prepareMutationBatch().setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel);
                    rowMutation = m.withRow(ShardedDistributedMessageQueue.this.queueColumnFamily, shardName);
                    rowMutation.deleteColumn(lockColumn);
                    int lockCount = 0;
                    boolean lockAcquired = false;
                    lockColumnCount = result.size();
                    for (Column column : result) {
                        MessageQueueEntry lock = (MessageQueueEntry)column.getName();
                        if (lock.getType() != MessageQueueEntryType.Lock) continue;
                        ++lockColumnCount;
                        if (column.getLongValue() < curTimeMicros) {
                            ShardedDistributedMessageQueue.this.stats.incExpiredLockCount();
                            rowMutation.deleteColumn(lock);
                        } else {
                            if (lock.getState() == MessageQueueEntryState.Acquired) {
                                throw new BusyLockException("Not first lock");
                            }
                            if (++lockCount == 1 && lock.getTimestamp().equals(lockColumn.getTimestamp())) {
                                lockAcquired = true;
                            }
                        }
                        if (!lockAcquired) {
                            throw new BusyLockException("Not first lock");
                        }
                        lockColumn = MessageQueueEntry.newLockEntry(lockColumn.getTimestamp(), MessageQueueEntryState.Acquired);
                        rowMutation.putColumn(lockColumn, curTimeMicros + ShardedDistributedMessageQueue.this.lockTimeout, (Integer)ShardedDistributedMessageQueue.this.lockTtl);
                    }
                }
                catch (BusyLockException e) {
                    ShardedDistributedMessageQueue.this.stats.incLockContentionCount();
                    throw e;
                }
                catch (ConnectionException e) {
                    LOG.error("Error reading shard " + shardName, (Throwable)e);
                    throw new MessageQueueException("Error", e);
                }
                finally {
                    try {
                        m.execute();
                    }
                    catch (Exception e) {
                        throw new MessageQueueException("Error committing lock", e);
                    }
                }
                curTimeMicros = TimeUUIDUtils.getMicrosTimeFromUUID(lockColumn.getTimestamp());
                m = ShardedDistributedMessageQueue.this.keyspace.prepareMutationBatch().setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel);
                rowMutation = m.withRow(ShardedDistributedMessageQueue.this.queueColumnFamily, shardName);
                rowMutation.deleteColumn(lockColumn);
                try {
                    result = (ColumnList)ShardedDistributedMessageQueue.this.keyspace.prepareQuery(ShardedDistributedMessageQueue.this.queueColumnFamily).setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel).getKey(shardName).withColumnRange(new RangeBuilder().setLimit(itemsToPop + lockColumnCount + 1).setEnd(entrySerializer.makeEndpoint((byte)MessageQueueEntryType.Message.ordinal(), Equality.EQUAL).append((byte)0, Equality.EQUAL).append(lockColumn.getTimestamp(), Equality.LESS_THAN_EQUALS).toBytes()).build()).execute().getResult();
                    for (Column column : result) {
                        if (itemsToPop == 0) break;
                        MessageQueueEntry entry = (MessageQueueEntry)column.getName();
                        switch (entry.getType()) {
                            case Lock: {
                                if (entry.getState() != MessageQueueEntryState.Acquired || entry.getTimestamp().equals(lockColumn.getTimestamp())) break;
                                throw new BusyLockException("Someone else snuck in");
                            }
                            case Message: {
                                --itemsToPop;
                                rowMutation.deleteColumn(entry);
                                Message message = this.extractMessageFromColumn(column);
                                if (message != null) {
                                    Trigger trigger;
                                    MessageContext context = new MessageContext();
                                    context.setMessage(message);
                                    if (message.hasKey()) {
                                        m.withRow(ShardedDistributedMessageQueue.this.keyIndexColumnFamily, ShardedDistributedMessageQueue.this.getCompositeKey(ShardedDistributedMessageQueue.this.queueName, message.getKey())).deleteColumn(MessageMetadataEntry.newMessageId(ShardedDistributedMessageQueue.this.getCompositeKey(shardName, entry.getMessageId())));
                                        if (message.isKeepHistory().booleanValue()) {
                                            MessageHistory history = context.getHistory();
                                            history.setToken(entry.getTimestamp());
                                            history.setStartTime(curTimeMicros);
                                            history.setTriggerTime(message.getTrigger().getTriggerTime());
                                            history.setStatus(MessageStatus.RUNNING);
                                            try {
                                                m.withRow(ShardedDistributedMessageQueue.this.historyColumnFamily, message.getKey()).putColumn(entry.getTimestamp(), ShardedDistributedMessageQueue.this.serializeToString(history), ShardedDistributedMessageQueue.this.settings.getHistoryTtl());
                                            }
                                            catch (Exception e) {
                                                LOG.warn("Error serializing history for key '" + message.getKey() + "'", (Throwable)e);
                                            }
                                        }
                                    }
                                    if (message.getTimeout() != 0) {
                                        MessageQueueEntry timeoutEntry = MessageQueueEntry.newMessageEntry((byte)0, TimeUUIDUtils.getMicrosTimeUUID(curTimeMicros + TimeUnit.MICROSECONDS.convert(message.getTimeout().intValue(), TimeUnit.SECONDS) + ShardedDistributedMessageQueue.this.counter.incrementAndGet() % 1000L), MessageQueueEntryState.Busy);
                                        message.setToken(timeoutEntry.getTimestamp());
                                        m.withRow(ShardedDistributedMessageQueue.this.queueColumnFamily, ShardedDistributedMessageQueue.this.getShardKey(timeoutEntry)).putColumn(timeoutEntry, column.getStringValue(), ShardedDistributedMessageQueue.this.settings.getRetentionTimeout());
                                        if (message.hasKey()) {
                                            m.withRow(ShardedDistributedMessageQueue.this.keyIndexColumnFamily, ShardedDistributedMessageQueue.this.getCompositeKey(ShardedDistributedMessageQueue.this.queueName, message.getKey())).putEmptyColumn(MessageMetadataEntry.newMessageId(ShardedDistributedMessageQueue.this.getCompositeKey(ShardedDistributedMessageQueue.this.getShardKey(timeoutEntry), timeoutEntry.getMessageId())), ShardedDistributedMessageQueue.this.settings.getRetentionTimeout());
                                        }
                                    } else {
                                        message.setToken(null);
                                    }
                                    if (message.hasTrigger() && (trigger = message.getTrigger().nextTrigger()) != null) {
                                        Message nextMessage = message.clone();
                                        nextMessage.setTrigger(trigger);
                                        context.setNextMessage(nextMessage);
                                        ShardedDistributedMessageQueue.this.fillMessageMutation((MutationBatch)m, nextMessage);
                                    }
                                    switch (entry.getState()) {
                                        case Waiting: {
                                            ShardedDistributedMessageQueue.this.stats.incProcessCount();
                                            break;
                                        }
                                        case Busy: {
                                            ShardedDistributedMessageQueue.this.stats.incReprocessCount();
                                            break;
                                        }
                                        default: {
                                            LOG.warn("Unknown message state: " + (Object)((Object)entry.getState()));
                                        }
                                    }
                                    entries.add(context);
                                    break;
                                }
                                ShardedDistributedMessageQueue.this.stats.incInvalidMessageCount();
                                break;
                            }
                        }
                    }
                    ArrayList arrayList = entries;
                    return arrayList;
                }
                catch (BusyLockException e) {
                    ShardedDistributedMessageQueue.this.stats.incLockContentionCount();
                    throw e;
                }
                catch (Exception e) {
                    throw new MessageQueueException("Error processing queue shard : " + shardName, e);
                }
                finally {
                    try {
                        m.execute();
                    }
                    catch (Exception e) {
                        throw new MessageQueueException("Error processing queue shard : " + shardName, e);
                    }
                }
            }

            @Override
            public void ackMessage(MessageContext context) throws MessageQueueException {
                MutationBatch mb = ShardedDistributedMessageQueue.this.keyspace.prepareMutationBatch().setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel);
                this.fillAckMutation(context, mb);
                try {
                    mb.execute();
                }
                catch (ConnectionException e) {
                    throw new MessageQueueException("Failed to ack message", e);
                }
            }

            @Override
            public void ackMessages(Collection<MessageContext> messages) throws MessageQueueException {
                MutationBatch mb = ShardedDistributedMessageQueue.this.keyspace.prepareMutationBatch().setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel);
                for (MessageContext context : messages) {
                    this.fillAckMutation(context, mb);
                }
                try {
                    mb.execute();
                }
                catch (ConnectionException e) {
                    throw new MessageQueueException("Failed to ack messages", e);
                }
            }

            private void fillAckMutation(MessageContext context, MutationBatch mb) {
                Message message = context.getMessage();
                if (message.getToken() != null) {
                    MessageQueueEntry entry = ShardedDistributedMessageQueue.this.getBusyEntry(message);
                    ShardedDistributedMessageQueue.this.stats.incFinishMessageCount();
                    mb.withRow(ShardedDistributedMessageQueue.this.queueColumnFamily, ShardedDistributedMessageQueue.this.getShardKey(entry)).deleteColumn(entry);
                    if (message.hasKey()) {
                        mb.withRow(ShardedDistributedMessageQueue.this.keyIndexColumnFamily, ShardedDistributedMessageQueue.this.getCompositeKey(ShardedDistributedMessageQueue.this.queueName, message.getKey())).deleteColumn(MessageMetadataEntry.newMessageId(entry.getMessageId()));
                        if (message.isKeepHistory().booleanValue()) {
                            MessageHistory history = context.getHistory();
                            if (history.getStatus() == MessageStatus.RUNNING) {
                                history.setStatus(MessageStatus.DONE);
                            }
                            history.setEndTime(TimeUnit.MICROSECONDS.convert(System.currentTimeMillis(), TimeUnit.MILLISECONDS));
                            try {
                                mb.withRow(ShardedDistributedMessageQueue.this.historyColumnFamily, message.getKey()).putColumn(history.getToken(), ShardedDistributedMessageQueue.this.serializeToString(context.getHistory()), ShardedDistributedMessageQueue.this.settings.getHistoryTtl());
                            }
                            catch (Exception e) {
                                LOG.warn("Error serializing message history for " + message.getKey(), (Throwable)e);
                            }
                        }
                    }
                    for (MessageQueueHooks hook : ShardedDistributedMessageQueue.this.hooks) {
                        hook.beforeAckMessage(message, mb);
                    }
                }
            }

            private Message extractMessageFromColumn(Column<MessageQueueEntry> column) {
                Message message = null;
                try {
                    ByteArrayInputStream bais = new ByteArrayInputStream(column.getByteArrayValue());
                    message = (Message)mapper.readValue((InputStream)bais, Message.class);
                }
                catch (Exception e) {
                    try {
                        message = (Message)ShardedDistributedMessageQueue.this.invalidMessageHandler.apply((Object)column.getStringValue());
                    }
                    catch (Exception e2) {
                        // empty catch block
                    }
                }
                return message;
            }

            @Override
            public void ackPoisonMessage(MessageContext context) throws MessageQueueException {
                MutationBatch mb = ShardedDistributedMessageQueue.this.keyspace.prepareMutationBatch().setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel);
                this.fillAckMutation(context, mb);
                try {
                    mb.execute();
                }
                catch (ConnectionException e) {
                    throw new MessageQueueException("Failed to ack messages", e);
                }
            }
        };
    }

    @Override
    public MessageProducer createProducer() {
        return new MessageProducer(){

            @Override
            public String sendMessage(Message message) throws MessageQueueException {
                SendMessageResponse response = this.sendMessages(Lists.newArrayList((Object[])new Message[]{message}));
                if (!response.getNotUnique().isEmpty()) {
                    throw new KeyExistsException("Key already exists ." + message.getKey());
                }
                return (String)((Map.Entry)Iterables.getFirst(response.getMessages().entrySet(), null)).getKey();
            }

            @Override
            public SendMessageResponse sendMessages(Collection<Message> messages) throws MessageQueueException {
                HashMap keys = Maps.newHashMap();
                HashSet notUniqueKeys = Sets.newHashSet();
                ArrayList notUniqueMessages = Lists.newArrayList();
                MutationBatch mb = ShardedDistributedMessageQueue.this.keyspace.prepareMutationBatch().setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel);
                MessageMetadataEntry lockColumn = MessageMetadataEntry.newUnique();
                for (Message message : messages) {
                    if (!message.hasKey()) continue;
                    String groupKey = ShardedDistributedMessageQueue.this.getCompositeKey(ShardedDistributedMessageQueue.this.queueName, message.getKey());
                    keys.put(groupKey, message);
                    mb.withRow(ShardedDistributedMessageQueue.this.keyIndexColumnFamily, groupKey).putEmptyColumn(lockColumn, ShardedDistributedMessageQueue.this.lockTtl);
                }
                if (!keys.isEmpty()) {
                    Rows result;
                    try {
                        mb.execute();
                    }
                    catch (ConnectionException e) {
                        throw new MessageQueueException("Failed to check keys for uniqueness (1): " + keys, e);
                    }
                    mb = ShardedDistributedMessageQueue.this.keyspace.prepareMutationBatch().setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel);
                    try {
                        result = (Rows)ShardedDistributedMessageQueue.this.keyspace.prepareQuery(ShardedDistributedMessageQueue.this.keyIndexColumnFamily).setConsistencyLevel(ShardedDistributedMessageQueue.this.consistencyLevel).getRowSlice(keys.keySet()).withColumnRange(metadataSerializer.buildRange().greaterThanEquals((byte)MessageMetadataEntryType.Unique.ordinal()).lessThanEquals((byte)MessageMetadataEntryType.Unique.ordinal()).build()).execute().getResult();
                    }
                    catch (ConnectionException e) {
                        throw new MessageQueueException("Failed to check keys for uniqueness (2): " + keys, e);
                    }
                    for (Row row : result) {
                        if (row.getColumns().size() != 1) {
                            String messageKey = ShardedDistributedMessageQueue.splitCompositeKey((String)row.getKey())[1];
                            notUniqueKeys.add(messageKey);
                            notUniqueMessages.add(keys.get(messageKey));
                            mb.withRow(ShardedDistributedMessageQueue.this.keyIndexColumnFamily, row.getKey()).deleteColumn(lockColumn);
                            continue;
                        }
                        mb.withRow(ShardedDistributedMessageQueue.this.keyIndexColumnFamily, row.getKey()).putEmptyColumn(lockColumn);
                    }
                }
                LinkedHashMap success = Maps.newLinkedHashMap();
                for (Message message : messages) {
                    if (message.hasKey() && notUniqueKeys.contains(message.getKey())) continue;
                    String messageId = ShardedDistributedMessageQueue.this.fillMessageMutation(mb, message);
                    success.put(messageId, message);
                }
                try {
                    mb.execute();
                }
                catch (ConnectionException e) {
                    throw new MessageQueueException("Failed to insert messages into queue.", e);
                }
                return new SendMessageResponse(success, notUniqueMessages);
            }
        };
    }

    private String fillMessageMutation(MutationBatch mb, Message message) throws MessageQueueException {
        long curTimeMicros = !message.hasTrigger() ? TimeUnit.MICROSECONDS.convert(System.currentTimeMillis(), TimeUnit.MILLISECONDS) + this.counter.incrementAndGet() % 1000L : TimeUnit.MICROSECONDS.convert(message.getTrigger().getTriggerTime(), TimeUnit.MILLISECONDS) + this.counter.incrementAndGet() % 1000L;
        message.setToken(TimeUUIDUtils.getMicrosTimeUUID(curTimeMicros));
        MessageQueueEntry entry = MessageQueueEntry.newMessageEntry(message.getPriority(), message.getToken(), MessageQueueEntryState.Waiting);
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            mapper.writeValue((OutputStream)baos, (Object)message);
            baos.flush();
        }
        catch (Exception e) {
            throw new MessageQueueException("Failed to serialize message data: " + message, e);
        }
        String shardKey = this.getShardKey(entry);
        mb.withRow(this.queueColumnFamily, shardKey).putColumn(entry, new String(baos.toByteArray()), this.settings.getRetentionTimeout());
        if (message.hasKey()) {
            mb.withRow(this.keyIndexColumnFamily, this.getCompositeKey(this.queueName, message.getKey())).putEmptyColumn(MessageMetadataEntry.newMessageId(this.getCompositeKey(shardKey, entry.getMessageId())), this.settings.getRetentionTimeout());
        }
        for (MessageQueueHooks hook : this.hooks) {
            hook.beforeSendMessage(message, mb);
        }
        this.stats.incSendMessageCount();
        return this.getCompositeKey(shardKey, entry.getMessageId());
    }

    @Override
    public Collection<MessageHistory> getKeyHistory(String key, Long startTime, Long endTime, int count) throws MessageQueueException {
        ColumnList columns;
        ArrayList list = Lists.newArrayList();
        try {
            columns = (ColumnList)this.keyspace.prepareQuery(this.historyColumnFamily).getRow(key).execute().getResult();
        }
        catch (ConnectionException e) {
            throw new MessageQueueException("Failed to load history for " + key, e);
        }
        for (Column column : columns) {
            try {
                list.add(this.deserializeString(column.getStringValue(), MessageHistory.class));
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
        return list;
    }

    public static class Builder {
        private ShardedDistributedMessageQueue queue = new ShardedDistributedMessageQueue();
        private String columnFamilyName = "Queues";

        public Builder withColumnFamily(String columnFamilyName) {
            this.columnFamilyName = columnFamilyName;
            return this;
        }

        public Builder withShardCount(int count) {
            this.queue.settings.setShardCount(count);
            return this;
        }

        public Builder withTimeBuckets(int bucketCount, int bucketDuration, TimeUnit units) {
            this.queue.settings.setPartitionDuration(TimeUnit.MICROSECONDS.convert(bucketDuration, units));
            this.queue.settings.setPartitionCount(bucketCount);
            return this;
        }

        public Builder withBuckets(int bucketCount, int bucketDuration, TimeUnit units) {
            this.queue.settings.setPartitionDuration(TimeUnit.MICROSECONDS.convert(bucketDuration, units));
            this.queue.settings.setPartitionCount(bucketCount);
            return this;
        }

        public Builder withRetentionTimeout(Long timeout, TimeUnit units) {
            this.queue.settings.setRetentionTimeout(timeout, units);
            return this;
        }

        public Builder withLockTimeout(Long timeout, TimeUnit units) {
            this.queue.lockTimeout = TimeUnit.MICROSECONDS.convert(timeout, units);
            return this;
        }

        public Builder withLockTtl(Long ttl, TimeUnit units) {
            this.queue.lockTtl = (int)TimeUnit.SECONDS.convert(ttl, units);
            return this;
        }

        public Builder withPollInterval(Long internval, TimeUnit units) {
            this.queue.pollInterval = TimeUnit.MILLISECONDS.convert(internval, units);
            return this;
        }

        public Builder withQueueName(String queueName) {
            this.queue.queueName = queueName;
            return this;
        }

        public Builder withConsistencyLevel(ConsistencyLevel level) {
            this.queue.consistencyLevel = level;
            return this;
        }

        public Builder withKeyspace(Keyspace keyspace) {
            this.queue.keyspace = keyspace;
            return this;
        }

        public Builder withStats(MessageQueueStats stats) {
            this.queue.stats = stats;
            return this;
        }

        public Builder withHook(MessageQueueHooks hooks) {
            this.queue.hooks.add(hooks);
            return this;
        }

        public Builder withHooks(Collection<MessageQueueHooks> hooks) {
            this.queue.hooks.addAll(hooks);
            return this;
        }

        public Builder withPoisonQueue(Boolean enabled) {
            this.queue.bPoisonQueueEnabled = enabled;
            return this;
        }

        public ShardedDistributedMessageQueue build() {
            this.queue.queueColumnFamily = ColumnFamily.newColumnFamily(this.columnFamilyName + ShardedDistributedMessageQueue.DEFAULT_QUEUE_SUFFIX, StringSerializer.get(), entrySerializer);
            this.queue.keyIndexColumnFamily = ColumnFamily.newColumnFamily(this.columnFamilyName + ShardedDistributedMessageQueue.DEFAULT_METADATA_SUFFIX, StringSerializer.get(), metadataSerializer);
            this.queue.historyColumnFamily = ColumnFamily.newColumnFamily(this.columnFamilyName + ShardedDistributedMessageQueue.DEFAULT_HISTORY_SUFFIX, StringSerializer.get(), TimeUUIDSerializer.get());
            this.queue.initialize();
            return this.queue;
        }
    }
}

