/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.astyanax.recipes.storage;

import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.netflix.astyanax.recipes.storage.AutoAllocatingLinkedBlockingQueue;
import com.netflix.astyanax.recipes.storage.ChunkedStorageProvider;
import com.netflix.astyanax.recipes.storage.NoOpObjectWriteCallback;
import com.netflix.astyanax.recipes.storage.ObjectMetadata;
import com.netflix.astyanax.recipes.storage.ObjectWriteCallback;
import com.netflix.astyanax.util.BlockingConcurrentWindowCounter;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ObjectWriter
implements Callable<ObjectMetadata> {
    private static final Logger LOG = LoggerFactory.getLogger(ObjectWriter.class);
    private static final int DEFAULT_CONCURRENCY_LEVEL = 4;
    private static final int MAX_WAIT_TIME_TO_FINISH = 60;
    private static final Integer NO_TTL = null;
    private final ChunkedStorageProvider provider;
    private final String objectName;
    private final InputStream is;
    private int chunkSize;
    private Integer ttl = NO_TTL;
    private int concurrencyLevel = 4;
    private int maxWaitTimeInSeconds = 60;
    private ObjectWriteCallback callback = new NoOpObjectWriteCallback();

    public ObjectWriter(ChunkedStorageProvider provider, String objectName, InputStream is) {
        this.provider = provider;
        this.objectName = objectName;
        this.chunkSize = provider.getDefaultChunkSize();
        this.is = is;
    }

    public ObjectWriter withChunkSize(int size) {
        this.chunkSize = size;
        return this;
    }

    public ObjectWriter withTtl(Integer ttl) {
        this.ttl = ttl;
        return this;
    }

    public ObjectWriter withConcurrencyLevel(int level) {
        this.concurrencyLevel = level;
        return this;
    }

    public ObjectWriter withMaxWaitTime(int maxWaitTimeInSeconds) {
        this.maxWaitTimeInSeconds = maxWaitTimeInSeconds;
        return this;
    }

    public ObjectWriter withCallback(ObjectWriteCallback callback) {
        this.callback = callback;
        return this;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ObjectMetadata call() throws Exception {
        LOG.debug("Writing: " + this.objectName);
        Preconditions.checkNotNull((Object)this.objectName, (Object)"Must provide a valid object name");
        Preconditions.checkNotNull((Object)this.is, (Object)"Must provide a valid input stream");
        Preconditions.checkNotNull((Object)this.chunkSize, (Object)"Must provide a valid chunkSize");
        final AtomicLong nBytesWritten = new AtomicLong(0L);
        final AtomicInteger nChunksWritten = new AtomicInteger(0);
        final AtomicReference exception = new AtomicReference();
        try {
            ExecutorService executor = Executors.newFixedThreadPool(this.concurrencyLevel, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("ChunkWriter-" + this.objectName + "-%d").build());
            final BlockingConcurrentWindowCounter chunkCounter = new BlockingConcurrentWindowCounter(this.concurrencyLevel);
            final AutoAllocatingLinkedBlockingQueue<ByteBuffer> blocks = new AutoAllocatingLinkedBlockingQueue<ByteBuffer>(this.concurrencyLevel);
            try {
                boolean done = false;
                while (!done && exception.get() == null) {
                    final int chunkNumber = chunkCounter.incrementAndGet();
                    final ByteBuffer bb = blocks.poll(new Supplier<ByteBuffer>(){

                        public ByteBuffer get() {
                            return ByteBuffer.allocate(ObjectWriter.this.chunkSize);
                        }
                    });
                    bb.position(0);
                    int nBytesRead = ObjectWriter.readFully(this.is, bb.array(), 0, this.chunkSize);
                    if (nBytesRead > 0) {
                        bb.limit(nBytesRead);
                        executor.submit(new Runnable(){

                            /*
                             * WARNING - Removed try catching itself - possible behaviour change.
                             */
                            @Override
                            public void run() {
                                try {
                                    if (exception.get() == null) {
                                        LOG.debug("WRITE " + chunkNumber + " size=" + bb.limit());
                                        ObjectWriter.this.provider.writeChunk(ObjectWriter.this.objectName, chunkNumber, bb, ObjectWriter.this.ttl);
                                        ObjectWriter.this.callback.onChunk(chunkNumber, bb.limit());
                                        nBytesWritten.addAndGet(bb.limit());
                                        nChunksWritten.incrementAndGet();
                                    }
                                }
                                catch (Exception e) {
                                    LOG.error(e.getMessage());
                                    exception.compareAndSet(null, e);
                                    ObjectWriter.this.callback.onChunkException(chunkNumber, e);
                                }
                                finally {
                                    blocks.add(bb);
                                    chunkCounter.release(chunkNumber);
                                }
                            }
                        });
                        continue;
                    }
                    done = true;
                }
                if (exception.get() != null) {
                    throw (Exception)exception.get();
                }
            }
            finally {
                executor.shutdown();
                if (!executor.awaitTermination(this.maxWaitTimeInSeconds, TimeUnit.SECONDS)) {
                    throw new Exception("Took too long to write object: " + this.objectName);
                }
            }
            ObjectMetadata attr = new ObjectMetadata().setChunkCount(nChunksWritten.get()).setObjectSize(nBytesWritten.get()).setChunkSize(this.chunkSize).setTtl(this.ttl);
            this.provider.writeMetadata(this.objectName, attr);
            this.callback.onSuccess();
            return attr;
        }
        catch (Exception e) {
            this.callback.onFailure(e);
            LOG.warn(e.getMessage());
            e.printStackTrace();
            try {
                this.provider.deleteObject(this.objectName, nChunksWritten.get() + this.concurrencyLevel);
            }
            catch (Exception e2) {
                LOG.warn(e2.getMessage());
            }
            throw e;
        }
    }

    private static int readFully(InputStream in, byte[] b, int off, int len) throws IOException {
        int got;
        int total = 0;
        do {
            if ((got = in.read(b, off + total, len - total)) >= 0) continue;
            return total == 0 ? -1 : total;
        } while ((total += got) != len);
        return total;
    }
}

