package org.gcube.portal.oidc.lr62;

import java.util.Collections;
import java.util.Comparator;
import java.util.Map;
import java.util.TreeMap;

import javax.servlet.http.HttpSession;

import org.gcube.oidc.rest.JWTToken;

import com.liferay.portal.model.User;

public class IntraVMJWTCacheProxy extends JWTCacheProxy {

    private Map<User, Map<HttpSession, String>> oidcTokens;
    private Map<User, Map<HttpSession, String>> rptTokens;

    protected IntraVMJWTCacheProxy() {
        log.info("Initializing tokens caches");
        oidcTokens = Collections.synchronizedMap(new TreeMap<>());
        rptTokens = Collections.synchronizedMap(new TreeMap<>());
    }

    public void checkMap(Map<User, Map<HttpSession, String>> map, User user) {
        if (!map.containsKey(user)) {
            log.debug("Initializing map for user: " + user.getScreenName());
            map.put(user, Collections.synchronizedMap(new TreeMap<>(new Comparator<HttpSession>() {
                @Override
                public int compare(HttpSession s1, HttpSession s2) {
                    return s1.getId().compareTo(s2.getId());
                }
            })));
        }
    }

    @Override
    public void setOIDCToken(User user, HttpSession session, JWTToken token) {
        checkMap(oidcTokens, user);
        Map<HttpSession, String> oidcUserMap = oidcTokens.get(user);
        log.trace("OIDC user's sessions recorded: " + oidcUserMap.keySet().toArray());
        oidcUserMap.put(session, JWTTokenUtil.getRawContent(token));
    }

    @Override
    public void setUMAToken(User user, HttpSession session, JWTToken token) {
        checkMap(rptTokens, user);
        Map<HttpSession, String> umaUserMap = rptTokens.get(user);
        log.trace("UMA user's sessions recorded objs: " + umaUserMap.keySet().toArray());
        umaUserMap.put(session, JWTTokenUtil.getRawContent(token));
    }

    @Override
    public JWTToken getOIDCToken(User user, HttpSession session) {
        checkMap(oidcTokens, user);
        log.trace("Getting OIDC token for user '" + user.getScreenName() + "' and session obj: " + session);
        return JWTTokenUtil.fromString(oidcTokens.get(user).get(session));
    }

    @Override
    public JWTToken getUMAToken(User user, HttpSession session) {
        checkMap(rptTokens, user);
        log.trace("Getting UMA token for user '" + user.getScreenName() + "' and session obj: " + session);
        return JWTTokenUtil.fromString(rptTokens.get(user).get(session));
    }

    @Override
    public void removeOIDCToken(User user, HttpSession session) {
        checkMap(oidcTokens, user);
        if (oidcTokens.get(user).containsKey(session)) {
            log.trace("Removing OIDC token for user '" + user.getScreenName() + "' and session obj: " + session);
            oidcTokens.get(user).remove(session);
        }
    }

    @Override
    public void removeUMAToken(User user, HttpSession session) {
        checkMap(rptTokens, user);
        if (rptTokens.get(user).containsKey(session)) {
            log.trace("Removing UMA token for user '" + user.getScreenName() + "' and session obj: " + session);
            oidcTokens.get(user).remove(session);
        }
    }

    @Override
    public void removeAllOIDCTokens(User user) {
        log.trace("Removing OIDC session's tokens of user: " + user.getScreenName());
        oidcTokens.remove(user);
    }

    @Override
    public void removeAllUMATokens(User user) {
        log.trace("Removing UMA session's tokens of user: " + user.getScreenName());
        rptTokens.remove(user);
    }

    @Override
    public void clearOIDCTokens() {
        log.trace("Clearing OIDC tokens cache");
        oidcTokens.clear();
    }

    @Override
    public void clearUMATokens() {
        log.trace("Clearing UMA tokens cache");
        rptTokens.clear();
    }

    @Override
    public void clearAllTokens() {
        clearOIDCTokens();
        clearUMATokens();
    }

}
