package org.gcube.portal.oidc.lr62;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.gcube.oidc.D4ScienceMappings;
import org.gcube.oidc.OIDCToSitesAndRolesMapper;
import org.gcube.oidc.Site;
import org.gcube.oidc.SitesMapperExecption;
import org.gcube.vomanagement.usermanagement.GroupManager;
import org.gcube.vomanagement.usermanagement.RoleManager;
import org.gcube.vomanagement.usermanagement.UserManager;
import org.gcube.vomanagement.usermanagement.exception.GroupRetrievalFault;
import org.gcube.vomanagement.usermanagement.exception.RoleRetrievalFault;
import org.gcube.vomanagement.usermanagement.exception.UserManagementPortalException;
import org.gcube.vomanagement.usermanagement.exception.UserManagementSystemException;
import org.gcube.vomanagement.usermanagement.exception.UserRetrievalFault;
import org.gcube.vomanagement.usermanagement.impl.LiferayGroupManager;
import org.gcube.vomanagement.usermanagement.impl.LiferayRoleManager;
import org.gcube.vomanagement.usermanagement.impl.LiferayUserManager;
import org.gcube.vomanagement.usermanagement.model.GCubeGroup;
import org.gcube.vomanagement.usermanagement.model.GCubeRole;

import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.model.User;
import com.liferay.portal.security.auth.PrincipalThreadLocal;
import com.liferay.portal.security.permission.PermissionCheckerFactoryUtil;
import com.liferay.portal.security.permission.PermissionThreadLocal;
import com.liferay.portal.service.UserLocalServiceUtil;

public class UserSitesToGroupsAndRolesMapper {

    protected static final Log log = LogFactoryUtil.getLog(UserSitesToGroupsAndRolesMapper.class);

    protected User user;
    protected OIDCToSitesAndRolesMapper mapper;
    protected UserManager userManager;
    protected GroupManager groupManager;
    protected RoleManager roleManager;
    protected String rootVOName;
    protected Map<GCubeGroup, List<GCubeRole>> actualGroupAndRoles;
    protected Map<String, GCubeRole> roleNameToRole;

    public UserSitesToGroupsAndRolesMapper(User user, OIDCToSitesAndRolesMapper mapper) {
        this.user = user;
        this.mapper = mapper;
        try {
            if (log.isDebugEnabled()) {
                log.debug("Creating the permission checker for admin user");
            }
            long adminUserId = LiferayUserManager.getAdmin().getUserId();
            PrincipalThreadLocal.setName(adminUserId);
            PermissionThreadLocal.setPermissionChecker(
                    PermissionCheckerFactoryUtil.create(UserLocalServiceUtil.getUser(adminUserId)));

        } catch (Exception e) {
            log.fatal("Cannot create permission checker for admin user", e);
            return;
        }
        userManager = new LiferayUserManager();
        groupManager = new LiferayGroupManager();
        roleManager = new LiferayRoleManager();
        try {
            this.rootVOName = groupManager.getRootVOName();
        } catch (UserManagementSystemException | GroupRetrievalFault e) {
            log.error("Cannot get infrastructure's Root VO", e);
            return;
        }
        try {
            actualGroupAndRoles = groupManager.listGroupsAndRolesByUser(user.getUserId());
        } catch (UserManagementSystemException e) {
            log.error("Cannot get sites and roles membership for user", e);
            return;
        }
        roleNameToRole = new TreeMap<>();
        for (GCubeRole role : roleManager.listAllGroupRoles()) {
            roleNameToRole.put(role.getRoleName(), role);
        }
    }

    public void map() {
        log.info("Mapping roles to sites for user: " + user.getScreenName());
        Site gwSitesTree = null;
        try {
            gwSitesTree = mapper.map(rootVOName);
            if (log.isDebugEnabled()) {
                log.debug("Sites tree is: " + (gwSitesTree != null ? gwSitesTree.dump() : "null"));
            }
        } catch (SitesMapperExecption e) {
            log.error("Computing sites tree in concrete mapper class", e);
        }
        if (gwSitesTree != null) {
            if (log.isDebugEnabled()) {
                log.debug("Check user to sites assignemnts");
            }
            rolesToSiteDescendant(gwSitesTree, null);
        }
        if (log.isDebugEnabled()) {
            log.debug("Check user to sites removal");
        }
        checkForSiteRemoval(gwSitesTree);
    }


    protected void rolesToSiteDescendant(Site actualSite, GCubeGroup parentGroup) {
        GCubeGroup actualSiteGroup = null;
        try {
            if (log.isDebugEnabled()) {
                log.debug("Getting actual site group from group manager, actual site name=" + actualSite.getName());
            }
            actualSiteGroup = groupManager.getGroup(groupManager.getGroupId(actualSite.getName()));
        } catch (UserManagementSystemException | GroupRetrievalFault e) {
            log.error("Cannot retrieve group for site: " + actualSite.getName(), e);
            return;
        }
        try {
            if (groupManager.isVRE(actualSiteGroup.getGroupId()) && !actualGroupAndRoles.containsKey(actualSiteGroup)) {
                log.info("Assigning user to new VRE site: " + actualSiteGroup.getGroupName());
                userManager.assignUserToGroup(actualSiteGroup.getGroupId(), user.getUserId());
                if (actualSite.getRoles() != null && !actualSite.getRoles().isEmpty()) {
                    log.info("Assiging roles for the VRE site");
                    for (String roleName : actualSite.getRoles()) {
                        if (D4ScienceMappings.Role.MEMBER.asString().equals(roleName)) {
                            // Member role is only to assure that the user belongs to context
                            continue;
                        }
                        roleManager.assignRoleToUser(user.getUserId(), actualSiteGroup.getGroupId(),
                                roleNameToRole.get(roleName).getRoleId());
                    }
                    // Since it's a VRE we can return
                    return;
                } else {
                    log.info("User has no roles in the VRE site");
                }
            }
        } catch (UserManagementSystemException | GroupRetrievalFault | UserRetrievalFault
                | UserManagementPortalException | RoleRetrievalFault | RuntimeException e) {

            log.error("Assigning user to new VRE site: " + actualSiteGroup.getGroupName(), e);
        }
        if (actualSite.getRoles() != null) {
            List<GCubeRole> actualSiteGroupRoles = actualGroupAndRoles.get(actualSiteGroup);
            List<String> newRoles = new ArrayList<>(actualSite.getRoles());
            // Removing the Member role that is not a real role in LR
            newRoles.remove(D4ScienceMappings.Role.MEMBER.asString());
            if (actualSiteGroupRoles != null && !actualSiteGroupRoles.isEmpty()) {
                log.info("Checking actual roles in the dite group");
                for (GCubeRole gcRole : actualSiteGroupRoles) {
                    String actualSiteName = actualSite.getName();
                    String gcRoleName = gcRole.getRoleName();
                    if (!actualSite.getRoles().contains(gcRoleName)) {
                        try {
                            log.info("Removing '" + gcRoleName + "' user's role for site: " + actualSiteName);
                            roleManager.removeRoleFromUser(user.getUserId(), actualSiteGroup.getGroupId(),
                                    gcRole.getRoleId());
                        } catch (UserManagementSystemException | UserRetrievalFault | GroupRetrievalFault
                                | RoleRetrievalFault e) {
                            log.error(
                                    "Cannot remove user's role '" + gcRoleName + "' for site: " + actualSite.getName(),
                                    e);
                            continue;
                        }
                    } else {
                        if (log.isDebugEnabled()) {
                            log.debug("Removing site role from the roles list: " + gcRoleName);
                        }
                        newRoles.remove(gcRoleName);
                    }
                }
            } else {
                log.info("User actually has no roles in the site group");
            }
            // Adding roles that remaining in newRoles list, if any, for the user in this
            // site
            for (String newRole : newRoles) {
                if (log.isDebugEnabled()) {
                    log.debug("Adding new role to user. New role=" + newRole);
                }
                GCubeRole newGcRole = roleNameToRole.get(newRole);
                if (newGcRole != null) {
                    try {
                        log.info("Assinging new role '" + newRole + "' to user");
                        roleManager.assignRoleToUser(user.getUserId(), actualSiteGroup.getGroupId(),
                                newGcRole.getRoleId());
                    } catch (UserManagementSystemException | UserRetrievalFault | GroupRetrievalFault
                            | RoleRetrievalFault e) {
                        log.error("Cannot assign new role '" + newRole + "' for site: " + actualSite.getName(), e);
                        continue;
                    }
                } else {
                    log.warn("New site's gc role is null (doesn't exist?) after getting it from role manager: "
                            + newRole);
                }
            }
        } else {
            log.info("Roles were not set, continuing descending letting them untouched in site: "
                    + actualSite.getName());
        }
        for (String childSite : actualSite.getChildren().keySet()) {
            log.info("Recursive call to child site: " + childSite);
            rolesToSiteDescendant(actualSite.getChildren().get(childSite), actualSiteGroup);
        }
    }

    protected void checkForSiteRemoval(Site gwSitesTree) {
        List<String> vreNames = new ArrayList<>();
        if (gwSitesTree != null) {
            log.debug("Collecting VREs user belongs to");
            for (String voName : gwSitesTree.getChildren().keySet()) {
                for (String vreName : gwSitesTree.getChildren().get(voName).getChildren().keySet()) {
                    log.debug("Adding VRE to the list: " + vreName);
                    vreNames.add(vreName);
                }
            }
        } else {
            log.info("User not belongs to any site");
        }
        for (GCubeGroup actualGroup : actualGroupAndRoles.keySet()) {
            try {
                if (groupManager.isVRE(actualGroup.getGroupId()) && !vreNames.contains(actualGroup.getGroupName())) {
                    log.info("Removing user from VRE: " + actualGroup.getGroupName());
                    try {
                        userManager.dismissUserFromGroup(actualGroup.getGroupId(), user.getUserId());
                    } catch (UserRetrievalFault e) {
                        log.error("Removing user from VRE: " + actualGroup.getGroupName(), e);
                    }
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("User still belong to VRE: " + actualGroup.getGroupName());
                    }
                }
            } catch (UserManagementSystemException | GroupRetrievalFault e) {
                log.error("Checking if site group is a VRE", e);
            }
        }
    }

}
