package org.gcube.portal.istischeduler.impl;

import java.io.File;
import java.io.FileInputStream;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.ResourceBundle;

import org.gcube.portal.istischeduler.SiManager;
import org.gcube.portal.istischeduler.model.SiEntry;
import org.gcube.portal.istischeduler.model.SiServerEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.liferay.portal.kernel.dao.db.DB;
import com.liferay.portal.kernel.dao.db.DBFactoryUtil;
import com.liferay.portal.kernel.dao.jdbc.DataAccess;
/**
 * Create 2 auxiliary tables needed by Si (Sistema informativo ISTI) to update the Liferay DB correctly
 * @author Massimiliano Assante, ISTI-CNR
 *
 */
public class LiferaySiManager implements SiManager {
	/**
	 * logger
	 */
	private static final Logger _log = LoggerFactory.getLogger(LiferaySiManager.class);

	private static final String ISTI_SI_FLOWS_TABLE = "_isti_flows";
	private static final String ISTI_SI_LOGS_TABLE = "_isti_logs";

	private Connection con = null;
	private Statement st = null;
	private ResultSet rs = null;

	/**
	 * Check if the tables exist
	 */
	public LiferaySiManager() {
		try {
			con = getDBConnection();
			if (! tableExists(con)) {
				initializeTables();
			}
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean addSiEntry(SiEntry entry) throws SQLException {
		if (con == null || con.isClosed())
			con = getDBConnection();
		_log.trace("Adding Si Entry in Table Flows");
		String insert = "INSERT into " + ISTI_SI_FLOWS_TABLE + "(id , tab, operation, flowid, lastupdate) values(?, ?, ?, ?, ?)";
		PreparedStatement statement = con.prepareStatement(insert);

		statement.setLong(1, entry.getId());
		statement.setString(2, entry.getTable());
		statement.setString(3, entry.getOperation());
		statement.setLong(4, entry.getIdFlow());
		long timeInMillis = entry.getLastUpdate() != null ? entry.getLastUpdate().getTime() : 0;
		statement.setTimestamp(5, new Timestamp(timeInMillis));
		statement.executeUpdate();
		return true;
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean updateSiEntry(SiEntry entry) throws SQLException {
		_log.trace("Updating Si Entry in Table Flows where id = " + entry.getId());
		String insert = "UPDATE "+ISTI_SI_FLOWS_TABLE+" SET tab = ?, operation = ?, flowid = ?, lastupdate = ? WHERE id = ?";

		PreparedStatement statement = con.prepareStatement(insert);

		statement.setLong(5, entry.getId());
		statement.setString(1, entry.getTable());
		statement.setString(2, entry.getOperation());
		statement.setLong(3, entry.getIdFlow());
		long timeInMillis = entry.getLastUpdate() != null ? entry.getLastUpdate().getTime() : 0;
		statement.setTimestamp(4, new Timestamp(timeInMillis));
		statement.executeUpdate();
		return true;
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public SiEntry getSiEntryById(long id) throws SQLException {

		String select = "SELECT * FROM "+ISTI_SI_FLOWS_TABLE+" where id = ?";
		PreparedStatement statement = con.prepareStatement(select);
		statement.setLong(1, id);
		ResultSet rs = statement.executeQuery();
		if (rs.next()) {
			long timeInMillis =  (rs.getTimestamp("lastupdate") != null) ? rs.getTimestamp("lastupdate").getTime() : 0;
			SiEntry toReturn = new SiEntry(id, rs.getString("tab"), rs.getString("operation"), rs.getLong("flowid"), new Date(timeInMillis));
			return toReturn;
		}
		return null;
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<SiEntry> listSiEntries() throws SQLException {
		String select = "SELECT * FROM "+ISTI_SI_FLOWS_TABLE;
		PreparedStatement statement = con.prepareStatement(select);
		ResultSet rs = statement.executeQuery();

		List<SiEntry> toReturn = new ArrayList<SiEntry>();
		while (rs.next()) {
			long timeInMillis =  (rs.getTimestamp("lastupdate") != null) ? rs.getTimestamp("lastupdate").getTime() : 0;
			SiEntry toAdd = new SiEntry( rs.getLong("id"), rs.getString("tab"), rs.getString("operation"), rs.getLong("flowid"), new Date(timeInMillis));
			toReturn.add(toAdd);
		}		
		return toReturn;
	}
	/**
	 * {@inheritDoc}
	 * @throws SQLException 
	 */
	@Override
	public boolean addSiLogEntry(Date logDate, String logResult, String logInfo) throws SQLException {
		_log.trace("Adding Si Entry in Table Logs");
		String insert = "INSERT into " + ISTI_SI_LOGS_TABLE + "(timestamp, result, info) values(?, ?, ?)";


		PreparedStatement statement = con.prepareStatement(insert);

		long timeInMillis = logDate != null ? logDate.getTime() : 0;
		statement.setTimestamp(1, new Timestamp(timeInMillis));
		statement.setString(2, logResult);
		statement.setString(3, logInfo);
		statement.executeUpdate();
		return true;
	}

	/**
	 * read from the liferay portal ext props the coordinates of the db and return a {@link Connection}
	 * @return a {@link Connection} for the db
	 * @throws SQLException
	 */
	private Connection getDBConnection() throws SQLException {
//		Properties props = new Properties();
//		String propertyfile = getTomcatFolder()+"webapps/ROOT/WEB-INF/classes/portal-ext.properties";			
//		File propsFile = new File(propertyfile);
//		FileInputStream fis;
//
//		String dbUrl = "";
//		String dbUser = "";
//		String dbPassword = "";
//
//		try {
//			fis = new FileInputStream(propsFile);
//			props.load( fis);
//			dbUrl = props.getProperty("jdbc.default.url");
//			dbUser = props.getProperty("jdbc.default.username");
//			dbPassword = props.getProperty("jdbc.default.password");
//
//		} catch (Exception e) {
//			_log.error("Problems reading LR portal-ext.properties file, using fallback configuration see org.gcube.portal.istischeduler#settings.properties");
//
//			ResourceBundle rb = ResourceBundle.getBundle("org.gcube.portal.istischeduler.settings");
//			dbUrl = rb.getString("PG_PROVIDER_URL");
//			dbUser = rb.getString("PG_SECURITY_PRINCIPAL");
//			dbPassword = rb.getString("PG_SECURITY_CREDENTIALS");
//		} 
//
//		_log.info("Liferay DB Settings Found, using db at: " + dbUrl);
//		_log.info("passwd=" + dbPassword);
//
//		return DriverManager.getConnection(dbUrl, dbUser, dbPassword);	
		_log.info("Trying to get Connection From API");
		Connection toReturn = DataAccess.getConnection();
		_log.info("CONNECTED TO LR DB!");
		return toReturn;
	}

	/**
	 * check if tables exist in the database
	 * @param conn .
	 * @throws SQLException 
	 */
	private boolean tableExists(Connection conn) throws SQLException {
		_log.debug("Looking for Auxiliary Tables ... ");
		Statement s = con.createStatement();
		ResultSet rs = s.executeQuery("SELECT * FROM  pg_tables where schemaname='public' and  tablename = '" + ISTI_SI_FLOWS_TABLE +"' ");
		boolean toReturn =  rs.next();
		if (toReturn)
			_log.info("Auxiliary Tables Found! Returning ... ");
		return toReturn;
	}
	/**
	 * create the table for flowEntries, logEntries and populate flowEntried with default data
	 * @param s
	 * @return
	 * @throws SQLException
	 */
	private boolean initializeTables() {

		boolean resultFlow = false;
		boolean resultLog = false;
		try {
			Statement s = con.createStatement();
			_log.info("Flow Entries Table to initiliaze");
			resultFlow = s.execute("CREATE TABLE " + ISTI_SI_FLOWS_TABLE 
					+ " (id INT NOT NULL DEFAULT '0', "
					+ "tab varchar(128)  DEFAULT NULL, "
					+ "operation varchar(128)  NOT NULL, "
					+ "flowid varchar(128)  DEFAULT NULL, "
					+ "lastupdate TIMESTAMP DEFAULT NULL)");

			_log.info("Flow Entries Table created OK!");

			_log.info("Log Entries Table to initiliaze");
			resultLog = s.execute("CREATE TABLE " + ISTI_SI_LOGS_TABLE 
					+ " (id BIGSERIAL PRIMARY KEY, "
					+ "timestamp TIMESTAMP DEFAULT NULL, "
					+ "result varchar(256)  NOT NULL, "
					+ "info varchar(256)  DEFAULT NULL) ");

			_log.info("Log Entries Table created OK!");

			//populate table flow
			_log.info("Populating Table Flows");
			String insert = "INSERT into " + ISTI_SI_FLOWS_TABLE + "(id , tab, operation, flowid, lastupdate) values(?, ?, ?, ?, ?)";
			PreparedStatement statement = con.prepareStatement(insert);

			statement.setInt(1, 1);
			statement.setString(2, "anagrafico");
			statement.setString(3, "R");
			statement.setLong(4, 0L);
			statement.setTimestamp(5, null);
			statement.executeUpdate();

			statement.setInt(1, 2);
			statement.setString(2, "mail");
			statement.executeUpdate();

			statement.setInt(1, 3);
			statement.setString(2, "posizioni");
			statement.executeUpdate();

			_log.info("Table Flows Popoluted, returning ...");

		} catch (SQLException e) {
			e.printStackTrace();
		} finally {
			try {
				if (rs != null) rs.close();
				if (st != null) st.close();
				if (con != null) con.close();
			} catch (SQLException ex) {
				ex.printStackTrace();
			}
		}
		return true;
	}

	/**
	 * 
	 * @return $CATALINA_HOME
	 */
	public static String getTomcatFolder() {
		if (System.getenv("CATALINA_HOME") != null) 
			return (System.getenv("CATALINA_HOME").endsWith("/") ? System.getenv("CATALINA_HOME") : System.getenv("CATALINA_HOME")+"/");
		_log.error("CATALINA_HOME not Found");
		return "CATALINA_HOME not Found";
	}
	/**
	 * {@inheritDoc}
	 */
	//TODO: Missing implementation
	@Override
	public SiServerEntry getSiEndPoint() {
		return new SiServerEntry("sistemainformativo", "SI DB Endpoint", "sistemainformativo-dev.isti.cnr.it", "5432", "socialisti", "!socialisti_123");
	}
}
