package org.gcube.portal.istischeduler.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.Set;

import org.gcube.portal.istischeduler.PortalUserManager;
import org.gcube.portal.istischeduler.ex.UserManagementSystemException;
import org.gcube.portal.istischeduler.ex.UserRetrievalFault;
import org.gcube.portal.istischeduler.model.PortalUserModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.exception.SystemException;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.PropsUtil;
import com.liferay.portal.model.Company;
import com.liferay.portal.model.Group;
import com.liferay.portal.model.User;
import com.liferay.portal.security.permission.PermissionChecker;
import com.liferay.portal.security.permission.PermissionCheckerFactoryUtil;
import com.liferay.portal.security.permission.PermissionThreadLocal;
import com.liferay.portal.service.CompanyLocalServiceUtil;
import com.liferay.portal.service.GroupLocalServiceUtil;
import com.liferay.portal.service.ServiceContext;
import com.liferay.portal.service.UserLocalServiceUtil;
/**
 * 
 * @author Massimiliano Assante, ISTI-CNR
 *
 */
public class LiferayPortalUserManager implements PortalUserManager {
	/**
	 * logger
	 */
	private static final Logger _log = LoggerFactory.getLogger(LiferaySiManager.class);

	private static final String ROOT_ORG = "rootorganization";
	private static final String DEFAULT_COMMUNITY_PROP_NAME = "defaultcommunity";
	private static final String DEFAULT_COMPANY_WEB_ID = "liferay.com";
	/**
	 * these are the custom added fields for the Liferay Users Table
	 */
	private final static String POSITION = "POSITION";
	private final static String LABS = "LABS";
	private final static String FISCAL_CODE = "FISCAL_CODE";
	private final static String OPTIONAL_IDENTIFIER = "OPTIONAL_IDENTIFIER";

	private String administratorUsername;
	/**
	 * @param administratorUsername one of the administrators username (screenname) of the portal
	 */
	public LiferayPortalUserManager(String administratorUsername) {
		this.administratorUsername = administratorUsername;
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean createUser(PortalUserModel user2Add) throws UserManagementSystemException, UserRetrievalFault {
		_log.debug("Creating user with username = " + user2Add.getUsername());
		try {
			long companyId = getCompany().getCompanyId();

			User user = UserLocalServiceUtil.getUserByScreenName(companyId, administratorUsername);
			PermissionChecker permissionChecker = PermissionCheckerFactoryUtil.create(user, false);
			PermissionThreadLocal.setPermissionChecker(permissionChecker);

			String password1 = null;
			Locale locale = new Locale("en_US");
			ServiceContext serviceContext = new ServiceContext();
			String password2 = null;		

			Group defaultCommunity = GroupLocalServiceUtil.getGroup(companyId, getDefaultCommunityName());
			long[] groupids = {defaultCommunity.getGroupId()};

			User toAdd = UserLocalServiceUtil.addUser(
					0L, companyId, true, password1, password2, false, 
					user2Add.getUsername(), user2Add.getEmail(), 0L,"",
					locale, user2Add.getFirstname(), "", user2Add.getLastname(), 0, 0, true, 
					1, 1, 1940, "", groupids, null, null, null, true, serviceContext);


			//add the custom attrs
			if (! toAdd.getExpandoBridge().hasAttribute(POSITION)) 	toAdd.getExpandoBridge().addAttribute(POSITION);
			if (! toAdd.getExpandoBridge().hasAttribute(LABS)) 	toAdd.getExpandoBridge().addAttribute(LABS);
			if (! toAdd.getExpandoBridge().hasAttribute(FISCAL_CODE)) toAdd.getExpandoBridge().addAttribute(FISCAL_CODE);
			if (! toAdd.getExpandoBridge().hasAttribute(OPTIONAL_IDENTIFIER)) toAdd.getExpandoBridge().addAttribute(OPTIONAL_IDENTIFIER);

			toAdd.getExpandoBridge().setAttribute(POSITION, user2Add.getPosition());
			toAdd.getExpandoBridge().setAttribute(LABS, user2Add.getGroups());
			toAdd.getExpandoBridge().setAttribute(FISCAL_CODE, user2Add.getFiscalCode());
			toAdd.getExpandoBridge().setAttribute(OPTIONAL_IDENTIFIER, user2Add.getOptionalIdentifier());

		} catch (SystemException e) {
			throw new UserManagementSystemException("Error creating user ",user2Add.getUsername(), e);
		} catch (PortalException e) {			
			throw new UserRetrievalFault("Error adding user this user here -> ", user2Add.toString(),e);
		} catch (Exception e) {
			e.printStackTrace();
		}
		_log.debug("Creating user OK! ");
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean updateUser(PortalUserModel user2Update) throws UserManagementSystemException, UserRetrievalFault {
		try {
			User user = null;
			if (user2Update.getUsername() == null || user2Update.getUsername().isEmpty())
				return false;

			long companyId = getCompany().getCompanyId();
			User user4Permissions = UserLocalServiceUtil.getUserByScreenName(companyId, administratorUsername);
			PermissionChecker permissionChecker = PermissionCheckerFactoryUtil.create(user4Permissions, false);
			PermissionThreadLocal.setPermissionChecker(permissionChecker);
			
			user = UserLocalServiceUtil.getUserByScreenName(companyId, user2Update.getUsername());
			
			System.out.println("updateUser param = "+user2Update.getUsername());
			
			if (user2Update.getEmail() != null && user2Update.getEmail().compareTo("") != 0)
				user.setEmailAddress(user2Update.getEmail());
			
			if (user2Update.getFirstname() != null && user2Update.getFirstname().compareTo("") != 0)
				user.setFirstName(user2Update.getFirstname());
			
			if (user2Update.getLastname() != null && user2Update.getLastname().compareTo("") != 0)
				user.setLastName(user2Update.getLastname());
			
			if (user2Update.getPosition() != null && user2Update.getPosition().compareTo("") != 0)
				user.getExpandoBridge().setAttribute(POSITION, user2Update.getPosition());
			
			if (user2Update.getGroups() != null && user2Update.getGroups().compareTo("") != 0)
				user.getExpandoBridge().setAttribute(LABS, user2Update.getGroups());
			
			if (user2Update.getFiscalCode() != null && user2Update.getFiscalCode().compareTo("") != 0)
				user.getExpandoBridge().setAttribute(FISCAL_CODE, user2Update.getFiscalCode());
			
			if (user2Update.getOptionalIdentifier() != null && user2Update.getOptionalIdentifier().compareTo("") != 0)
				user.getExpandoBridge().setAttribute(OPTIONAL_IDENTIFIER, user2Update.getOptionalIdentifier());		
			
			UserLocalServiceUtil.updateUser(user);			
			System.out.println("User and Custom Attributes updated: " + user.getScreenName());
		}
		catch (com.liferay.portal.NoSuchUserException es) {
			_log.warn("User to modify does not exist");
			return false;
		}
		catch (PortalException e) {
			throw new UserRetrievalFault("Error retrieving user by screen name ",user2Update.toString(), e);
		}
		catch (Exception ex) {
			ex.printStackTrace();
		}
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public PortalUserModel getUserByUsername(String username) throws UserManagementSystemException {
		User user = null;
		PortalUserModel um = null;
		try {
			long companyId = getCompany().getCompanyId();
			user = UserLocalServiceUtil.getUserByScreenName(companyId, administratorUsername);
			PermissionChecker permissionChecker = PermissionCheckerFactoryUtil.create(user, false);
			PermissionThreadLocal.setPermissionChecker(permissionChecker);

			user = UserLocalServiceUtil.getUserByScreenName(companyId, username);
			String position = (user.getExpandoBridge().getAttribute(POSITION) != null) ? user.getExpandoBridge().getAttribute(POSITION).toString() : "";
			String labs =  (user.getExpandoBridge().getAttribute(LABS) != null) ? user.getExpandoBridge().getAttribute(LABS).toString() : "";
			String fiscalCode = (user.getExpandoBridge().getAttribute(FISCAL_CODE) != null) ? user.getExpandoBridge().getAttribute(FISCAL_CODE).toString() : "";
			String optionalId = (user.getExpandoBridge().getAttribute(OPTIONAL_IDENTIFIER) != null) ? user.getExpandoBridge().getAttribute(OPTIONAL_IDENTIFIER).toString() : "";
			um = new PortalUserModel(
					user.getScreenName(), 
					user.getFirstName(), 
					user.getLastName(), 
					user.getEmailAddress(),
					fiscalCode,
					optionalId,
					position, 
					labs
					);
		} catch (SystemException e) {
			throw new UserManagementSystemException("Error retrieving user by screename  ", username, e);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return um;
	}
	/**
	 * {@inheritDoc}
	 * @throws UserManagementSystemException 
	 */
	@Override
	public PortalUserModel getUserByFiscalCode(String fiscalCode) throws UserManagementSystemException {
		for (PortalUserModel user : listPortalUsers()) {
			if (user.getFiscalCode().compareTo(fiscalCode) == 0)
				return user;
		}
		return null;
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<PortalUserModel> listPortalUsers() throws UserManagementSystemException {
		List<User> users = null;
		List<PortalUserModel> userModels = new ArrayList<PortalUserModel>();
		try {
			long companyId = getCompany().getCompanyId();
			User permissionUser = UserLocalServiceUtil.getUserByScreenName(companyId, administratorUsername);
			PermissionChecker permissionChecker = PermissionCheckerFactoryUtil.create(permissionUser, false);
			PermissionThreadLocal.setPermissionChecker(permissionChecker);
			
			users = UserLocalServiceUtil.getUsers(0, UserLocalServiceUtil.getUsersCount());
			for(User user : users){
				String position = (user.getExpandoBridge().getAttribute(POSITION) != null) ? user.getExpandoBridge().getAttribute(POSITION).toString() : "";
				String labs =  (user.getExpandoBridge().getAttribute(LABS) != null) ? user.getExpandoBridge().getAttribute(LABS).toString() : "";
				String fiscalCode = (user.getExpandoBridge().getAttribute(FISCAL_CODE) != null) ? user.getExpandoBridge().getAttribute(FISCAL_CODE).toString() : "";
				String optionalId = (user.getExpandoBridge().getAttribute(OPTIONAL_IDENTIFIER) != null) ? user.getExpandoBridge().getAttribute(OPTIONAL_IDENTIFIER).toString() : "";

				PortalUserModel um = new PortalUserModel(
						user.getScreenName(), 
						user.getFirstName(), 
						user.getLastName(), 
						user.getEmailAddress(),
						fiscalCode,
						optionalId,
						position,
						labs
						);
				userModels.add(um);
			}
		}catch (SystemException e) {
			throw new UserManagementSystemException("Error retrieving list of users ", e);
		} catch (PortalException e) {
			e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		}
		return userModels;
	}

	/**
	 * return the companyId
	 * @param webId .
	 * @return the company bean
	 * @throws PortalException .
	 * @throws SystemException .
	 */
	public static Company getCompany() throws PortalException, SystemException {
		return CompanyLocalServiceUtil.getCompanyByWebId(getDefaultCompanyWebId());
	}
	/**
	 * 
	 * @return the default company web-id (e.g. iMarine.eu)
	 */
	public static String getDefaultCompanyWebId() {
		String defaultWebId = "";
		try {
			defaultWebId = GetterUtil.getString(PropsUtil.get("company.default.web.id"));
		}
		catch (NullPointerException e) {
			_log.error("Cound not find property company.default.web.id in portal.ext file returning default web id: " + DEFAULT_COMPANY_WEB_ID);
			return DEFAULT_COMPANY_WEB_ID;
		}
		return defaultWebId;
	}
	/**
	 * read the root VO name from a property file and returns it
	 */
	protected static String getRootOrganizationName() {
		//get the portles to look for from the property file
		Properties props = new Properties();
		String toReturn = "";

		try {
			String propertyfile = LiferaySiManager.getTomcatFolder()+"conf/gcube-data.properties";			
			File propsFile = new File(propertyfile);
			FileInputStream fis = new FileInputStream(propsFile);
			props.load( fis);
			toReturn = props.getProperty(ROOT_ORG);
		}
		//catch exception in case properties file does not exist
		catch(IOException e) {
			toReturn = "gcube";
			_log.error("gcube-data.properties file not found under $CATALINA_HOME/conf dir, returning default VO Name " + toReturn);
			return toReturn;
		}
		_log.debug("Returning Root VO Name: " + toReturn );
		return toReturn;
	}
	/**
	 * read the root Default Community name from a property file and returns it
	 */
	protected static String getDefaultCommunityName() {
		//get the portles to look for from the property file
		Properties props = new Properties();
		String toReturn = "";

		try {
			String propertyfile = LiferaySiManager.getTomcatFolder()+"conf/gcube-data.properties";			
			File propsFile = new File(propertyfile);
			FileInputStream fis = new FileInputStream(propsFile);
			props.load( fis);
			toReturn = props.getProperty(DEFAULT_COMMUNITY_PROP_NAME);
		}
		//catch exception in case properties file does not exist
		catch(IOException e) {
			toReturn = "Data e-Infrastructure gateway";
			_log.error("gcube-data.properties file not found under $CATALINA_HOME/conf dir, returning default Community Name " + toReturn);
			return toReturn;
		}
		_log.debug("Returning Default Community Name: " + toReturn );
		return toReturn;
	}

	public void updateGroups() throws Exception {
		LiferaySiManager.log2SIStart("anagrafico,gruppo_pers");
		_log.info("log2SIStart = OK");
		
		String dbUrl = "jdbc:postgresql://" + LiferaySiManager.getSiTestEndPoint().getHostName()+":"+LiferaySiManager.getSiTestEndPoint().getPort()+"/"+LiferaySiManager.getSiTestEndPoint().getDbName();
		String dbUser = LiferaySiManager.getSiTestEndPoint().getUser();
		String dbPassword = LiferaySiManager.getSiTestEndPoint().getPassword();
		Connection con = null;
		try {
			List<PortalUserModel> currUsers = listPortalUsers();
			_log.info("currUsers size = "  + currUsers.size());

			con = DriverManager.getConnection(dbUrl, dbUser, dbPassword);
			_log.info("getConnection to SI OK");
			String select = 
					"SELECT anagrafico.codicefiscale, anagrafico.cognome, anagrafico.nome, gruppo, gruppo_pers.al "
							+ "FROM gruppo_pers, anagrafico "
							+ "WHERE gruppo_pers.cf = anagrafico.codicefiscale and anagrafico.codicefiscale = ? "
							+ "GROUP BY anagrafico.cognome, anagrafico.nome, gruppo_pers.gruppo, anagrafico.codicefiscale, gruppo_pers.al";

			PreparedStatement statement = con.prepareStatement(select);

			for (PortalUserModel user : currUsers) {
				System.out.println("->" + user.getUsername());
				if (user.getFiscalCode() != null) {
					statement.setString(1, user.getFiscalCode());
					ResultSet rs = statement.executeQuery();
					Set<String> set = new HashSet<String>();
					//add the groups to the set
					while (rs.next()) {
						set.add(rs.getString(4));				
					}
					String groups = set.toString();
					groups = groups.substring(1, groups.length()-1);
					user.setGroups(groups);
					boolean result = updateUser(user);
					_log.info("Updated user " + user.getUsername() + "=" + result);
				}
			}
			con.close();

			LiferaySiManager.log2SIFinish("anagrafico,gruppo_pers");
			_log.info("log2SIFinish = OK");
		} catch (SQLException e) {
			e.printStackTrace();
		} finally {
			con.close();
		}
	}
}
