package org.gcube.personalization.profileadministration.impl;


import java.io.File;
import java.io.InputStream;
import java.io.StringReader;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.apache.axis.message.addressing.EndpointReferenceType;
import org.apache.commons.io.IOUtils;
import org.gcube.common.core.contexts.GCUBERemotePortTypeContext;
import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.faults.GCUBERetrySameException;
import org.gcube.common.core.faults.GCUBERetrySameFault;
import org.gcube.common.core.faults.GCUBEUnrecoverableException;
import org.gcube.common.core.faults.GCUBEUnrecoverableFault;
import org.gcube.common.core.informationsystem.client.AtomicCondition;
import org.gcube.common.core.informationsystem.client.ISClient;
import org.gcube.common.core.informationsystem.client.RPDocument;
import org.gcube.common.core.informationsystem.client.queries.GCUBEGenericResourceQuery;
import org.gcube.common.core.informationsystem.client.queries.GCUBERIQuery;
import org.gcube.common.core.informationsystem.client.queries.WSResourceQuery;
import org.gcube.common.core.informationsystem.publisher.ISPublisher;
import org.gcube.common.core.porttypes.GCUBEPortType;
import org.gcube.common.core.resources.GCUBEGenericResource;
import org.gcube.common.core.resources.GCUBERunningInstance;
import org.gcube.common.core.types.VOID;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.contentmanagement.blobstorage.service.IClient;
import org.gcube.contentmanager.storageclient.wrapper.AccessType;
import org.gcube.contentmanager.storageclient.wrapper.StorageClient;
import org.gcube.personalization.profileadministration.stubs.ValidateProfile;
import org.gcube.personalization.userprofileaccess.stubs.UserProfileAccessFactoryPortType;
import org.gcube.personalization.userprofileaccess.stubs.UserProfileAccessPortType;
import org.gcube.personalization.userprofileaccess.stubs.service.UserProfileAccessFactoryServiceAddressingLocator;
import org.gcube.personalization.userprofileaccess.stubs.service.UserProfileAccessServiceAddressingLocator;
import org.oasis.wsrf.lifetime.Destroy;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

/**
 * 
 * @author Panagiota Koltsida, NKUA
 *
 */
public class ProfileAdministrationService extends GCUBEPortType{

	static GCUBELog logger = new GCUBELog(ProfileAdministrationService.class);
	static ISClient client = null;

	private static final String serviceName = "UserProfileAccess";
	private static final String serviceClass = "Personalisation";
	private static final String directory = "/userprofiles/";

	private final static String userProfileIDProperty = "UserProfileID";

	/**
	 * The default constructor
	 * 
	 * @throws GCUBEFault Failed to contact the IS client
	 */
	public ProfileAdministrationService() throws GCUBEFault {
		try {
			client =GHNContext.getImplementation(ISClient.class);
		} catch (Exception e) {
			throw new GCUBERetrySameFault("Failes to call the IS client");
		}
	}

	/** {@inheritDoc} */
	protected GCUBEServiceContext getServiceContext() {return ServiceContext.getContext();}


	/**
	 * This method is used to find an address of a running instance of
	 * the user profile access service.
	 * 
	 * @return the port type of the RI
	 * @throws GCUBEFault an error occurred
	 */
	private UserProfileAccessFactoryPortType getUserProfileAccessServiceAddress() throws GCUBEFault{
		try {
			List<GCUBERunningInstance> result = null;
			GCUBERIQuery query = client.getQuery(GCUBERIQuery.class);
			query.addAtomicConditions(new AtomicCondition("//ServiceName","UserProfileAccess"),
					new AtomicCondition("//ServiceClass","Personalisation"));
			result = client.execute(query, ServiceContext.getContext().getScope());
			if (result==null || result.size()==0)
				throw new GCUBERetrySameException("Failed to get address of the running instance of the service UserProfileAccess.");
			EndpointReferenceType endpoint = result.get(new Random().nextInt(result.size())).getAccessPoint().getEndpoint("gcube/personalization/userprofileaccess/UserProfileAccessFactory");
			logger.debug("UPA RI endpoint is to be used: '" + endpoint.toString() + "'");
			UserProfileAccessFactoryPortType upa = null;
			UserProfileAccessFactoryServiceAddressingLocator upalocator = new UserProfileAccessFactoryServiceAddressingLocator();
			upa = upalocator.getUserProfileAccessFactoryPortTypePort(endpoint);
			upa = GCUBERemotePortTypeContext.getProxy(upa,ServiceContext.getContext().getScope(), ServiceContext.getContext());
			return upa;
		} catch (Exception e) {
			logger.error("Failed to contact UserProfileAccess service. Could not get a running instance.",e);
			throw new GCUBEUnrecoverableFault("Failed to contact UserProfileAccess service. Could not get a running instance.");
		}
	}

	/**
	 * Returns the EPRs of the resources of the service UserProfileAccess with a specific
	 * resource property
	 * 
	 * @param username: the value of the resource property
	 * @return an array which holds the EPRs
	 * @throws GCUBEFault an error occurred
	 */
	private List<EndpointReferenceType> getResourceEpr(String username) throws  GCUBEFault{
		List<EndpointReferenceType> result = new LinkedList<EndpointReferenceType>();
		try {
			AtomicCondition cond = new AtomicCondition("/../SourceKey", username+"_"+ServiceContext.getContext().getScope().getName());
			WSResourceQuery gquery = client.getQuery(WSResourceQuery.class);
			gquery.addAtomicConditions(cond);
			gquery.addGenericCondition("$result//Data/child::*[local-name()='ServiceName']/string() eq 'UserProfileAccess'");
			for(RPDocument d : client.execute(gquery, ServiceContext.getContext().getScope()))
				result.add(d.getEndpoint());
			if (result == null || result.size() == 0) {
				logger.info("There is no WS-Resource for the requested profile with username: " + username);
			}
		}catch(Exception e){
			logger.error("Failed to get the WS-Resources",e);
			throw new GCUBEUnrecoverableFault("Failed to get the WS-Resources.");
		}
		return result;
	}



	/**
	 * This method gets the ID of the profile, which is saved as a resource property
	 * in a resource of the service UserProfileAccess.
	 * 
	 * @param epr the Epr of the resource
	 * @return the unique Id of the user's profile stored in CMS
	 * @throws GCUBEFault an error occurred
	 */
	//	private String getProfileId(EndpointReferenceType epr) throws  GCUBEFault{
	//		/* Gets the value of the Resource property: "UserProfileId" */
	//		try {
	//			WSResourceQuery gquery = client.getQuery(WSResourceQuery.class);
	//			gquery.addAtomicConditions(
	//					new AtomicCondition("/../Source", epr.getAddress().toString()),
	//					new AtomicCondition("/../SourceKey", epr.getProperties().get_any()[0].getValue())
	//			);              
	//			for(RPDocument d : client.execute(gquery, ServiceContext.getContext().getScope())){
	//				return d.evaluate("//" + userProfileIDProperty + "/text()").get(0);
	//			}
	//		}catch(Exception e) {
	//			logger.error("Failed to get the requested property.",e);
	//			throw new GCUBEUnrecoverableFault("Failed to get the requested property.");
	//		}
	//		return null;
	//	}

	/**
	 * 
	 * @param userProfileId The unique ID of the user's profile
	 * @param cms The port type of the Content Management Service 
	 * @return A String that contains the user's profile.
	 * @throws GCUBEFault an error occurred
	 */
	private String getProfile(String profileName) throws  GCUBEFault{
		String ret = null;
		try {
			IClient client = new StorageClient(serviceClass, serviceName, serviceName, AccessType.SHARED).getClient();
			InputStream in = client.get().RFileAsInputStream(directory + profileName + ".xml");
			ret = IOUtils.toString(in, "UTF-8");
		}catch (Exception e) {
			logger.error("Failed to get the requested profile with name: " + profileName);
			throw new GCUBEUnrecoverableFault("Failed to get the requested profile with name: " + profileName);
		}
		return ret;
	}


	/**
	 * This method gets the ID of the default user profile which is stored to IS
	 * 
	 * @param name The name of the default user profile
	 * @return The GCUBEResource or null if there is no default profile
	 * @throws GCUBEFault an error occurred
	 */
	private GCUBEGenericResource getDefaultProfileGenericResource(String name) throws GCUBEFault{
		try{
			GCUBEGenericResourceQuery query = client.getQuery(GCUBEGenericResourceQuery.class);
			query.addAtomicConditions(new AtomicCondition("/Profile/Name", name));
			List<GCUBEGenericResource> result = client.execute(query, ServiceContext.getContext().getScope());
			if (result==null || result.size()==0) {
				logger.info("The resource: DefaultUserProfile is null");
				return null;
			}
			logger.info("The ID of the generic resource DefaultUserProfile is: " + result.get(0).getID());
			return result.get(0);
		}catch (Exception e) {
			logger.error("Error while trying to retrieve the ID of the default user profile",e);
			throw new GCUBEUnrecoverableFault("Error while trying to retrieve the ID of the default user profile");
		}
	}


	/*************************************************************************************************************************************
	 * 													PUBLIC METHODS
	 *************************************************************************************************************************************/

	/**
	 * This method sets the default user profile as a generic resource to the IS.
	 * 
	 * @param profile The default profile
	 * @return No Return type
	 * @throws GCUBEFault an error occurred
	 */
	public VOID setDefaultProfile(String profile) throws GCUBEFault{
		String name = "DefaultUserProfile";
		String desc = "This is the default user profile";
		String secondaryType = "UserProfile";

		ISPublisher publisher;
		GCUBEGenericResource resource;
		try {
			publisher = GHNContext.getImplementation(ISPublisher.class);
		} catch (Exception e1) {
			logger.error("Failes to call the IS publisher. Retrying.", e1);
			throw new GCUBERetrySameFault("Failes to call the IS publisher.");
		}

		resource = getDefaultProfileGenericResource(name);
		/* Default profile already exists */
		if (resource != null) {
			logger.info("DefaultUserProfile generic resource already exists. It will be updated");
			try {
				resource.setBody(profile);
				publisher.updateGCUBEResource(resource, ServiceContext.getContext().getScope(),ServiceContext.getContext());
				logger.info("DefaultUserProfile generic resource updated succesfully.");
			} catch (Exception e) {
				logger.error("Failed to update the Default user profile.",e);
				throw new GCUBERetrySameFault("Failed to update the Default user profile.");
			}
		}
		/* Default profile does not exist */
		else {
			logger.info("DefaultUserProfile generic resource does not exist. A new one will be created");
			try {
				resource = GHNContext.getImplementation(GCUBEGenericResource.class);
			} catch (Exception e1) {
				logger.error("Failed to call the GCUBEResource client. Retrying.", e1);
				throw new GCUBERetrySameFault("Failed to call the GCUBEResource client");
			}
			try {
				resource.setName(name);
				resource.setDescription(desc);
				resource.setSecondaryType(secondaryType);
				resource.setBody(profile);
				resource.setID("");
				publisher.registerGCUBEResource(resource, ServiceContext.getContext().getScope(), ServiceContext.getContext());
				logger.info("DefaultUserProfile generic resource created succesfully.");
			} catch (Exception e) {
				logger.error("Failed to create the Default user profile.",e);
				throw new GCUBERetrySameFault("Failed to create the Default user profile.");
			}
		}
		return (new VOID());
	}

	/**
	 * This method creates a new user profile.
	 * 
	 * @param userName It is used as the logical name of the document with the profile
	 * @return no Return type
	 * @throws GCUBEFault an error occurred
	 */
	public VOID createUserProfile(String username) throws  GCUBEFault{
		UserProfileAccessFactoryPortType port = getUserProfileAccessServiceAddress();
		try {
			port.createResource(username);
			logger.info("Profile for the user: '" + username + "' created succesfully.");
		} catch (Exception e) {
			logger.error("Failed to create profile for the user: '" + username + "'.",e);
			throw new GCUBERetrySameFault("Failed to create profile for the user: " + username + ".");
		}
		return (new VOID());
	}

	/**
	 * This method deletes the user's profile. The username is used to locate the profile.
	 * 
	 * 
	 * @param username The logical name of the document
	 * @return no Return type
	 * @throws GCUBEFault an error occurred
	 */
	public VOID dropUserProfile(String username) throws  GCUBEFault{

		/* Find the epr for all resources with property 'username' */
		List<EndpointReferenceType> eprs = getResourceEpr(username);
		logger.info("Number of WS resources for the user: '" + username + "' is: " + eprs.size());
		if (eprs != null && eprs.size() > 0) {
			String sourceKey = eprs.get(0).getProperties().get_any()[0].getValue();
			try {
				IClient client = new StorageClient(serviceClass, serviceName, serviceName, AccessType.SHARED).getClient();
				client.remove().RFile(directory + sourceKey + ".xml"); 
			} catch (Exception e1) {
				logger.error("Failed to delete the user's profile from SMS. Document with the specified ID does not exist on SMS", e1);
				throw new GCUBEUnrecoverableFault("Failed to delete the user's profile. " + e1.getMessage());
			}
			for (int i=0; i<eprs.size(); i++) {
				try {
					UserProfileAccessServiceAddressingLocator instanceLocator = new UserProfileAccessServiceAddressingLocator();
					UserProfileAccessPortType port;
					port = instanceLocator.getUserProfileAccessPortTypePort(eprs.get(i));
					port = GCUBERemotePortTypeContext.getProxy(port,ServiceContext.getContext().getScope(), ServiceContext.getContext());
					/* Destroys the resource */
					/* The removal from IS is done automatically, after the destroy of the resource */
					// It un-publishes the resource only from the specific scope.
					port.destroy(new Destroy());
					logger.info("WS-resource for the user: '" + username + "' deleted succesfully.");
				} catch (Exception e) {
					logger.error("Failed to delete user's profile with username: " + username,e);
					throw new GCUBEUnrecoverableFault("Failed to delete user's profile with username: " + username);
				}
			}
		}
		return (new VOID());
	}

	/**
	 * This method validates the default user profile with the given schema
	 * 
	 * @param userName It is used to find the resource of the user's profile.
	 * @param validationSchema The path where the schema file is stored.
	 * @return no return type
	 * @throws GCUBEFault an error occurred
	 */
	public VOID validateProfile(ValidateProfile params) throws  GCUBEFault{
		List<EndpointReferenceType> eprs = getResourceEpr(params.getUserName());
		String profile = getProfile(eprs.get(0).getProperties().get_any()[0].getValue());

		DocumentBuilder builder = null;
		Document docTP = null;
		DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
		SchemaFactory sFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
		try 
		{
			builder = dbFactory.newDocumentBuilder();

			/* Parse and validate the profile's schema */
			docTP = builder.parse(new InputSource(new StringReader(profile)));
			Source schemaFile = new StreamSource(new File(params.getValidationSchema()));
			Schema schema = sFactory.newSchema(schemaFile);
			Validator validator = schema.newValidator();
			validator.validate(new DOMSource(docTP));
		} catch (Exception e) {
			logger.error("Failed to validate user's profile.",e);
			throw new GCUBEUnrecoverableFault("Failed to validate user's profile.");
		}

		return (new VOID());
	}
}


