package org.gcube.informationsystem.resourceregistry.utils;

import java.util.HashMap;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.base.reference.properties.PropertyElement;
import org.gcube.informationsystem.model.reference.entities.Entity;
import org.gcube.informationsystem.model.reference.properties.Header;
import org.gcube.informationsystem.model.reference.relations.Relation;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.contexts.ContextUtility;
import org.gcube.informationsystem.resourceregistry.contexts.security.AdminSecurityContext;
import org.gcube.informationsystem.resourceregistry.contexts.security.SecurityContext.PermissionMode;
import org.gcube.informationsystem.resourceregistry.instances.base.ElementManagementUtility;
import org.gcube.informationsystem.utils.ElementMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.orientechnologies.orient.core.db.document.ODatabaseDocument;
import com.orientechnologies.orient.core.record.OEdge;
import com.orientechnologies.orient.core.record.OElement;
import com.orientechnologies.orient.core.record.ORecord;
import com.orientechnologies.orient.core.record.OVertex;
import com.orientechnologies.orient.core.record.impl.ODocument;
import com.orientechnologies.orient.core.sql.executor.OResult;
import com.orientechnologies.orient.core.sql.executor.OResultSet;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class Utility {
	
	private static final Logger logger = LoggerFactory.getLogger(Utility.class);
	
	public static final String SHOULD_NOT_OCCUR_ERROR_MESSAGE = "This is really strange and should not occur. Please contact the system administrator.";
	
	public static JsonNode toJsonNode(OElement element, boolean raw) throws ResourceRegistryException {
		ORecord oRecord = element.getRecord();
		return Utility.toJsonNode(oRecord, raw);
	}
	
	public static JsonNode toJsonNode(ORecord oRecord, boolean raw) throws ResourceRegistryException {
		try {
			ObjectMapper objectMapper = new ObjectMapper();
			return objectMapper.readTree(toJsonString(oRecord, raw));
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	public static String toJsonString(OElement element, boolean raw) {
		ORecord oRecord = element.getRecord();
		return Utility.toJsonString(oRecord, raw);
	}
	
	
	public static String toJsonString(ORecord oRecord, boolean raw) {
		if(raw) {
			return oRecord.toJSON();
		}
		return oRecord.toJSON("class");
	}
	
	public static <El extends OElement> El getElementByUUIDAsAdmin(String elementType, UUID uuid,
			Class<? extends El> clz) throws NotFoundException, ResourceRegistryException {
		ODatabaseDocument adminDatabaseDocument = null;
		ODatabaseDocument current = ContextUtility.getCurrentODatabaseDocumentFromThreadLocal();
		try {
			current = ContextUtility.getCurrentODatabaseDocumentFromThreadLocal();
			AdminSecurityContext adminSecurityContext = ContextUtility.getAdminSecurityContext();
			adminDatabaseDocument = adminSecurityContext.getDatabaseDocument(PermissionMode.READER);
			return Utility.getElementByUUID(adminDatabaseDocument, elementType, uuid, clz);
		} finally {
			if(adminDatabaseDocument != null) {
				adminDatabaseDocument.close();
			}
			if(current!=null) {
				current.activateOnCurrentThread();
			}
		}
	}
	
	public static <El extends OElement> El getElementByUUID(ODatabaseDocument oDatabaseDocument, String elementType, UUID uuid,
			Class<? extends El> clz) throws NotFoundException, ResourceRegistryException {
		
		if(elementType == null || elementType.compareTo("") == 0) {
			if(OVertex.class.isAssignableFrom(clz)) {
				elementType = Entity.NAME;
			}
			if(OEdge.class.isAssignableFrom(clz)) {
				elementType = Relation.NAME;
			}
		}
		
		// TODO Rewrite using Gremlin
		String select = "SELECT FROM " + elementType + " WHERE " + IdentifiableElement.HEADER_PROPERTY + "." + Header.UUID_PROPERTY
				+ " = \"" + uuid.toString() + "\"";
		
		OResultSet resultSet = oDatabaseDocument.query(select, new HashMap<>());
		
		
		if(resultSet == null || !resultSet.hasNext()) {
			String error = String.format("No %s with UUID %s was found", elementType, uuid.toString());
			logger.debug(error);
			throw new NotFoundException(error);
		}
		
		OResult oResult = resultSet.next();
		@SuppressWarnings("unchecked")
		El element = (El) ElementManagementUtility.getElementFromOptional(oResult.getElement());
		
		logger.trace("{} with id {} is : {}", elementType, uuid.toString(), Utility.toJsonString(element, true));
		
		if(resultSet.hasNext()) {
			throw new ResourceRegistryException("Found more than one " + elementType + " with uuid " + uuid.toString()
					+ ". This is a fatal error please contact Admnistrator");
		}
		
		return element;
	}
	
	public static <P extends PropertyElement> P getPropertyDocument(Class<P> clz, OElement element, String property)
			throws ResourceRegistryException {
		try {
			ODocument oDocument = element.getProperty(property);
			P e = ElementMapper.unmarshal(clz, oDocument.toJSON());
			return e;
		} catch(Exception ex) {
			String error = String.format("Error while getting %s from %s", property, toJsonString(element, true));
			throw new ResourceRegistryException(error, ex);
		}
	}
	
	public static UUID getUUID(OElement element) throws ResourceRegistryException {
		/*
		 * ODocument header = element.getProperty(Entity.HEADER_PROPERTY); String
		 * contextID = header.field(Header.UUID_PROPERTY); return
		 * UUID.fromString(contextID);
		 */
		Header header = HeaderUtility.getHeader(element);
		return header.getUUID();
	}
		
}