package org.gcube.informationsystem.resourceregistry.instances.model.entities;

import java.util.Iterator;

import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceAvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceNotFoundException;
import org.gcube.informationsystem.resourceregistry.contexts.security.SecurityContext;
import org.gcube.informationsystem.resourceregistry.contexts.security.SecurityContext.PermissionMode;
import org.gcube.informationsystem.resourceregistry.instances.model.relations.ConsistsOfManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.relations.IsRelatedToManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.relations.RelationManagement;
import org.gcube.informationsystem.resourceregistry.utils.Utility;

import com.orientechnologies.orient.core.db.document.ODatabaseDocument;
import com.orientechnologies.orient.core.metadata.schema.OClass;
import com.orientechnologies.orient.core.record.ODirection;
import com.orientechnologies.orient.core.record.OEdge;
import com.orientechnologies.orient.core.record.OVertex;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ResourceManagement extends EntityManagement<Resource> {
	
	public ResourceManagement() {
		super(AccessType.RESOURCE);
	}
	
	public ResourceManagement(SecurityContext workingContext, ODatabaseDocument oDatabaseDocument) {
		super(AccessType.RESOURCE, workingContext, oDatabaseDocument);
	}
	
	@Override
	protected ResourceNotFoundException getSpecificElementNotFoundException(NotFoundException e) {
		return new ResourceNotFoundException(e.getMessage(), e.getCause());
	}
	
	@Override
	protected ResourceAvailableInAnotherContextException getSpecificERAvailableInAnotherContextException(
			String message) {
		return new ResourceAvailableInAnotherContextException(message);
	}
	
	@Override
	protected ResourceAlreadyPresentException getSpecificERAlreadyPresentException(String message) {
		return new ResourceAlreadyPresentException(message);
	}
	
	@Override
	public String serialize() throws ResourceRegistryException {
		return serializeAsJson().toString();
	}

	@Override
	public JsonNode serializeAsJson() throws ResourceRegistryException {
		
		JsonNode sourceResource = serializeSelfOnly();
		
		/*
		 * Cannot get ConsistsOf edge only because is not polymorphic for a
		 * com.tinkerpop.blueprints.Vertex vertex.getEdges(Direction.OUT,
		 * ConsistsOf.NAME); TODO Looks for a different query
		 */
		
		Iterable<OEdge> edges = getElement().getEdges(ODirection.OUT);
		for(OEdge edge : edges) {
			
			@SuppressWarnings("rawtypes")
			RelationManagement relationManagement = getRelationManagement(edge);
			relationManagement.setReload(reload);
			
			if(relationManagement.giveMeSourceEntityManagementAsIs() == null) {
				relationManagement.setSourceEntityManagement(this);
			}
			
			if(relationManagement.giveMeSourceEntityManagementAsIs() != this) {
				StringBuilder errorMessage = new StringBuilder();
				errorMessage.append("SourceEntityManagement for ");
				errorMessage.append(relationManagement.getClass().getSimpleName());
				errorMessage.append(" is not the one expected. ");
				errorMessage.append(Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
				throw new ResourceRegistryException(errorMessage.toString());
			}
			
			if(relationManagement instanceof ConsistsOfManagement) {
				try {
					JsonNode consistsOf = relationManagement.serializeAsJson(true, true);
					sourceResource = addConsistsOf(sourceResource, consistsOf);
				} catch(ResourceRegistryException e) {
					logger.error("Unable to correctly serialize {}. {}", edge, Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
					throw e;
				} catch(Exception e) {
					logger.error("Unable to correctly serialize {}. {}", edge, Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
					throw new ResourceRegistryException(e);
				}
				
			}
			/*
			 * This comment is just to show that IsRelatedTo is not serialized by default as
			 * design choice and not because forget
			 * 
			 * else if(orientEdgeType.isSubClassOf(IsRelatedTo.NAME)){ JsonNode
			 * isRelatedTo = relationManagement.serializeAsJson(true, true); sourceResource
			 * = addIsRelatedTo(sourceResource, isRelatedTo); }
			 */
		}
		
		return sourceResource;
	}
	
	public static JsonNode addConsistsOf(JsonNode sourceResource, JsonNode consistsOf)
			throws ResourceRegistryException {
		return addRelation(sourceResource, consistsOf, AccessType.CONSISTS_OF.lowerCaseFirstCharacter());
	}
	
	public static JsonNode addIsRelatedTo(JsonNode sourceResource, JsonNode isRelatedTo)
			throws ResourceRegistryException {
		return addRelation(sourceResource, isRelatedTo, AccessType.IS_RELATED_TO.lowerCaseFirstCharacter());
	}
	
	@Override
	protected OVertex reallyCreate() throws ResourceAlreadyPresentException, ResourceRegistryException {
		
		createVertex();
		
		String property = AccessType.CONSISTS_OF.lowerCaseFirstCharacter();
		if(jsonNode.has(property)) {
			JsonNode jsonNodeArray = jsonNode.get(property);
			for(JsonNode consistOfJsonNode : jsonNodeArray) {
				ConsistsOfManagement com = new ConsistsOfManagement(getWorkingContext(), oDatabaseDocument);
				com.setJsonNode(consistOfJsonNode);
				com.setSourceEntityManagement(this);
				com.internalCreate();
				addToRelationManagement(com);
			}
		}
		
		property = AccessType.IS_RELATED_TO.lowerCaseFirstCharacter();
		if(jsonNode.has(property)) {
			JsonNode jsonNodeArray = jsonNode.get(property);
			for(JsonNode relationJsonNode : jsonNodeArray) {
				IsRelatedToManagement irtm = new IsRelatedToManagement(getWorkingContext(), oDatabaseDocument);
				irtm.setJsonNode(relationJsonNode);
				irtm.setSourceEntityManagement(this);
				irtm.internalCreate();
				addToRelationManagement(irtm);
			}
		}
		
		return element;
	}
	
	@Override
	protected OVertex reallyUpdate() throws ResourceNotFoundException, ResourceRegistryException {
		
		getElement();
		
		String property = AccessType.CONSISTS_OF.lowerCaseFirstCharacter();
		if(jsonNode.has(property)) {
			JsonNode jsonNodeArray = jsonNode.get(property);
			for(JsonNode relationJsonNode : jsonNodeArray) {
				ConsistsOfManagement com = new ConsistsOfManagement(getWorkingContext(), oDatabaseDocument);
				com.setJsonNode(relationJsonNode);
				com.internalCreateOrUdate();
				addToRelationManagement(com);
			}
		}
		
		property = AccessType.IS_RELATED_TO.lowerCaseFirstCharacter();
		if(jsonNode.has(property)) {
			JsonNode jsonNodeArray = jsonNode.get(property);
			for(JsonNode relationJsonNode : jsonNodeArray) {
				IsRelatedToManagement irtm = new IsRelatedToManagement(getWorkingContext(), oDatabaseDocument);
				irtm.setJsonNode(relationJsonNode);
				irtm.internalUpdate();
				addToRelationManagement(irtm);
			}
		}
		
		return element;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	protected boolean reallyDelete() throws ResourceNotFoundException, ResourceRegistryException {
		// internalDeleteResource(orientGraph, uuid, null);
		
		getElement();
		
		Iterable<OEdge> iterable = element.getEdges(ODirection.OUT);
		Iterator<OEdge> iterator = iterable.iterator();
		while(iterator.hasNext()) {
			
			OEdge edge = iterator.next();
			OClass oClass = getOClass(edge);
			
			@SuppressWarnings("rawtypes")
			RelationManagement relationManagement = null;
			if(oClass.isSubClassOf(IsRelatedTo.NAME)) {
				relationManagement = new IsRelatedToManagement(getWorkingContext(), oDatabaseDocument);
			} else if(oClass.isSubClassOf(ConsistsOf.NAME)) {
				relationManagement = new ConsistsOfManagement(getWorkingContext(), oDatabaseDocument);
			} else {
				logger.warn("{} is not a {} nor a {}. {}", Utility.toJsonString(edge, true), IsRelatedTo.NAME,
						ConsistsOf.NAME, Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
			}
			if(relationManagement != null) {
				relationManagement.setElement(edge);
				relationManagement.internalDelete();
			}
			
		}
		
		element.delete();
		
		return true;
	}
	
	public String all(boolean polymorphic) throws ResourceRegistryException {
		try {
			oDatabaseDocument = getWorkingContext().getDatabaseDocument(PermissionMode.READER);
			
			return reallyGetAll(polymorphic);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		} finally {
			if(oDatabaseDocument != null) {
				oDatabaseDocument.close();
			}
		}
	}
	
}
