package org.gcube.informationsystem.resourceregistry.schema;

import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.gcube.common.gxhttp.reference.GXConnection;
import org.gcube.common.gxhttp.request.GXHTTPStringRequest;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.context.ContextAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.context.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.schema.SchemaException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.schema.SchemaNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.rest.TypePath;
import org.gcube.informationsystem.resourceregistry.api.rest.httputils.HTTPUtility;
import org.gcube.informationsystem.resourceregistry.api.utils.Utility;
import org.gcube.informationsystem.types.TypeMapper;
import org.gcube.informationsystem.types.reference.Type;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ResourceRegistrySchemaClientImpl implements ResourceRegistrySchemaClient {
	
	private static final Logger logger = LoggerFactory.getLogger(ResourceRegistrySchemaClientImpl.class);
	
	protected final String address;
	
	public ResourceRegistrySchemaClientImpl(String address) {
		this.address = address;
		
	}
	
	@Override
	public <E extends Element> Type create(Class<E> clz)
			throws SchemaException, ResourceRegistryException {
		try {
			String typeDefinition = TypeMapper.serializeType(clz);
			// String type = AccessType.getAccessType(clz).getName();
			String res = create(typeDefinition);
			return TypeMapper.deserializeTypeDefinition(res);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String create(String typeDefinitition) throws ContextAlreadyPresentException, ResourceRegistryException {
		try {
			logger.trace("Going to create: {}", typeDefinitition);
			Type typeDefinitionObj = TypeMapper.deserializeTypeDefinition(typeDefinitition);
			
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistrySchemaClient.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header("Content-type", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(TypePath.TYPES_PATH_PART);
			gxHTTPStringRequest.path(typeDefinitionObj.getName());
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.put(typeDefinitition);
			String c = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.trace("{} successfully created", c);
			return c;
			
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	
	@Override
	public <ISM extends Element> List<Type> read(Class<ISM> clz, Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException {
		try {
			String typeName = Utility.getTypeName(clz);
			String res = read(typeName, polymorphic);
			return TypeMapper.deserializeTypeDefinitions(res);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String read(String typeName, Boolean polymorphic) throws ContextNotFoundException, ResourceRegistryException {
		try {
			logger.info("Going to get {} schema", typeName);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistrySchemaClient.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(TypePath.TYPES_PATH_PART);
			gxHTTPStringRequest.path(typeName);
			
			Map<String,String> parameters = new HashMap<>();
			if(polymorphic != null) {
				parameters.put(TypePath.POLYMORPHIC_PARAM, polymorphic.toString());
			}
			gxHTTPStringRequest.queryParams(parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String json = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got schema for {} is {}", typeName, json);
			return json;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
}
