package org.gcube.informationsystem.resourceregistry.queries.templates;

import java.util.List;

import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.informationsystem.model.reference.ERElement;
import org.gcube.informationsystem.queries.templates.reference.entities.QueryTemplate;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.queries.templates.QueryTemplateAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.queries.templates.QueryTemplateNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.request.RequestInfo;

/**
 * Client interface for managing query templates in the Resource Registry system.
 * 
 * <strong>Purpose:</strong>
 * This interface provides methods for creating, reading, updating, deleting, and executing query templates
 * within the Resource Registry. Query templates allow users to define reusable parameterized queries
 * for complex data retrieval operations on Information System model instances.
 * 
 * <strong>Instantiation:</strong>
 * Clients should be created using the factory pattern for proper configuration:
 * <pre>
 * // Recommended way to create a client
 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
 * </pre>
 * 
 * <strong>Supported Operations:</strong>
 * <ul>
 * <li><strong>CRUD Operations</strong>: Create, Read, Update, Delete query templates with full lifecycle management</li>
 * <li><strong>Template Execution</strong>: Execute parameterized queries with variable substitution</li>
 * <li><strong>List Operations</strong>: Retrieve all available query templates with filtering and pagination</li>
 * <li><strong>Validation</strong>: Validate template syntax and parameter bindings before execution.</li>
 * </ul>
 * 
 * <strong>Query Templates:</strong>
 * Templates are parameterized queries that can be reused with different parameter values:
 * <ul>
 * <li>Support for variable substitution using placeholder syntax</li>
 * <li>Type-safe parameter binding and validation</li>
 * <li>Complex queries spanning multiple entity and relation types</li>
 * <li>Result filtering and transformation capabilities.</li>
 * </ul>
 * 
 * <strong>Query Parameter Configuration:</strong>
 * Most methods support optional query parameters that can be configured via the client configuration:
 * <ul>
 * <li><strong>includeMeta</strong>: Include metadata in template and result responses</li>
 * <li><strong>limit</strong>: Maximum number of templates or results per request</li>
 * <li><strong>offset</strong>: Number of items to skip for pagination.</li>
 * </ul>
 * 
 * <strong>Authorization:</strong>
 * All operations respect authorization context derived from the current authentication token.
 * Users can only access and execute templates within their authorized contexts and see results
 * appropriate for their role and context access permissions.
 * 
 * <strong>Context Support:</strong>
 * Query template execution automatically operates within the context determined by the authorization token.
 * Templates can be designed to work across multiple contexts for users with appropriate permissions.
 * 
 * @author Luca Frosini (ISTI - CNR)
 */
public interface ResourceRegistryQueryTemplateClient extends RequestInfo {

	/**
	 * Adds a custom HTTP header to be included in requests.
	 * 
	 * @param name Header name
	 * @param value Header value
	 */
	public void addHeader(String name, String value);
	
	/**
	 * Adds a custom HTTP header to be included in requests.
	 * 
	 * @param name Header name
	 * @param value boolean value
	 */
	public void addHeader(String name, boolean value);
	
	/**
	 * Retrieves all query templates from the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /query-templates[?limit={number}&offset={number}&includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Returns all query templates accessible to the current user</li>
	 * <li>Results are filtered based on user authorization</li>
	 * <li>Provides pagination support via limit and offset parameters</li>
	 * <li>Can include metadata with role-based filtering when requested.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query templates successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query parameters (e.g., invalid limit/offset values)</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to access query templates</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of query templates to return in a single response</li>
	 * <li>Default value: 10</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 query templates per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of query templates to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 100 to skip the first 100 query templates (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response query templates</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>No input parameters required for this operation</li>
	 * <li>All available query templates are retrieved based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Access level depends on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to all query templates with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to all query templates with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Get all query templates with default pagination
	 * List&lt;QueryTemplate&gt; templates = client.all();
	 * 
	 * // Get query templates with custom pagination and metadata
	 * client.setLimit(20);
	 * client.setOffset(40);
	 * client.setIncludeMeta(true);
	 * List&lt;QueryTemplate&gt; templatesWithMeta = client.all();
	 * </pre>
	 * 
	 * @return List of all query templates accessible to the current user
	 * @throws ResourceRegistryException If an error occurs during query template retrieval
	 */
	public List<QueryTemplate> all() throws ResourceRegistryException;
	
	/**
	 * Creates a new query template in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /query-templates/{query-template-name}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new query template if no template with the same name exists</li>
	 * <li>If a template with the same name already exists, throws {@link QueryTemplateAlreadyPresentException}</li>
	 * <li>Validates the query template definition including query syntax and parameter declarations</li>
	 * <li>Performs automatic query validation using default parameter values</li>
	 * <li>Returns the created template with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: Query template successfully created</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query template definition, malformed JSON, or query validation failure</li>
	 * <li><strong>409 Conflict</strong>: Query template with the same name already exists</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to create query templates</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response query template</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The query template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Query validation is performed using default values from templateVariables</li>
	 * <li>Every parameter used in the query (prefixed with $) must have a corresponding declaration in templateVariables</li>
	 * <li>Default values must be valid for query execution.</li>
	 * </ul>
	 *
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>Query template structure is validated against the QueryTemplate model</li>
	 * <li>Query syntax is validated by attempting execution with default parameter values</li>
	 * <li>The query must execute without errors (empty results are acceptable)</li>
	 * <li>Parameter declarations must match parameters used in the query.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global and not bound to a specific context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to create any query template with complete, unfiltered metadata in response</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to create any query template with complete, unfiltered metadata in response</li>
	 * <li><strong>All Other Users</strong>: Limited create permissions based on context access; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Create a new query template
	 * QueryTemplate template = new QueryTemplate();
	 * template.setName("GetAllEServiceWithState");
	 * template.setDescription("Retrieves all EService instances with a specific state");
	 * template.setQuery("SELECT FROM EService WHERE state = $state");
	 * // ... set templateVariables and other properties
	 * QueryTemplate createdTemplate = client.create(template);
	 * </pre>
	 * 
	 * @param queryTemplate The query template to create in the Resource Registry
	 * @return The created query template with server-generated metadata
	 * @throws QueryTemplateAlreadyPresentException If a query template with the same name already exists
	 * @throws ResourceRegistryException If an error occurs during query template creation or validation
	 */
	public QueryTemplate create(QueryTemplate queryTemplate) throws QueryTemplateAlreadyPresentException, ResourceRegistryException;

	/**
	 * Creates a new query template from JSON representation in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /query-templates/{query-template-name}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new query template from JSON if no template with the same name exists</li>
	 * <li>If a template with the same name already exists, throws {@link QueryTemplateAlreadyPresentException}</li>
	 * <li>Validates the JSON against the QueryTemplate model schema</li>
	 * <li>Performs automatic query validation using default parameter values</li>
	 * <li>The template name is extracted from the JSON "name" property</li>
	 * <li>Returns the created template as JSON with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: Query template successfully created</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON data, schema validation failure, or query validation failure</li>
	 * <li><strong>409 Conflict</strong>: Query template with the same name already exists</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to create query templates</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Must contain a valid "name" property specifying the query template name</li>
	 * <li>Must conform to the QueryTemplate model JSON schema</li>
	 * <li>Must include "query" property with valid query syntax</li>
	 * <li>TemplateVariables must be properly defined for all query parameters</li>
	 * <li>All required properties for QueryTemplate must be present.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>JSON structure is validated against the QueryTemplate model</li>
	 * <li>Query syntax is validated by attempting execution with default parameter values</li>
	 * <li>The query must execute without errors (empty results are acceptable)</li>
	 * <li>Parameter declarations must match parameters used in the query.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global and not bound to a specific context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to create any query template with complete, unfiltered metadata in response</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to create any query template with complete, unfiltered metadata in response</li>
	 * <li><strong>All Other Users</strong>: Limited create permissions based on context access; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Create a new query template from JSON
	 * String templateJson = "{\"name\":\"GetAllEServiceWithState\",\"description\":\"...\",\"query\":\"SELECT FROM EService WHERE state = $state\",\"templateVariables\":{...}}";
	 * String createdTemplateJson = client.create(templateJson);
	 * </pre>
	 * 
	 * @param queryTemplate JSON representation of the query template to create
	 * @return JSON representation of the created query template with server-generated metadata
	 * @throws QueryTemplateAlreadyPresentException If a query template with the same name already exists
	 * @throws ResourceRegistryException If an error occurs during query template creation, JSON parsing, or validation
	 */
	public String create(String queryTemplate) throws QueryTemplateAlreadyPresentException, ResourceRegistryException;

	/**
	 * Checks if a query template exists in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /query-templates/{query-template-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies the existence of the specified query template using its name</li>
	 * <li>The template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Does not return the template data, only confirms existence</li>
	 * <li>Checks accessibility based on user authorization.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template exists and is accessible</li>
	 * <li><strong>404 Not Found</strong>: Query template does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Query template exists but user lacks authorization to access it</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * <ul>
	 * <li>This operation does not use query parameters for existence checking</li>
	 * <li>Existence is determined by attempting to retrieve the template without metadata.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The query template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Only the name field is used for the existence check.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Access level depends on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to check existence of any query template</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to check existence of any query template</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; can check existence of accessible templates.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Check if a query template exists
	 * QueryTemplate template = new QueryTemplate();
	 * template.setName("GetAllEServiceWithState");
	 * boolean exists = client.exist(template);
	 * </pre>
	 * 
	 * @param queryTemplate The query template to check for existence (must have a valid name)
	 * @return true if the query template exists and is accessible, false if not found
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 */
	public boolean exist(QueryTemplate queryTemplate) throws ResourceRegistryException;

	/**
	 * Checks if a query template exists in the Resource Registry using template name.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /query-templates/{query-template-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies the existence of the specified query template using its name</li>
	 * <li>Does not return the template data, only confirms existence</li>
	 * <li>Checks accessibility based on user authorization</li>
	 * <li>Most direct method when template name is already known.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template exists and is accessible</li>
	 * <li><strong>404 Not Found</strong>: Query template does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Query template exists but user lacks authorization to access it</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * <ul>
	 * <li>This operation does not use query parameters for existence checking</li>
	 * <li>Existence is determined by attempting to retrieve the template without metadata.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Template name must be a valid query template name</li>
	 * <li>Template names are case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Access level depends on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to check existence of any query template</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to check existence of any query template</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; can check existence of accessible templates.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Check if a query template exists by name
	 * boolean exists = client.exist("GetAllEServiceWithState");
	 * </pre>
	 * 
	 * @param queryTemplateName The name of the query template to check for existence
	 * @return true if the query template exists and is accessible, false if not found
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 */
	public boolean exist(String queryTemplateName) throws ResourceRegistryException;
		
	/**
	 * Reads an existing query template from the Resource Registry using a provided template object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /query-templates/{query-template-name}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing query template by extracting name from the provided template object</li>
	 * <li>The template must be accessible based on user authorization</li>
	 * <li>Returns the complete template with all properties and metadata</li>
	 * <li>The name is automatically determined from the provided template object.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No query template found with the specified name</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response query template</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The query template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Only the name field is used for template retrieval.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Access level depends on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to retrieve any query template with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to retrieve any query template with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Read a query template using template object
	 * QueryTemplate template = new QueryTemplate();
	 * template.setName("GetAllEServiceWithState");
	 * QueryTemplate readTemplate = client.read(template);
	 * </pre>
	 * 
	 * @param queryTemplate Template object containing at least the name
	 * @return The complete query template from the Resource Registry
	 * @throws QueryTemplateNotFoundException If no query template is found with the specified name
	 * @throws ResourceRegistryException If an error occurs during template retrieval
	 */
	public QueryTemplate read(QueryTemplate queryTemplate) throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Reads an existing query template from the Resource Registry by its name.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /query-templates/{query-template-name}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing query template by its name</li>
	 * <li>The template must be accessible based on user authorization</li>
	 * <li>Returns the complete template with all properties and metadata</li>
	 * <li>Most direct method when template name is already known.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No query template found with the specified name</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response query template</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The specified name must be a valid query template name</li>
	 * <li>Template names are case-sensitive and must match exactly</li>
	 * <li>Name must correspond to an existing template.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Access level depends on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to retrieve any query template with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to retrieve any query template with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Read a query template by name
	 * QueryTemplate template = client.read("GetAllEServiceWithState");
	 * </pre>
	 * 
	 * @param queryTemplateName The name of the query template to read
	 * @return The complete query template from the Resource Registry
	 * @throws QueryTemplateNotFoundException If no query template is found with the specified name
	 * @throws ResourceRegistryException If an error occurs during template retrieval
	 */
	public QueryTemplate read(String queryTemplateName) throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Reads an existing query template from the Resource Registry as JSON string.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /query-templates/{query-template-name}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing query template by its name</li>
	 * <li>The template must be accessible based on user authorization</li>
	 * <li>Returns the complete template as JSON with all properties and metadata</li>
	 * <li>Returns the response as a JSON object string.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No query template found with the specified name</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response query template</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The specified name must be a valid query template name</li>
	 * <li>Template names are case-sensitive and must match exactly</li>
	 * <li>Name must correspond to an existing template.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Access level depends on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to retrieve any query template with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to retrieve any query template with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Read a query template as JSON string
	 * String templateJson = client.readAsString("GetAllEServiceWithState");
	 * </pre>
	 * 
	 * @param queryTemplateName The name of the query template to read
	 * @return JSON representation of the complete query template from the Resource Registry
	 * @throws QueryTemplateNotFoundException If no query template is found with the specified name
	 * @throws ResourceRegistryException If an error occurs during template retrieval
	 */
	public String readAsString(String queryTemplateName) throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Updates an existing query template in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /query-templates/{query-template-name}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing query template if a template with the same name exists</li>
	 * <li>If no template with the same name exists, throws {@link QueryTemplateNotFoundException}</li>
	 * <li>Validates the updated query template definition including query syntax and parameter declarations</li>
	 * <li>Performs automatic query validation using default parameter values</li>
	 * <li>Returns the updated template with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query template definition, malformed JSON, or query validation failure</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to update query templates</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response query template</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The query template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Query validation is performed using default values from templateVariables</li>
	 * <li>Every parameter used in the query (prefixed with $) must have a corresponding declaration in templateVariables</li>
	 * <li>Default values must be valid for query execution.</li>
	 * </ul>
	 *
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>Query template structure is validated against the QueryTemplate model</li>
	 * <li>Query syntax is validated by attempting execution with default parameter values</li>
	 * <li>The query must execute without errors (empty results are acceptable)</li>
	 * <li>Parameter declarations must match parameters used in the query.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global and not bound to a specific context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to update any query template with complete, unfiltered metadata in response</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to update any query template with complete, unfiltered metadata in response</li>
	 * <li><strong>All Other Users</strong>: Limited update permissions based on context access; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global and not bound to a specific context.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Update an existing query template
	 * QueryTemplate template = client.read("GetAllEServiceWithState");
	 * template.setDescription("Updated description for EService state query");
	 * template.setQuery("SELECT FROM EService WHERE state = $state AND name LIKE $namePattern");
	 * // ... update templateVariables and other properties
	 * QueryTemplate updatedTemplate = client.update(template);
	 * </pre>
	 * 
	 * @param queryTemplate The query template to update in the Resource Registry
	 * @return The updated query template with server-generated metadata
	 * @throws QueryTemplateNotFoundException If no query template with the specified name exists
	 * @throws ResourceRegistryException If an error occurs during query template update or validation
	 */
	public QueryTemplate update(QueryTemplate queryTemplate) throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Updates an existing query template from JSON representation in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /query-templates/{query-template-name}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing query template from JSON if a template with the same name exists</li>
	 * <li>If no template with the same name exists, throws {@link QueryTemplateNotFoundException}</li>
	 * <li>Validates the JSON against the QueryTemplate model schema</li>
	 * <li>Performs automatic query validation using default parameter values</li>
	 * <li>The template name is extracted from the JSON "name" property</li>
	 * <li>Returns the updated template as JSON with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON data, schema validation failure, or query validation failure</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to update query templates</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Must contain a valid "name" property specifying the query template name</li>
	 * <li>Must conform to the QueryTemplate model JSON schema</li>
	 * <li>Must include "query" property with valid query syntax</li>
	 * <li>TemplateVariables must be properly defined for all query parameters</li>
	 * <li>All required properties for QueryTemplate must be present.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>JSON structure is validated against the QueryTemplate model</li>
	 * <li>Query syntax is validated by attempting execution with default parameter values</li>
	 * <li>The query must execute without errors (empty results are acceptable)</li>
	 * <li>Parameter declarations must match parameters used in the query.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global and not bound to a specific context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to update any query template with complete, unfiltered metadata in response</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to update any query template with complete, unfiltered metadata in response</li>
	 * <li><strong>All Other Users</strong>: Limited update permissions based on context access; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global and not bound to a specific context.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Update an existing query template from JSON
	 * String updatedTemplateJson = "{\"name\":\"GetAllEServiceWithState\",\"description\":\"Updated description\",\"query\":\"SELECT FROM EService WHERE state = $state AND name LIKE $namePattern\",\"templateVariables\":{...}}";
	 * String updatedResult = client.update(updatedTemplateJson);
	 * </pre>
	 * 
	 * @param queryTemplate JSON representation of the query template to update
	 * @return JSON representation of the updated query template with server-generated metadata
	 * @throws QueryTemplateNotFoundException If no query template with the specified name exists
	 * @throws ResourceRegistryException If an error occurs during query template update, JSON parsing, or validation
	 */
	public String update(String queryTemplate) throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Executes a query template and returns results as JSON string.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /query-templates/{query-template-name}[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes the specified query template with no parameters</li>
	 * <li>Returns query results as a JSON array string</li>
	 * <li>Uses default parameter values if the template defines any parameters</li>
	 * <li>Results are filtered based on user authorization and context access.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query execution or parameter values</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 *
	 * <strong>Query Parameters:</strong>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Execute a query template and get results as JSON
	 * String resultsJson = client.runGetString("GetAllEService");
	 * </pre>
	 * 
	 * @param name The name of the query template to execute
	 * @return JSON array string containing query results
	 * @throws QueryTemplateNotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	public String runGetString(String name) throws QueryTemplateNotFoundException, ResourceRegistryException;
	
	/**
	 * Executes a query template and returns results as a list of Information System elements.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /query-templates/{query-template-name}[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes the specified query template with no parameters</li>
	 * <li>Returns query results as a typed list of Information System elements</li>
	 * <li>Uses default parameter values if the template defines any parameters</li>
	 * <li>Results are filtered based on user authorization and context access</li>
	 * <li>Automatically deserializes JSON results to specified type.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query execution or parameter values</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Execute a query template and get typed results
	 * List&lt;EService&gt; services = client.run("GetAllEService");
	 * </pre>
	 * 
	 * @param <E> The specific type of Information System element expected in results
	 * @param name The name of the query template to execute
	 * @return List of Information System elements from query execution
	 * @throws QueryTemplateNotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	public <E extends ERElement> List<E> run(String name) throws QueryTemplateNotFoundException, ResourceRegistryException;
	
	/**
	 * Executes a query template using template object and returns results as a list of Information System elements.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /query-templates/{query-template-name}[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes the query template with no parameters</li>
	 * <li>The template name is extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Returns query results as a typed list of Information System elements</li>
	 * <li>Uses default parameter values if the template defines any parameters</li>
	 * <li>Results are filtered based on user authorization and context access</li>
	 * <li>Automatically deserializes JSON results to specified type.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query execution or parameter values</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Execute a query template using template object
	 * QueryTemplate template = new QueryTemplate();
	 * template.setName("GetAllEService");
	 * List&lt;EService&gt; services = client.run(template);
	 * </pre>
	 * 
	 * @param <E> The specific type of Information System element expected in results
	 * @param queryTemplate The query template to execute (name is extracted from this object)
	 * @return List of Information System elements from query execution
	 * @throws QueryTemplateNotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	public <E extends ERElement> List<E> run(QueryTemplate queryTemplate) throws QueryTemplateNotFoundException, ResourceRegistryException;
	
	/**
	 * Executes a query template with string parameters and returns results as JSON string.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /query-templates/{query-template-name}[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes the specified query template with provided parameter values</li>
	 * <li>Parameters are provided as a JSON string containing parameter substitutions</li>
	 * <li>Returns query results as a JSON array string</li>
	 * <li>Template variables are replaced with provided parameter values</li>
	 * <li>Results are filtered based on user authorization and context access.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query execution, parameter values, or parameter format</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Execute with string parameters
	 * String params = "{\"state\":\"ready\",\"namePattern\":\"%Service%\"}";
	 * String resultsJson = client.run("GetAllEServiceWithState", params);
	 * </pre>
	 * 
	 * @param name The name of the query template to execute
	 * @param params JSON string containing parameter values for template substitution
	 * @return JSON array string containing query results
	 * @throws QueryTemplateNotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	public String run(String name, String params) throws QueryTemplateNotFoundException, ResourceRegistryException;
	
	/**
	 * Executes a query template with JsonNode parameters and returns results as a list of Information System elements.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /query-templates/{query-template-name}[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes the specified query template with provided parameter values</li>
	 * <li>Parameters are provided as a JsonNode object containing parameter substitutions</li>
	 * <li>Returns query results as a typed list of Information System elements</li>
	 * <li>Template variables are replaced with provided parameter values</li>
	 * <li>Results are filtered based on user authorization and context access</li>
	 * <li>Automatically deserializes JSON results to specified type.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query execution, parameter values, or parameter format</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Execute with JsonNode parameters
	 * ObjectMapper mapper = new ObjectMapper();
	 * JsonNode params = mapper.createObjectNode()
	 *     .put("state", "ready")
	 *     .put("namePattern", "%Service%");
	 * List&lt;EService&gt; services = client.run("GetAllEServiceWithState", params);
	 * </pre>
	 * 
	 * @param <E> The specific type of Information System element expected in results
	 * @param name The name of the query template to execute
	 * @param jsonNode JsonNode containing parameter values for template substitution
	 * @return List of Information System elements from query execution
	 * @throws QueryTemplateNotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	public <E extends ERElement> List<E> run(String name, JsonNode jsonNode) throws QueryTemplateNotFoundException, ResourceRegistryException;
	
	/**
	 * Executes a query template using template object with JsonNode parameters and returns results as a list of Information System elements.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /query-templates/{query-template-name}[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes the query template with provided parameter values</li>
	 * <li>The template name is extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Parameters are provided as a JsonNode object containing parameter substitutions</li>
	 * <li>Returns query results as a typed list of Information System elements</li>
	 * <li>Template variables are replaced with provided parameter values</li>
	 * <li>Results are filtered based on user authorization and context access</li>
	 * <li>Automatically deserializes JSON results to specified type.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query execution, parameter values, or parameter format</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The query template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Parameter values from JsonNode are used for template variable substitution</li>
	 * <li>All template parameters must be provided in the JsonNode or have default values.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Execute using template object with JsonNode parameters
	 * QueryTemplate template = new QueryTemplate();
	 * template.setName("GetAllEServiceWithState");
	 * 
	 * ObjectMapper mapper = new ObjectMapper();
	 * JsonNode params = mapper.createObjectNode()
	 *     .put("state", "ready")
	 *     .put("namePattern", "%Service%");
	 * 
	 * List&lt;EService&gt; services = client.run(template, params);
	 * </pre>
	 * 
	 * @param <E> The specific type of Information System element expected in results
	 * @param queryTemplate The query template to execute (name is extracted from this object)
	 * @param jsonNode JsonNode containing parameter values for template substitution
	 * @return List of Information System elements from query execution
	 * @throws QueryTemplateNotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	public <E extends ERElement> List<E> run(QueryTemplate queryTemplate, JsonNode jsonNode) throws QueryTemplateNotFoundException, ResourceRegistryException;
	
	/**
	 * Deletes an existing query template from the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /query-templates/{query-template-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently deletes the specified query template using its name</li>
	 * <li>The template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>This operation is irreversible and cannot be undone</li>
	 * <li>Once deleted, the template name becomes available for reuse</li>
	 * <li>The operation is atomic - either the template is completely deleted or the operation fails.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>204 No Content</strong>: Query template successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to delete the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * <ul>
	 * <li>This endpoint does not accept any query parameters</li>
	 * <li>No filtering or metadata options are available for delete operations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The query template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Only the name field is used for template deletion.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Deletion permissions depend on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to delete any query template</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to delete any query template</li>
	 * <li><strong>All Other Users</strong>: Basic access to delete query templates; same deletion permissions as administrative users.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Delete a query template using template object
	 * QueryTemplate template = new QueryTemplate();
	 * template.setName("GetAllEServiceWithState");
	 * boolean deleted = client.delete(template);
	 * </pre>
	 * 
	 * @param queryTemplate The query template to delete (must have a valid name)
	 * @return true if the query template was successfully deleted
	 * @throws QueryTemplateNotFoundException If no query template is found with the specified name
	 * @throws ResourceRegistryException If an error occurs during template deletion
	 */
	public boolean delete(QueryTemplate queryTemplate) throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Deletes an existing query template from the Resource Registry by its name.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /query-templates/{query-template-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently deletes the specified query template using its name</li>
	 * <li>This operation is irreversible and cannot be undone</li>
	 * <li>Once deleted, the template name becomes available for reuse</li>
	 * <li>The operation is atomic - either the template is completely deleted or the operation fails</li>
	 * <li>Most direct method when template name is already known.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>204 No Content</strong>: Query template successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to delete the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * <ul>
	 * <li>This endpoint does not accept any query parameters</li>
	 * <li>No filtering or metadata options are available for delete operations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The specified name must be a valid query template name</li>
	 * <li>Template names are case-sensitive and must match exactly</li>
	 * <li>Name must correspond to an existing template.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Deletion permissions depend on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to delete any query template</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to delete any query template</li>
	 * <li><strong>All Other Users</strong>: Basic access to delete query templates; same deletion permissions as administrative users.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryQueryTemplateClient client = ResourceRegistryQueryTemplateClientFactory.create();
	 * 
	 * // Delete a query template by name
	 * boolean deleted = client.delete("GetAllEServiceWithState");
	 * </pre>
	 * 
	 * @param queryTemplateName The name of the query template to delete
	 * @return true if the query template was successfully deleted
	 * @throws QueryTemplateNotFoundException If no query template is found with the specified name
	 * @throws ResourceRegistryException If an error occurs during template deletion
	 */
	public boolean delete(String queryTemplateName) throws QueryTemplateNotFoundException, ResourceRegistryException;

}
