package org.gcube.informationsystem.resourceregistry.publisher;

import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.gcube.common.gxhttp.reference.GXConnection;
import org.gcube.common.gxhttp.request.GXHTTPStringRequest;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.context.reference.entities.Context;
import org.gcube.informationsystem.model.impl.properties.HeaderImpl;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.properties.Header;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCache;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCacheRenewal;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextUtility;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.context.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.facet.FacetAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.facet.FacetNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.rest.AccessPath;
import org.gcube.informationsystem.resourceregistry.api.rest.InstancePath;
import org.gcube.informationsystem.resourceregistry.api.rest.SharingPath;
import org.gcube.informationsystem.resourceregistry.api.rest.httputils.HTTPUtility;
import org.gcube.informationsystem.utils.ElementMapper;
import org.gcube.informationsystem.utils.Utility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ResourceRegistryPublisherImpl implements ResourceRegistryPublisher {
	
	private static final Logger logger = LoggerFactory.getLogger(ResourceRegistryPublisherImpl.class);
	
	protected final String address;
	
	private GXHTTPStringRequest includeAdditionalQueryParameters(GXHTTPStringRequest gxHTTPStringRequest) throws UnsupportedEncodingException{
		return includeAdditionalQueryParameters(gxHTTPStringRequest, null);
	}
	
	private GXHTTPStringRequest includeAdditionalQueryParameters(GXHTTPStringRequest gxHTTPStringRequest, Map<String,String> queryParams) throws UnsupportedEncodingException{
		gxHTTPStringRequest = checkHierarchicalMode(gxHTTPStringRequest, queryParams);
		return checkIncludeContextsInInstanceHeader(gxHTTPStringRequest, queryParams);
	}
	
	private GXHTTPStringRequest checkHierarchicalMode(GXHTTPStringRequest gxHTTPStringRequest, Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(ResourceRegistryPublisherFactory.isHierarchicalMode()) {
			if(queryParams==null) {
				queryParams = new HashMap<>();
			}
			queryParams.put(AccessPath.HIERARCHICAL_MODE_PARAM, Boolean.toString(true));
		}
		return gxHTTPStringRequest.queryParams(queryParams);
	}
	
	private GXHTTPStringRequest checkIncludeContextsInInstanceHeader(GXHTTPStringRequest gxHTTPStringRequest, Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(ResourceRegistryPublisherFactory.includeContextsInInstanceHeader()) {
			if(queryParams==null) {
				queryParams = new HashMap<>();
			}
			queryParams.put(AccessPath.INCLUDE_CONTEXTS_IN_HEADER_PARAM, Boolean.toString(true));
		}
		return gxHTTPStringRequest.queryParams(queryParams);
	}
	
	protected ContextCacheRenewal contextCacheRenewal = new ContextCacheRenewal() {
		
		@Override
		public List<Context> renew() throws ResourceRegistryException {
			return getAllContextFromServer();
		}
		
	};
	
	public ResourceRegistryPublisherImpl(String address) {
		this.address = address;
		ContextCache contextCache = ContextCache.getInstance();
		contextCache.setContextCacheRenewal(contextCacheRenewal);
	}
	
	public List<Context> getAllContextFromServer() throws ResourceRegistryException {
		try {
			logger.info("Going to read all {}s", Context.NAME);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.CONTEXTS_PATH_PART);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got Contexts are {}", ret);
			return ElementMapper.unmarshalList(Context.class, ret);
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {}schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public List<Context> getAllContext() throws ResourceRegistryException {
		ContextCache contextCache = ContextCache.getInstance();
		return contextCache.getContexts();
	}
	
	protected Context getContextFromServer(String id) throws ContextNotFoundException, ResourceRegistryException {
		try {
			// TODO use cache
			
			logger.info("Going to get current {} ", Context.NAME);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(id);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			Context context = HTTPUtility.getResponse(Context.class, httpURLConnection);
			
			logger.debug("Got Context is {}", ElementMapper.marshal(context));
			return context;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {}schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public Context getContext(UUID uuid) throws ContextNotFoundException, ResourceRegistryException {
		ContextCache contextCache = ContextCache.getInstance();
		Context context = ContextCache.getInstance().getContextByUUID(uuid);;
		if(context == null) {
			context = getContextFromServer(uuid.toString());
			contextCache.cleanCache();
			contextCache.refreshContextsIfNeeded();
			Context c = contextCache.getContextByUUID(context.getHeader().getUUID());
			if(c!=null){
				context = c;
			}else {
				logger.error("Context with UUID {} is {}. It is possibile to get it from the server but not from the cache. This is very strange and should not occur.", uuid, context);
			}
		}
		return context;
	}
	
	@Override
	public Context getCurrentContext() throws ContextNotFoundException, ResourceRegistryException {
		String contextFullName = ResourceRegistryPublisherFactory.getCurrentContextFullName();
		ContextCache contextCache = ContextCache.getInstance();
		UUID uuid = contextCache.getUUIDByFullName(contextFullName);
		Context context = null;
		if(uuid == null) {
			context = getContextFromServer(AccessPath.CURRENT_CONTEXT);
			contextCache.cleanCache();
			contextCache.refreshContextsIfNeeded();
			Context c = contextCache.getContextByUUID(context.getHeader().getUUID());
			if(c!=null){
				context = c;
			}else {
				logger.error("Current Context is {}. It is possibile to get it from the server but not from the cache. This is very strange and should not occur.", contextFullName);
			}
		}else {
			context = contextCache.getContextByUUID(uuid);
		}
		return context;
	}
	
	private UUID getCurrentContextUUID() throws ResourceRegistryException {
		return getCurrentContext().getHeader().getUUID();
	}
	
	protected String create(String type, String json, UUID uuid)
			throws AlreadyPresentException, ResourceRegistryException {
		try {
			logger.trace("Going to create {} : {}", type, json);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header("Content-type", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.put(json);
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.trace("{} successfully created", ret);
			return ret;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	public <IE extends IdentifiableElement> String internalCreate(IE identifiableElement) throws AlreadyPresentException, ResourceRegistryException {
		try {
			String type = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
			String json = ElementMapper.marshal(identifiableElement);
			Header header = identifiableElement.getHeader();
			if(header==null) {
				header = new HeaderImpl(UUID.randomUUID());
				identifiableElement.setHeader(header);
			}
			UUID uuid = identifiableElement.getHeader().getUUID();
			return create(type, json, uuid);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	
	@SuppressWarnings("unchecked")
	@Override
	public <IE extends IdentifiableElement> IE create(IE identifiableElement) throws AlreadyPresentException, ResourceRegistryException {
		try {
			String ret = internalCreate(identifiableElement);
			return (IE) ElementMapper.unmarshal(IdentifiableElement.class, ret);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String create(String json) throws AlreadyPresentException, ResourceRegistryException {
		try {
			IdentifiableElement e = ElementMapper.unmarshal(IdentifiableElement.class, json);
			return internalCreate(e);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean exists(Class<IE> clazz, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		String type = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(clazz);
		return exists(type, uuid);
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean exists(IE identifiableElement)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		String type = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
		UUID uuid = identifiableElement.getHeader().getUUID();
		return exists(type, uuid);
	}
	
	@Override
	public boolean exists(String type, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		try {
			logger.info("Going to check if {} with UUID {} exists", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			
			includeAdditionalQueryParameters(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.head();
			HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("{} with UUID {} exists", type, uuid);
			return true;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while checking if {} with UUID {} exists.", type, uuid,
			// e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while checking if {} with UUID {} exists.", type, uuid,
			// e);
			throw new RuntimeException(e);
		}
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <IE extends IdentifiableElement> IE read(IE identifiableElement) throws NotFoundException, ResourceRegistryException {
		try {
			String type = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
			UUID uuid = identifiableElement.getHeader().getUUID();
			String ret = read(type, uuid);
			return (IE) ElementMapper.unmarshal(IdentifiableElement.class, ret);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String read(String type, UUID uuid) throws NotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to read {} with UUID {}", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			
			includeAdditionalQueryParameters(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got {} with UUID {} is {}", type, uuid, ret);
			return ret;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	protected String update(String type, String json, UUID uuid)
			throws AlreadyPresentException, ResourceRegistryException {
		try {
			logger.trace("Going to create {} : {}", type, json);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header("Content-type", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.put(json);
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.trace("{} with UUID {} successfully created : {}", type, uuid, ret);
			return ret;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <IE extends IdentifiableElement> IE update(IE identifiableElement) throws NotFoundException, ResourceRegistryException {
		try {
			String type = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
			String json = ElementMapper.marshal(identifiableElement);
			UUID uuid = identifiableElement.getHeader().getUUID();
			String ret = update(type, json, uuid);
			return (IE) ElementMapper.unmarshal(Element.class, ret);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String update(String type, String json) throws NotFoundException, ResourceRegistryException {
		try {
			UUID uuid = Utility.getUUIDFromJSONString(json);
			return update(type, json, uuid);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String update(String json) throws NotFoundException, ResourceRegistryException {
		try {
			String type = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getClassFromJsonString(json);
			return update(type, json);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean delete(IE identifiableElement) throws NotFoundException, ResourceRegistryException {
		try {
			String type = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
			UUID uuid = identifiableElement.getHeader().getUUID();
			return delete(type, uuid);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public boolean delete(String type, UUID uuid) throws NotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to delete {} with UUID {}", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.delete();
			HTTPUtility.getResponse(String.class, httpURLConnection);
			
			boolean deleted = true;
			
			logger.info("{} with UUID {} {}", type, uuid, deleted ? " successfully deleted" : "was NOT deleted");
			return deleted;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public <F extends Facet> F createFacet(F facet) throws FacetAlreadyPresentException, ResourceRegistryException {
		return create(facet);
	}
	
	@Override
	public String createFacet(String facet) throws FacetAlreadyPresentException, ResourceRegistryException {
		return create(facet);
	}
	
	@Override
	public <F extends Facet> F readFacet(F facet) throws FacetNotFoundException, ResourceRegistryException {
		return read(facet);
	}
	
	@Override
	public String readFacet(String facetType, UUID uuid) throws FacetNotFoundException, ResourceRegistryException {
		return read(facetType, uuid);
	}
	
	@Override
	public <F extends Facet> F updateFacet(F facet) throws FacetNotFoundException, ResourceRegistryException {
		return update(facet);
	}
	
	@Override
	public String updateFacet(String facet) throws FacetNotFoundException, ResourceRegistryException {
		return update(facet);
	}
	
	@Override
	public <F extends Facet> boolean deleteFacet(F facet) throws FacetNotFoundException, ResourceRegistryException {
		return delete(facet);
	}
	
	@Override
	public boolean deleteFacet(String facetType, UUID uuid) throws FacetNotFoundException, ResourceRegistryException {
		return delete(facetType, uuid);
	}
	
	@Override
	public <R extends Resource> R createResource(R resource)
			throws ResourceAlreadyPresentException, ResourceRegistryException {
		return create(resource);
	}
	
	@Override
	public String createResource(String resource) throws ResourceAlreadyPresentException, ResourceRegistryException {
		return create(resource);
	}
	
	@Override
	public <R extends Resource> R readResource(R resource) throws ResourceNotFoundException, ResourceRegistryException {
		return read(resource);
	}
	
	@Override
	public String readResource(String resourceType, UUID uuid)
			throws ResourceNotFoundException, ResourceRegistryException {
		return read(resourceType, uuid);
	}
	
	@Override
	public <R extends Resource> R updateResource(R resource)
			throws ResourceNotFoundException, ResourceRegistryException {
		return update(resource);
	}
	
	@Override
	public String updateResource(String resource) throws ResourceNotFoundException, ResourceRegistryException {
		return update(resource);
	}
	
	@Override
	public <R extends Resource> boolean deleteResource(R resource)
			throws ResourceNotFoundException, ResourceRegistryException {
		return delete(resource);
	}
	
	@Override
	public boolean deleteResource(String resourceType, UUID uuid)
			throws ResourceNotFoundException, ResourceRegistryException {
		return delete(resourceType, uuid);
	}
	
	@Override
	public <C extends ConsistsOf<? extends Resource,? extends Facet>> C createConsistsOf(C consistsOf)
			throws NotFoundException, ResourceRegistryException {
		return create(consistsOf);
	}
	
	@Override
	public String createConsistsOf(String consistsOf) throws NotFoundException, ResourceRegistryException {
		return create(consistsOf);
	}
	
	@Override
	public <C extends ConsistsOf<? extends Resource,? extends Facet>> C readConsistsOf(C consistsOf)
			throws NotFoundException, ResourceRegistryException {
		return read(consistsOf);
	}
	
	@Override
	public String readConsistsOf(String consistsOfType, UUID uuid) throws NotFoundException, ResourceRegistryException {
		return read(consistsOfType, uuid);
	}
	
	@Override
	public <C extends ConsistsOf<? extends Resource,? extends Facet>> C updateConsistsOf(C consistsOf)
			throws NotFoundException, ResourceRegistryException {
		return update(consistsOf);
	}
	
	@Override
	public String updateConsistsOf(String consistsOf) throws NotFoundException, ResourceRegistryException {
		return update(consistsOf);
	}
	
	@Override
	public <C extends ConsistsOf<? extends Resource,? extends Facet>> boolean deleteConsistsOf(C consistsOf)
			throws ResourceRegistryException {
		return delete(consistsOf);
	}
	
	@Override
	public boolean deleteConsistsOf(String consistsOfType, UUID uuid) throws ResourceRegistryException {
		return delete(consistsOfType, uuid);
	}
	
	@Override
	public <I extends IsRelatedTo<? extends Resource,? extends Resource>> I createIsRelatedTo(I isRelatedTo)
			throws ResourceNotFoundException, ResourceRegistryException {
		return create(isRelatedTo);
	}
	
	@Override
	public String createIsRelatedTo(String isRelatedTo) throws ResourceNotFoundException, ResourceRegistryException {
		return create(isRelatedTo);
	}
	
	@Override
	public <I extends IsRelatedTo<? extends Resource,? extends Resource>> I readIsRelatedTo(I isRelatedTo)
			throws NotFoundException, ResourceRegistryException {
		return read(isRelatedTo);
	}
	
	@Override
	public String readIsRelatedTo(String isRelatedToType, UUID uuid)
			throws NotFoundException, ResourceRegistryException {
		return read(isRelatedToType, uuid);
	}
	
	@Override
	public <I extends IsRelatedTo<? extends Resource,? extends Resource>> I updateIsRelatedTo(I isRelatedTo)
			throws NotFoundException, ResourceRegistryException {
		return update(isRelatedTo);
	}
	
	@Override
	public String updateIsRelatedTo(String isRelatedTo) throws NotFoundException, ResourceRegistryException {
		return update(isRelatedTo);
	}
	
	@Override
	public <I extends IsRelatedTo<? extends Resource,? extends Resource>> boolean deleteIsRelatedTo(I isRelatedTo)
			throws ResourceRegistryException {
		return delete(isRelatedTo);
	}
	
	@Override
	public boolean deleteIsRelatedTo(String isRelatedToType, UUID uuid) throws ResourceRegistryException {
		return delete(isRelatedToType, uuid);
	}
	
	@Override
	public boolean addToContext(String type, UUID instanceUUID, UUID contextUUID)
			throws NotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to add {} with UUID {} to {} with UUID {} ", type, instanceUUID, Context.NAME,
					contextUUID);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.path(SharingPath.SHARING_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(instanceUUID.toString());
			gxHTTPStringRequest.path(SharingPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(contextUUID.toString());
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.put();
			boolean added = HTTPUtility.getResponse(Boolean.class, httpURLConnection);
			
			logger.info("{} with UUID {} {} to {} with UUID {}", type, instanceUUID,
					added ? " successfully added" : "was NOT added", Context.NAME, contextUUID);
			return added;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean addToContext(IE identifiableElement, UUID contextUUID)
			throws NotFoundException, ResourceRegistryException {
		try {
			String type = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
			UUID instanceUUID = identifiableElement.getHeader().getUUID();
			return addToContext(type, instanceUUID, contextUUID);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public boolean addToCurrentContext(String type, UUID instanceUUID)
			throws NotFoundException, ResourceRegistryException {
		UUID contextUUID = getCurrentContextUUID();
		return addToContext(type, instanceUUID, contextUUID);
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean addToCurrentContext(IE identifiableElement) throws NotFoundException, ResourceRegistryException {
		UUID contextUUID = getCurrentContextUUID();
		return addToContext(identifiableElement, contextUUID);
	}
	
	@Override
	public boolean removeFromContext(String type, UUID instanceUUID, UUID contextUUID)
			throws NotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to add {} with UUID {} to {} with UUID {} ", type, instanceUUID, Context.NAME,
					contextUUID);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.path(SharingPath.SHARING_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(instanceUUID.toString());
			gxHTTPStringRequest.path(SharingPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(contextUUID.toString());
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.delete();
			boolean removed = HTTPUtility.getResponse(Boolean.class, httpURLConnection);
			
			logger.info("{} with UUID {} {} to {} with UUID {}", type, instanceUUID,
					removed ? " successfully removed" : "was NOT removed", Context.NAME, contextUUID);
			return removed;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean removeFromContext(IE identifiableElement, UUID contextUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		try {
			String type = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
			UUID instanceUUID = identifiableElement.getHeader().getUUID();
			return removeFromContext(type, instanceUUID, contextUUID);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public boolean removeFromCurrentContext(String type, UUID instanceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		UUID contextUUID = getCurrentContextUUID();
		return removeFromContext(type, instanceUUID, contextUUID);
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean removeFromCurrentContext(IE identifiableElement)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		UUID contextUUID = getCurrentContextUUID();
		return removeFromContext(identifiableElement, contextUUID);
	}
	
	@Override
	public Set<UUID> getElementContexts(String type, UUID instanceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to get contexts of {} with UUID {}", type, instanceUUID);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.path(SharingPath.SHARING_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(instanceUUID.toString());
			gxHTTPStringRequest.path(SharingPath.CONTEXTS_PATH_PART);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String jsonArray = HTTPUtility.getResponse(String.class, httpURLConnection);
			Set<UUID> contexts = ContextUtility.getContextUUIDSet(jsonArray);
			
			logger.info("Contexts of {} with UUID {} are {}", type, instanceUUID, contexts);
			
			return contexts;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

	@Override
	public <IE extends IdentifiableElement> Set<UUID> getElementContexts(IE identifiableElement)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		String type = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
		UUID instanceUUID = identifiableElement.getHeader().getUUID();
		return getElementContexts(type, instanceUUID);
	}
	
	
	@Override
	public boolean addResourceToContext(String resourceType, UUID resourceUUID, UUID contextUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToContext(resourceType, resourceUUID, contextUUID);
	}
	
	@Override
	public <R extends Resource> boolean addResourceToContext(R resource, UUID contextUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToContext(resource, contextUUID);
	}
	
	@Override
	public boolean addResourceToCurrentContext(String resourceType, UUID resourceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToCurrentContext(resourceType, resourceUUID);
	}
	
	@Override
	public <R extends Resource> boolean addResourceToCurrentContext(R resource)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToCurrentContext(resource);
	}
	
	@Override
	public boolean removeResourceFromContext(String resourceType, UUID resourceUUID, UUID contextUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromContext(resourceType, resourceUUID, contextUUID);
	}
	
	@Override
	public <R extends Resource> boolean removeResourceFromContext(R resource, UUID contextUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromContext(resource, contextUUID);
	}
	
	@Override
	public boolean removeResourceFromCurrentContext(String resourceType, UUID resourceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromCurrentContext(resourceType, resourceUUID);
	}
	
	@Override
	public <R extends Resource> boolean removeResourceFromCurrentContext(R resource)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromCurrentContext(resource);
	}
	
	@Override
	public boolean addFacetToContext(String facetType, UUID facetUUID, UUID contextUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToContext(facetType, facetUUID, contextUUID);
	}
	
	@Override
	public <F extends Facet> boolean addFacetToContext(F facet, UUID contextUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToContext(facet, contextUUID);
	}
	
	@Override
	public boolean addFacetToCurrentContext(String facetType, UUID facetUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToCurrentContext(facetType, facetUUID);
	}
	
	@Override
	public <F extends Facet> boolean addFacetToCurrentContext(F facet)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToCurrentContext(facet);
	}
	
	@Override
	public boolean removeFacetFromContext(String facetType, UUID facetUUID, UUID contextUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromContext(facetType, facetUUID, contextUUID);
	}
	
	@Override
	public <F extends Facet> boolean removeFacetFromContext(F facet, UUID contextUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromContext(facet, contextUUID);
	}
	
	@Override
	public boolean removeFacetFromCurrentContext(String facetType, UUID facetUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromCurrentContext(facetType, facetUUID);
	}
	
	@Override
	public <F extends Facet> boolean removeFacetFromCurrentContext(F facet)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromCurrentContext(facet);
	}

	@Override
	public Set<UUID> getResourceContexts(String resourceType, UUID resourceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return getElementContexts(resourceType, resourceUUID);
	}

	@Override
	public <R extends Resource> Set<UUID> getResourceContexts(R resource)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return getElementContexts(resource);
	}

	@Override
	public Set<UUID> getFacetContexts(String facetType, UUID facetUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return getElementContexts(facetType, facetUUID);
	}

	@Override
	public <F extends Facet> Set<UUID> getFacetContexts(F facet)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return getElementContexts(facet);
	}
	
}
