package org.gcube.smartgears.connector.resourceregistry;

import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;

import org.gcube.common.security.AuthorizedTasks;
import org.gcube.common.security.factories.AuthorizationProvider;
import org.gcube.common.security.factories.AuthorizationProviderFactory;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.resourcemanagement.model.reference.entities.resources.EService;
import org.gcube.resourcemanagement.model.reference.entities.resources.HostingNode;
import org.gcube.smartgears.configuration.AuthorizationProviderConfiguration;
import org.gcube.smartgears.connector.resourceregistry.resourcemanager.EServiceManager;
import org.gcube.smartgears.connector.resourceregistry.resourcemanager.HostingNodeManager;
import org.gcube.smartgears.connector.resourceregistry.resourcemanager.ResourceManager;
import org.gcube.smartgears.context.application.ApplicationContext;
import org.gcube.smartgears.context.container.ContainerContext;
import org.gcube.smartgears.publishing.Publisher;
import org.gcube.smartgears.publishing.SmartgearsProfilePublisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A facet-based publisher for SmartGears applications that manages resource
 * registration
 * and publication in the Resource Registry.
 */
@SmartgearsProfilePublisher
public class FacetBasedPublisher implements Publisher {

	private static Logger logger = LoggerFactory.getLogger(FacetBasedPublisher.class);

	private static Map<String, ResourceManager> managersCache;

	static {
		managersCache = new ConcurrentHashMap<>();
	}

	/**
	 * Gets the authorization provider from an application context.
	 * 
	 * @param applicationContext the application context
	 * @return the authorization provider
	 */
	public static AuthorizationProvider getAuthorizationProvider(ApplicationContext applicationContext) {
		return FacetBasedPublisher.getAuthorizationProvider(applicationContext.container());
	}

	/**
	 * Gets the authorization provider from a container context.
	 * 
	 * @param containerContext the container context
	 * @return the authorization provider
	 */
	public static AuthorizationProvider getAuthorizationProvider(ContainerContext containerContext) {
		AuthorizationProviderConfiguration authorizationProviderConfiguration = containerContext.configuration()
				.authorizationConfiguration();
		AuthorizationProviderFactory<?> authorizationProviderFactory = authorizationProviderConfiguration
				.getAuthProviderFactory();
		AuthorizationProvider authorizationProvider = authorizationProviderFactory
				.connect(authorizationProviderConfiguration.getCredentials());
		return authorizationProvider;
	}

	@Override
	public boolean create(ContainerContext containerContext, Set<String> contexts) {
		AuthorizationProvider authorizationProvider = FacetBasedPublisher.getAuthorizationProvider(containerContext);
		SortedSet<String> contextToIterate = new TreeSet<>(contexts);
		final String context = contextToIterate.first();
		String containerID = containerContext.id();

		try {
			AuthorizedTasks.executeSafely(() -> {
			managersCache.computeIfAbsent(containerID, id -> {
				HostingNodeManager hostingNodeManager = new HostingNodeManager(containerContext);
				for (final String c : contextToIterate) {
					if (context.equals(c)) {
						try {
							hostingNodeManager.createHostingNode();
						} catch (ResourceRegistryException e) {
							logger.error("Error while creating {} id={} in context {}", HostingNode.NAME, containerID, context, e);
							return hostingNodeManager;
						}
					} else {
						try {
							hostingNodeManager.addToContext(c);
						} catch (ResourceRegistryException e) {
							logger.error("Error while adding {} id={} to context '{}'", HostingNode.NAME, containerID, c, e);
						}
					}
				}
				return hostingNodeManager;
			});
				
			}, authorizationProvider.getSecretForContext(context));
		} catch (Throwable e) {
			logger.error("Error while intiliating {} id={}. Available contexts {}", HostingNode.NAME, containerID, contextToIterate, e);
		}
		return true;
	}

	@Override
	public boolean create(ApplicationContext applicationContext, Set<String> contexts) {
		AuthorizationProvider authorizationProvider = FacetBasedPublisher.getAuthorizationProvider(applicationContext);
		SortedSet<String> contextToIterate = new TreeSet<>(contexts);
		final String context = contextToIterate.first();
		String applicationID = applicationContext.id();

		try {
			AuthorizedTasks.executeSafely(() -> {
				managersCache.computeIfAbsent(applicationID, id -> {
					EServiceManager eServiceManager = new EServiceManager(applicationContext);
					managersCache.put(applicationID, eServiceManager);
					for (final String c : contextToIterate) {
						if (context.equals(c)) {
							try {
								eServiceManager.createEService();
							} catch (ResourceRegistryException e) {
								logger.error("Error while creating {} name={} id={} in context {}", EService.NAME,
									applicationContext.name(), applicationID, context, e);
								return eServiceManager;
							}
						} else {
							try {
								eServiceManager.addToContext(c);
							} catch (Throwable e) {
								logger.error("Error while adding {} name={} id={} to context {}", EService.NAME,
									applicationContext.name(), applicationID, c, e);
							}
						}
					}
					return eServiceManager;
				});

			}, authorizationProvider.getSecretForContext(context));
		} catch (Throwable e) {
			logger.error("Error while initializing {} name={} id={}. Available contexts {}", EService.NAME,
									applicationContext.name(), applicationID, contextToIterate, e);
		}
		return true;
	}

	@Override
	public boolean remove(ContainerContext containerContext, Set<String> contexts) {
		HostingNodeManager hostingNodeManager = (HostingNodeManager) managersCache.get(containerContext.id());
		AuthorizationProvider authorizationProvider = FacetBasedPublisher.getAuthorizationProvider(containerContext);
		for (String c : contexts) {
			AuthorizedTasks.executeSafely(() -> {
				try {
					hostingNodeManager.removeFromCurrentContext();
				} catch (Throwable e) {
					logger.error("Unable to remove {} id={} from context {}", HostingNode.NAME, containerContext.id(), c, e);
				}
			}, authorizationProvider.getSecretForContext(c));
		}
		return true;
	}

	@Override
	public boolean remove(ApplicationContext applicationContext, Set<String> contexts) {
		EServiceManager eServiceManager = (EServiceManager) managersCache.get(applicationContext.id());
		AuthorizationProvider authorizationProvider = FacetBasedPublisher.getAuthorizationProvider(applicationContext);
		for (String c : contexts) {
			AuthorizedTasks.executeSafely(() -> {
				try {
					eServiceManager.removeFromCurrentContext();
				} catch (Exception e) {
					logger.error("Unable to remove {} name={} id={} from context {}", EService.NAME,
							applicationContext.name(), applicationContext.id(), c, e);
				}
			}, authorizationProvider.getSecretForContext(c));
		}
		return true;
	}

	@Override
	public boolean update(ContainerContext containerContext) {
		HostingNodeManager hostingNodeManager = (HostingNodeManager) managersCache.get(containerContext.id());
		AuthorizationProvider authorizationProvider = FacetBasedPublisher.getAuthorizationProvider(containerContext);
		SortedSet<String> contextToIterate = new TreeSet<>(authorizationProvider.getContexts());
		final String context = contextToIterate.first();
		AuthorizedTasks.executeSafely(() -> {
			try {
				hostingNodeManager.updateFacets();
			} catch (ResourceRegistryException e) {
				logger.error("Unable to update {} id={}", HostingNode.NAME, containerContext.id(), e);
			}
		}, authorizationProvider.getSecretForContext(context));
		return true;
	}

	@Override
	public boolean update(ApplicationContext applicationContext) {
		EServiceManager eServiceManager = (EServiceManager) managersCache.get(applicationContext.id());
		AuthorizationProvider authorizationProvider = FacetBasedPublisher.getAuthorizationProvider(applicationContext);
		SortedSet<String> contextToIterate = new TreeSet<>(authorizationProvider.getContexts());
		final String context = contextToIterate.first();
		AuthorizedTasks.executeSafely(() -> {
			try {
				eServiceManager.updateFacets();
			} catch (ResourceRegistryException e) {
				logger.error("Unable to update {} name={} id={}", EService.NAME,
				applicationContext.name(), applicationContext.id(), e);
			}
		}, authorizationProvider.getSecretForContext(context));
		return true;
	}

}
