package org.gcube.smartgears.connector.resourceregistry;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.UUID;

import org.gcube.common.security.AuthorizedTasks;
import org.gcube.common.security.factories.AuthorizationProvider;
import org.gcube.common.security.factories.AuthorizationProviderFactory;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCache;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.publisher.ResourceRegistryPublisher;
import org.gcube.informationsystem.resourceregistry.publisher.ResourceRegistryPublisherFactory;
import org.gcube.resourcemanagement.model.reference.entities.resources.EService;
import org.gcube.resourcemanagement.model.reference.entities.resources.HostingNode;
import org.gcube.smartgears.configuration.AuthorizationProviderConfiguration;
import org.gcube.smartgears.connector.resourceregistry.resourcemanager.EServiceManager;
import org.gcube.smartgears.connector.resourceregistry.resourcemanager.HostingNodeManager;
import org.gcube.smartgears.connector.resourceregistry.resourcemanager.ResourceManager;
import org.gcube.smartgears.context.application.ApplicationContext;
import org.gcube.smartgears.context.container.ContainerContext;
import org.gcube.smartgears.publishing.Publisher;
import org.gcube.smartgears.publishing.SmartgearsProfilePublisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A facet-based publisher for SmartGears applications that manages resource
 * registration
 * and publication in the Resource Registry.
 */
@SmartgearsProfilePublisher
public class FacetBasedPublisher implements Publisher {

	private static Logger logger = LoggerFactory.getLogger(FacetBasedPublisher.class);

	private static Map<String, ResourceManager> managersCache;

	static {
		managersCache = new HashMap<>();
	}

	/**
	 * Gets the authorization provider from an application context.
	 * 
	 * @param applicationContext the application context
	 * @return the authorization provider
	 */
	public static AuthorizationProvider getAuthorizationProvider(ApplicationContext applicationContext) {
		return FacetBasedPublisher.getAuthorizationProvider(applicationContext.container());
	}

	/**
	 * Gets the authorization provider from a container context.
	 * 
	 * @param containerContext the container context
	 * @return the authorization provider
	 */
	public static AuthorizationProvider getAuthorizationProvider(ContainerContext containerContext) {
		AuthorizationProviderConfiguration authorizationProviderConfiguration = containerContext.configuration()
				.authorizationConfiguration();
		AuthorizationProviderFactory<?> authorizationProviderFactory = authorizationProviderConfiguration
				.getAuthProviderFactory();
		AuthorizationProvider authorizationProvider = authorizationProviderFactory
				.connect(authorizationProviderConfiguration.getCredentials());
		return authorizationProvider;
	}

	/**
	 * The Resource Registry publisher instance.
	 */
	protected ResourceRegistryPublisher resourceRegistryPublisher;

	/**
	 * Gets the UUID of a context by its full name.
	 * 
	 * @param contextFullName the full name of the context
	 * @return the UUID of the context
	 * @throws ResourceRegistryException if an error occurs
	 */
	public UUID getContextUUID(String contextFullName) throws ResourceRegistryException {
		ContextCache contextCache = resourceRegistryPublisher.getContextCache();
		UUID contextUUID = contextCache.getUUIDByFullName(contextFullName);
		return contextUUID;
	}

	@Override
	public boolean create(ContainerContext containerContext, Set<String> contexts) {
		AuthorizationProvider authorizationProvider = FacetBasedPublisher.getAuthorizationProvider(containerContext);
		SortedSet<String> contextToIterate = new TreeSet<>(contexts);
		final String context = contextToIterate.first();
		String containerID = containerContext.id();

		try {
			AuthorizedTasks.executeSafely(() -> {
				if (resourceRegistryPublisher == null) {
					resourceRegistryPublisher = ResourceRegistryPublisherFactory.create();
				}

				HostingNodeManager hostingNodeManager = null;

				if (managersCache.containsKey(containerID)) {
					hostingNodeManager = (HostingNodeManager) managersCache.get(containerID);
				} else {
					hostingNodeManager = new HostingNodeManager(containerContext, resourceRegistryPublisher);
					managersCache.put(containerID, hostingNodeManager);
				}

				for (final String c : contextToIterate) {
					try {
						if (context.compareTo(c) == 0) {
							hostingNodeManager.createHostingNode();
						} else {
							hostingNodeManager.addToContext(c);
						}
					} catch (Throwable e) {
						logger.error("Error while publishing {} (id='{}') in context '{}'", HostingNode.NAME,
								containerID, c, e);
					}
				}
			}, authorizationProvider.getSecretForContext(context));
		} catch (Throwable e) {
			logger.error("Error while publishing {} (id='{}')", HostingNode.NAME, containerID, e);
		}
		return true;
	}

	@Override
	public boolean create(ApplicationContext applicationContext, Set<String> contexts) {
		AuthorizationProvider authorizationProvider = FacetBasedPublisher.getAuthorizationProvider(applicationContext);
		SortedSet<String> contextToIterate = new TreeSet<>(contexts);
		final String context = contextToIterate.first();
		String applicationID = applicationContext.id();

		try {
			AuthorizedTasks.executeSafely(() -> {
				if (resourceRegistryPublisher == null) {
					resourceRegistryPublisher = ResourceRegistryPublisherFactory.create();
				}
				EServiceManager eServiceManager = null;

				if (managersCache.containsKey(applicationID)) {
					eServiceManager = (EServiceManager) managersCache.get(applicationID);
				} else {
					eServiceManager = new EServiceManager(applicationContext, resourceRegistryPublisher);
					managersCache.put(applicationID, eServiceManager);
				}

				for (final String c : contextToIterate) {
					try {
						if (context.compareTo(c) == 0) {
							eServiceManager.createEService();
						} else {
							eServiceManager.addToContext(c);
						}
					} catch (Throwable e) {
						logger.error("Error while publishing {} (name='{}', id='{}') in context '{}'", EService.NAME,
								applicationContext.name(), applicationID, c, e);
					}
				}
			}, authorizationProvider.getSecretForContext(context));
		} catch (Throwable e) {
			logger.error("Error while publishing {} (name='{}', id='{}')", EService.NAME, applicationContext.name(),
					applicationID, e);
		}
		return true;
	}

	@Override
	public boolean remove(ContainerContext containerContext, Set<String> contexts) {
		HostingNodeManager hostingNodeManager = (HostingNodeManager) managersCache.get(containerContext.id());
		AuthorizationProvider authorizationProvider = FacetBasedPublisher.getAuthorizationProvider(containerContext);
		for (String c : contexts) {
			try {
				AuthorizedTasks.executeSafely(() -> {
					try {
						hostingNodeManager.removeFromCurrentContext();
					} catch (Exception e) {
						throw new RuntimeException(e);
					}

				}, authorizationProvider.getSecretForContext(c));
			} catch (Throwable e) {
				logger.error("Unable to remove {} (id='{}') from context '{}'", HostingNode.NAME, containerContext.id(),
						c, e);
			}
		}
		return false;
	}

	@Override
	public boolean remove(ApplicationContext applicationContext, Set<String> contexts) {
		EServiceManager eServiceManager = (EServiceManager) managersCache.get(applicationContext.id());
		AuthorizationProvider authorizationProvider = FacetBasedPublisher.getAuthorizationProvider(applicationContext);
		for (String c : contexts) {
			try {
				AuthorizedTasks.executeSafely(() -> {
					try {
						eServiceManager.removeFromCurrentContext();
					} catch (Exception e) {
						throw new RuntimeException(e);
					}

				}, authorizationProvider.getSecretForContext(c));
			} catch (Throwable e) {
				logger.error("Unable to remove {} (name='{}', id='{}') from context '{}'", EService.NAME,
						applicationContext.name(), applicationContext.id(), c, e);
			}
		}
		return true;
	}

	@Override
	public boolean update(ContainerContext containerContext) {
		HostingNodeManager hostingNodeManager = (HostingNodeManager) managersCache.get(containerContext.id());
		AuthorizationProvider authorizationProvider = FacetBasedPublisher.getAuthorizationProvider(containerContext);
		SortedSet<String> contextToIterate = new TreeSet<>(authorizationProvider.getContexts());
		final String context = contextToIterate.first();
		try {
			AuthorizedTasks.executeSafely(() -> {
				try {
					hostingNodeManager.updateFacets();
				} catch (ResourceRegistryException e) {
					throw new RuntimeException(e);
				}
			}, authorizationProvider.getSecretForContext(context));
		} catch (Throwable e) {
			logger.error("Unable to update {} (id='{}')", HostingNode.NAME, containerContext.id(), e);
		}
		return true;
	}

	@Override
	public boolean update(ApplicationContext applicationContext) {
		EServiceManager eServiceManager = (EServiceManager) managersCache.get(applicationContext.id());
		AuthorizationProvider authorizationProvider = FacetBasedPublisher.getAuthorizationProvider(applicationContext);
		SortedSet<String> contextToIterate = new TreeSet<>(authorizationProvider.getContexts());
		final String context = contextToIterate.first();
		try {
			AuthorizedTasks.executeSafely(() -> {
				try {
					eServiceManager.updateFacets();
				} catch (ResourceRegistryException e) {
					throw new RuntimeException(e);
				}
			}, authorizationProvider.getSecretForContext(context));
		} catch (Throwable e) {
			logger.error("Unable to update {} (name='{}', id='{}') ", EService.NAME, applicationContext.name(),
					applicationContext.id(), e);
		}
		return true;
	}

}
