package org.gcube.informationsystem.resourceregistry.api.request;

/**
 * Configuration interface for Resource Registry request parameters.
 * Provides methods to configure pagination, metadata inclusion, and context behavior for Resource Registry operations.
 * 
 * @author Luca Frosini (ISTI - CNR)
 */
public interface RequestInfo {
	
	/** The Constant DEFAULT_OFFSET. */
	public static final Integer DEFAULT_OFFSET = 0;
	/** The Constant DEFAULT_LIMIT. */
	public static final Integer DEFAULT_LIMIT = 10;

	/** The Constant UNBOUNDED_LIMIT. */
	public static final Integer UNBOUNDED_LIMIT = -1;

	/**
	 * Gets the maximum number of instances to return in a single response.
	 * 
	 * @return the limit value for pagination, or null if not set (uses server default)
	 */
	public Integer getLimit();

	/**
	 * Sets the maximum number of instances to return in a single response.
	 * Used for pagination to control response size and manage memory usage.
	 * 
	 * <ul>
	 * <li><strong>Default value:</strong> Server configuration dependent</li>
	 * <li><strong>Example:</strong> Set to 50 to get at most 50 instances per request</li>
	 * <li><strong>Unlimited results:</strong> Set to {@link #UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li><strong>Query parameter:</strong> limit</li>
	 * </ul>
	 * 
	 * @param limit Maximum number of instances to return, or {@link #UNBOUNDED_LIMIT} for unlimited results
	 */
	public void setLimit(Integer limit);

	/**
	 * Gets the number of instances to skip from the beginning of the result set.
	 * 
	 * @return the offset value for pagination, or null if not set (defaults to 0)
	 */
	public Integer getOffset();

	/**
	 * Sets the number of instances to skip from the beginning of the result set.
	 * Used for pagination to navigate through large result sets.
	 * 
	 * <ul>
	 * <li><strong>Default value:</strong> 0</li>
	 * <li><strong>Example:</strong> Set to 10 to skip the first 10 instances (useful for pagination)</li>
	 * <li><strong>Query parameter:</strong> {@code offset}</li>
	 * </ul>
	 * 
	 * @param offset Number of instances to skip from the beginning
	 */
	public void setOffset(Integer offset);

	/**
	 * Checks whether to include metadata in the response instance.
	 * 
	 * @return true if metadata should be included, false for basic information only
	 */
	public boolean includeMeta();

	/**
	 * Sets whether to include metadata in the response instance.
	 * Controls whether to include {@link org.gcube.informationsystem.model.reference.properties.Metadata} in the response.
	 * 
	 * <ul>
	 * <li><strong>Default value:</strong> false (basic information only)</li>
	 * <li><strong>Values:</strong> </li>
	 *   <li>true: includes metadata with role-based filtering</li>
	 *   <li>false: basic information only</li>
	 * <li><strong>Query parameter:</strong> includeMeta</li>
	 * </ul>
	 * 
	 * <strong>Authorization Restriction:</strong>
	 * <ul>
	 * <li><strong>IS-Manager, Infrastructure-Manager, and Context-Manager:</strong> See complete metadata including sensitive information (createdBy, lastUpdatedBy)</li>
	 * <li><strong>Other users:</strong> See filtered metadata with sensitive fields obfuscated</li>
	 * </ul>
	 * 
	 * @param includeMeta Whether to include metadata in the response instance
	 */
	public void setIncludeMeta(boolean includeMeta);

	/**
	 * Checks whether to include metadata for all nested instances in the response.
	 * 
	 * @return true if metadata should be included for all nested elements, false for main instance only
	 */
	public boolean allMeta();

	/**
	 * Sets whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response.
	 * Controls whether to include {@link org.gcube.informationsystem.model.reference.properties.Metadata} 
	 * in all {@link org.gcube.informationsystem.base.reference.IdentifiableElement} or just in the root instance.
	 * Must be used in conjunction with includeMeta=true.
	 * 
	 * <ul>
	 * <li><strong>Default value:</strong> false (metadata only for main instance, more human-readable)</li>
	 * <li><strong>Values:</strong></li>
	 *   <li>true: complete metadata for all nested IdentifiableElement instances</li>
	 *   <li>false: metadata only in the root instance, more readable</li>
	 * <li><strong>Purpose:</strong> When false, produces more human-readable responses with less JSON to process</li>
	 * <li><strong>Query parameter:</strong> allMeta</li>
	 * </ul>
	 * 
	 * @param allMeta Whether to include metadata for all nested instances in the response
	 */
	public void setAllMeta(boolean allMeta);

	/**
	 * Checks whether to include instances from child contexts of the current context.
	 * This is an admin-only feature.
	 * 
	 * @return true if hierarchical mode is enabled, false for current context only
	 */
	public boolean isHierarchicalMode();

	/**
	 * Sets whether to include instances from child contexts of the current context.
	 * 
	 * <ul>
	 * <li><strong>Default value:</strong> false (child contexts not included)</li>
	 * <li><strong>Values:</strong></li>
	 *   <li>true: includes child contexts</li>
	 *   <li>false: current context only</li>
	 * <li><strong>Query parameter:</strong> {@code hierarchical}</li>
	 * </ul>
	 * 
	 * <strong>Authorization Restriction:</strong>
	 * <ul>
	 * <li><strong>IS-Manager and Infrastructure-Manager:</strong> Global roles, can use hierarchical mode in any context</li>
	 * <li><strong>Context-Manager:</strong> Context-specific role, can use hierarchical mode only in contexts where they have this role</li>
	 * <li><strong>Other users:</strong> Not available</li>
	 * </ul>
	 * 
	 * @param hierarchicalMode Whether to include instances from child contexts of the current context
	 */
	public void setHierarchicalMode(boolean hierarchicalMode);

	/**
	 * Checks whether to include the list of contexts where instances and their nested elements are available.
	 * 
	 * @return true if context information should be included, false if context information is not included
	 */
	public boolean includeContexts();

	/**
	 * Sets whether to include the list of contexts where the instance and its nested elements are available in the response.
	 * 
	 * <ul>
	 * <li><strong>Default value:</strong> false (context information not included)</li>
	 * <li><strong>Values:</strong></li>
	 *   <li>true: shows context availability</li>
	 *   <li>false: no context information</li>
	 * <li><strong>Query parameter:</strong> {@code includeContexts}</li>
	 * </ul>
	 * 
	 * <strong>Context Rules:</strong>
	 * <ul>
	 * <li><strong>Resources:</strong> Present in all contexts that form the union of contexts of all their Facets</li>
	 * <li><strong>ConsistsOf relations:</strong> Present in all contexts where their target Facets are present</li>
	 * <li><strong>Facets:</strong> Context availability depends on the Resources that include them via ConsistsOf relations</li>
	 * </ul>
	 * 
	 * @param includeContexts Whether to include the list of contexts where instances and their nested elements are available
	 */
	public void setIncludeContexts(boolean includeContexts);

}
