package org.gcube.informationsystem.resourceregistry.api.contexts;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import org.gcube.informationsystem.contexts.impl.entities.ContextImpl;
import org.gcube.informationsystem.contexts.impl.relations.IsParentOfImpl;
import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.contexts.reference.relations.IsParentOf;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.tree.Tree;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The Class ContextCache.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public class ContextCache {

	/** The logger. */
	private static Logger logger = LoggerFactory.getLogger(ContextCache.class);
	
	/** The Constant DEFAULT_EXPIRING_TIMEOUT. */
	// in millisec
	public static final long DEFAULT_EXPIRING_TIMEOUT;
	
	/** The expiring timeout. */
	public static int expiringTimeout;
	
	static {
		DEFAULT_EXPIRING_TIMEOUT = TimeUnit.HOURS.toMillis(6);
		expiringTimeout = (int) DEFAULT_EXPIRING_TIMEOUT;
	}
	
	/**
	 * Sets the expiring timeout.
	 *
	 * @param expiringTimeout the new expiring timeout
	 */
	public static void setExpiringTimeout(int expiringTimeout) {
		ContextCache.expiringTimeout = expiringTimeout;
	}

	/** The singleton. */
	protected static ContextCache singleton;
	
	/**
	 * Gets the single instance of ContextCache.
	 *
	 * @return single instance of ContextCache
	 */
	public synchronized static ContextCache getInstance() {
		if(singleton==null) {
			singleton = new ContextCache();
		}
		return singleton;
	}
	
	/** The context cache renewal. */
	protected ContextCacheRenewal contextCacheRenewal;
	
	/** The creation time. */
	// in millisec used only for logging and debugging
	protected Calendar creationTime;
	
	/** The expiring time. */
	// in millisec
	protected Calendar expiringTime;
	
	/** The contexts. */
	protected List<Context> contexts;
	
	/** The uuid to context. */
	protected Map<UUID, Context> uuidToContext;
	
	/** The uuid to context full name. */
	protected Map<UUID, String> uuidToContextFullName;
	
	/** The context full name to UUID. */
	protected Map<String, UUID> contextFullNameToUUID;
	
	/** The contexts tree. */
	protected Tree<Context> contextsTree;
	
	/** The initialized. */
	protected boolean initialized;
	
	/**
	 * Instantiates a new context cache.
	 */
	public ContextCache() {
		Calendar now = Calendar.getInstance();
		cleanCache(now);
		initialized = false;
	}
	
	/**
	 * Clean cache.
	 */
	public void cleanCache() {
		cleanCache(Calendar.getInstance());
	}
	
	/**
	 * Clean cache.
	 *
	 * @param calendar the calendar
	 */
	protected void cleanCache(Calendar calendar) {
		this.contexts = new ArrayList<>();
		this.uuidToContext = new LinkedHashMap<>();
		this.uuidToContextFullName = new LinkedHashMap<>();
		this.contextFullNameToUUID = new TreeMap<>();
		this.contextsTree = new Tree<Context>(new ContextInformation());
		this.contextsTree.setAllowMultipleInheritance(false);
		this.creationTime = Calendar.getInstance();
		this.creationTime.setTimeInMillis(calendar.getTimeInMillis());
		this.expiringTime = Calendar.getInstance();
		this.expiringTime.setTimeInMillis(calendar.getTimeInMillis());
		this.expiringTime.add(Calendar.MILLISECOND, -1);
		this.expiringTime.add(Calendar.MILLISECOND, expiringTimeout);
		initialized = false;
	}
	
	/**
	 * Renew.
	 *
	 * @throws ResourceRegistryException the resource registry exception
	 */
	public void renew() throws ResourceRegistryException {
		cleanCache();
		refreshContextsIfNeeded();
	}
	
	/**
	 * Gets the context cache renewal.
	 *
	 * @return the context cache renewal
	 */
	public ContextCacheRenewal getContextCacheRenewal() {
		return contextCacheRenewal;
	}

	/**
	 * Sets the context cache renewal.
	 *
	 * @param contextCacheRenewal the new context cache renewal
	 */
	public void setContextCacheRenewal(ContextCacheRenewal contextCacheRenewal) {
		if(this.contextCacheRenewal==null) {
			this.contextCacheRenewal = contextCacheRenewal;
		}
	}
	
	/**
	 * Refresh contexts if needed.
	 *
	 * @throws ResourceRegistryException the resource registry exception
	 */
	public synchronized void refreshContextsIfNeeded() throws ResourceRegistryException {
		Calendar now = Calendar.getInstance();
		if((now.after(expiringTime) || (!initialized)) && contextCacheRenewal!=null) {
			try {
				List<Context> contexts = contextCacheRenewal.renew();
				setContexts(now, contexts);
				initialized = true;
			} catch (ResourceRegistryException  e) {
				if(!initialized) {
					logger.error("Unable to initialize Context Cache", e);
					throw e;
				}else {
					logger.error("Unable to refresh Context Cache", e);
				}
			}
			
		}
	}
	
	/**
	 * Gets the contexts.
	 *
	 * @return the contexts
	 * @throws ResourceRegistryException the resource registry exception
	 */
	public synchronized List<Context> getContexts() throws ResourceRegistryException {
		refreshContextsIfNeeded();
		return contexts;
	}

	/**
	 * Sets the contexts.
	 *
	 * @param contexts the new contexts
	 */
	public void setContexts(List<Context> contexts) {
		Calendar now = Calendar.getInstance();
		setContexts(now, contexts);
	}
	
	/**
	 * Sets the contexts.
	 *
	 * @param calendar the calendar
	 * @param contexts the contexts
	 */
	protected void setContexts(Calendar calendar, List<Context> contexts) {
		cleanCache(calendar);
		
		for(Context c : contexts) {
			UUID uuid = c.getID();
			Context context = new ContextImpl(c.getName());
			context.setMetadata(c.getMetadata());
			context.setID(uuid);
			this.contexts.add(context);
			this.uuidToContext.put(uuid, context);
		}
		
		for(Context c : contexts) {
			UUID uuid = c.getID();
			Context context = this.uuidToContext.get(uuid);
			if(c.getParent()!=null) {
				IsParentOf ipo = c.getParent();
				UUID parentUUID = ipo.getSource().getID();
				Context parent = this.uuidToContext.get(parentUUID);
				IsParentOf isParentOf = new IsParentOfImpl(parent, context);
				isParentOf.setID(parentUUID);
				isParentOf.setMetadata(ipo.getMetadata());
				parent.addChild(isParentOf);
				context.setParent(isParentOf);
			}
		}
		
		for(Context context : contexts) {
			UUID uuid = context.getID();
			String fullName = getContextFullName(context);
			this.uuidToContextFullName.put(uuid, fullName);
			this.contextFullNameToUUID.put(fullName, uuid);
		}
		
		SortedSet<String> contextFullNames = new TreeSet<String>(contextFullNameToUUID.keySet());
		for(String contextFullName : contextFullNames) {
			UUID uuid = contextFullNameToUUID.get(contextFullName);
			Context context = uuidToContext.get(uuid);
			contextsTree.addNode(context);
		}
		
	}

	/**
	 * Gets the context full name.
	 *
	 * @param context the context
	 * @return the context full name
	 */
	protected String getContextFullName(Context context) {
		StringBuilder stringBuilder = new StringBuilder();
		IsParentOf ipo = context.getParent();
		if(ipo!=null) {
			Context c = ipo.getSource();
			c = uuidToContext.get(c.getID());
			String parentFullName = getContextFullName(c);
			stringBuilder.append(parentFullName);
		}
		stringBuilder.append("/");
		stringBuilder.append(context.getName());
		return stringBuilder.toString();
	}
	
	
	/**
	 * Gets the context full name by UUID.
	 *
	 * @param uuid the uuid
	 * @return the context full name by UUID
	 * @throws ResourceRegistryException the resource registry exception
	 */
	public synchronized String getContextFullNameByUUID(UUID uuid) throws ResourceRegistryException {
		refreshContextsIfNeeded();
		return uuidToContextFullName.get(uuid);
	}
	
	/**
	 * Gets the context full name by UUID.
	 *
	 * @param uuid the uuid
	 * @return the context full name by UUID
	 * @throws ResourceRegistryException the resource registry exception
	 */
	public synchronized String getContextFullNameByUUID(String uuid) throws ResourceRegistryException {
		refreshContextsIfNeeded();
		return uuidToContextFullName.get(UUID.fromString(uuid));
	}
	
	/**
	 * Gets the uuid by full name.
	 *
	 * @param contextFullName the context full name
	 * @return the uuid by full name
	 * @throws ResourceRegistryException the resource registry exception
	 */
	public synchronized UUID getUUIDByFullName(String contextFullName) throws ResourceRegistryException {
		refreshContextsIfNeeded();
		return contextFullNameToUUID.get(contextFullName);
	}
	
	/**
	 * Gets the context by UUID.
	 *
	 * @param uuid the uuid
	 * @return the context by UUID
	 * @throws ResourceRegistryException the resource registry exception
	 */
	public synchronized Context getContextByUUID(UUID uuid) throws ResourceRegistryException {
		refreshContextsIfNeeded();
		return uuidToContext.get(uuid);
	}
	
	/**
	 * Gets the context by UUID.
	 *
	 * @param uuid the uuid
	 * @return the context by UUID
	 * @throws ResourceRegistryException the resource registry exception
	 */
	public synchronized Context getContextByUUID(String uuid) throws ResourceRegistryException {
		refreshContextsIfNeeded();
		return getContextByUUID(UUID.fromString(uuid));
	}
	
	/**
	 * Gets the context by full name.
	 *
	 * @param contextFullName the context full name
	 * @return the context by full name
	 * @throws ResourceRegistryException the resource registry exception
	 */
	public synchronized Context getContextByFullName(String contextFullName) throws ResourceRegistryException {
		UUID uuid = getUUIDByFullName(contextFullName);
		return getContextByUUID(uuid);
	}
	
	/**
	 * Gets the UUID to context full name association.
	 *
	 * @return an Map containing UUID to Context FullName association
	 * @throws ResourceRegistryException the resource registry exception
	 */
	public synchronized Map<UUID, String> getUUIDToContextFullNameAssociation() throws ResourceRegistryException {
		refreshContextsIfNeeded();
		return new LinkedHashMap<>(uuidToContextFullName);
	}
	
	/**
	 * Gets the context full name to UUID association.
	 *
	 * @return an Map containing Context FullName to UUID association
	 * @throws ResourceRegistryException the resource registry exception
	 */
	public synchronized Map<String, UUID> getContextFullNameToUUIDAssociation() throws ResourceRegistryException {
		refreshContextsIfNeeded();
		return new TreeMap<>(contextFullNameToUUID);
	}
	
	/**
	 * Gets the contexts tree.
	 *
	 * @return the contexts tree
	 * @throws ResourceRegistryException the resource registry exception
	 */
	public Tree<Context> getContextsTree() throws ResourceRegistryException {
		refreshContextsIfNeeded();
		return contextsTree;
	}
	
}
