package org.gcube.informationsystem.utils;

import java.io.IOException;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.core.JsonProcessingException;
import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A utility class for handling UUIDs.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public class UUIDUtility {

	private static Logger logger = LoggerFactory.getLogger(UUIDUtility.class);

	/**
	 * Gets the UUID from a {@link JsonNode}.
	 *
	 * @param jsonNode The JSON node.
	 * @return The UUID, or {@code null} if not found.
	 */
	public static UUID getUUID(JsonNode jsonNode) {
		if (jsonNode.has(IdentifiableElement.ID_PROPERTY)) {
			JsonNode jsonNodeID = jsonNode.get(IdentifiableElement.ID_PROPERTY);
			if (jsonNodeID != null && jsonNodeID.isTextual()) {
				return UUID.fromString(jsonNodeID.asText());
			}
		}
		return null;
	}

	/**
	 * Gets the UUID from a JSON string.
	 *
	 * @param json The JSON string.
	 * @return The UUID.
	 * @throws IOException             if an I/O error occurs.
	 * @throws JsonProcessingException if a JSON processing error occurs.
	 */
	public static UUID getUUID(String json) throws JsonProcessingException, IOException {
		logger.trace("Trying to get UUID of {} ", json);
		JsonNode jsonNode = JsonUtility.getJsonNode(json);
		return getUUID(jsonNode);
	}

	/**
	 * Gets the UUID as a string from a {@link JsonNode}.
	 *
	 * @param jsonNode The JSON node.
	 * @return The UUID as a string, or {@code null} if not found.
	 */
	public static String getUUIDAsString(JsonNode jsonNode) {
		UUID uuid = getUUID(jsonNode);
		if (uuid != null) {
			return uuid.toString();
		}
		return null;
	}

	/**
	 * Gets the UUID as a string from a JSON string.
	 *
	 * @param json The JSON string.
	 * @return The UUID as a string.
	 * @throws IOException             if an I/O error occurs.
	 * @throws JsonProcessingException if a JSON processing error occurs.
	 */
	public static String getUUIDAsString(String json) throws JsonProcessingException, IOException {
		return getUUID(json).toString();
	}

	/**
	 * Converts a string to a UUID and validates it.
	 *
	 * @param uuidString The UUID string.
	 * @return The validated UUID.
	 * @throws Exception if the UUID is reserved.
	 */
	public static UUID fromString(String uuidString) throws Exception {
		UUID uuid = UUID.fromString(uuidString);
		UUIDManager.getInstance().validateUUID(uuid);
		return uuid;
	}

}
