package org.gcube.informationsystem.utils;

import java.util.HashSet;
import java.util.Set;
import java.util.TreeSet;
import java.util.UUID;

/**
 * Manages reserved UUIDs and provides validation and generation of valid UUIDs.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public final class UUIDManager {

	private static final Set<String> RESERVED_UUID_STRING;
	private static final Set<UUID> RESERVED_UUID;
	
	private static UUIDManager uuidManager;

	/**
	 * Returns the singleton instance of the UUIDManager.
	 *
	 * @return The UUIDManager instance.
	 */
	public static UUIDManager getInstance() {
		if(uuidManager==null) {
			uuidManager = new UUIDManager();
		}
		return uuidManager;
	}
	
	/**
	 * Private constructor to prevent instantiation.
	 */
	private UUIDManager() {
		
	}
	
	static {
		RESERVED_UUID_STRING = new HashSet<>();
		RESERVED_UUID = new HashSet<>();
		
		String[] uuidValidCharacters = {"0","1","2","3","4","5","6","7","8","9","a","b","c","d","e","f"};
		
		for(String string : uuidValidCharacters) {
			String uuidString = "XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX";
			uuidString = uuidString.replaceAll("X", string);
			RESERVED_UUID_STRING.add(uuidString);
			RESERVED_UUID.add(UUID.fromString(uuidString));
		}
	}

	/**
	 * Checks if a UUID is reserved.
	 *
	 * @param uuid The UUID to check.
	 * @return {@code true} if the UUID is reserved, {@code false} otherwise.
	 */
	public boolean isReservedUUID(UUID uuid) {
		return RESERVED_UUID.contains(uuid);
	}

	/**
	 * Checks if a UUID string is reserved.
	 *
	 * @param uuid The UUID string to check.
	 * @return {@code true} if the UUID is reserved, {@code false} otherwise.
	 */
	public boolean isReservedUUID(String uuid) {
		return RESERVED_UUID_STRING.contains(uuid);
	}

	/**
	 * Returns a set of all reserved UUIDs as strings.
	 *
	 * @return A set of reserved UUID strings.
	 */
	public Set<String> getAllReservedUUIDAsStrings(){
		return new TreeSet<>(RESERVED_UUID_STRING);
	}

	/**
	 * Returns a set of all reserved UUIDs.
	 *
	 * @return A set of reserved UUIDs.
	 */
	public Set<UUID> getAllReservedUUID(){
		return new TreeSet<>(RESERVED_UUID);
	}

	/**
	 * Generates a valid (non-reserved) UUID.
	 *
	 * @return A new valid UUID.
	 */
	public UUID generateValidUUID() {
		return generateValidUUID(null);
	}

	/**
	 * Generates a valid (non-reserved) UUID, optionally starting from a given UUID.
	 *
	 * @param uuid The UUID to start with, or {@code null} to generate a new random UUID.
	 * @return A valid UUID.
	 */
	public UUID generateValidUUID(UUID uuid) {
		uuid = uuid==null ? UUID.randomUUID() : uuid;
		while(RESERVED_UUID.contains(uuid)) {
			uuid = UUID.randomUUID();
		}
		return uuid;
	}

	/**
	 * Validates a UUID, throwing an exception if it is reserved.
	 *
	 * @param uuid The UUID to validate.
	 * @return The validated UUID.
	 * @throws Exception if the UUID is reserved.
	 */
	public UUID validateUUID(UUID uuid) throws Exception {
		if(RESERVED_UUID.contains(uuid)) {
			throw new Exception(uuid.toString() + " UUID is reserved. All the following UUID are reserved " + getAllReservedUUIDAsStrings().toString());
		}
		return uuid;
	}
	
}
