package org.gcube.informationsystem.utils;

import java.io.IOException;

import org.gcube.com.fasterxml.jackson.core.JsonProcessingException;
import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.types.TypeMapper;

/**
 * A utility class for retrieving the type name from various sources.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public class TypeUtility {

	/**
	 * Gets the type name from a {@link JsonNode}.
	 *
	 * @param jsonNode The JSON node.
	 * @return The type name, or {@code null} if not found.
	 */
	public static String getTypeName(JsonNode jsonNode) {
		if (jsonNode.has(Element.TYPE_PROPERTY)) {
			return jsonNode.get(Element.TYPE_PROPERTY).asText();
		}
		return null;
	}

	/**
	 * Gets the type name from a JSON string.
	 *
	 * @param json The JSON string.
	 * @return The type name.
	 * @throws IOException             if an I/O error occurs.
	 * @throws JsonProcessingException if a JSON processing error occurs.
	 */
	public static String getTypeName(String json) throws JsonProcessingException, IOException {
		JsonNode jsonNode = JsonUtility.getJsonNode(json);
		return getTypeName(jsonNode);
	}

	/**
	 * Gets the type name from a class that extends {@link Element}.
	 *
	 * @param <E> the element type
	 * @param clz The class.
	 * @return The type name.
	 */
	public static <E extends Element> String getTypeName(Class<E> clz) {
		return TypeMapper.getType(clz);
	}

}
