package org.gcube.informationsystem.types.reference.properties;

import org.gcube.com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.gcube.informationsystem.base.reference.properties.PropertyElement;
import org.gcube.informationsystem.types.annotations.Final;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.impl.properties.LinkedEntityImpl;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.utils.Version;

/**
 * A property type for defining relationships between entities, including
 * cardinality constraints.
 * <p>
 * This is used to specify:
 * <ul>
 * <li>Mandatory or recommended {@code ConsistsOf} relationships to facets.</li>
 * <li>Suggested {@code IsRelatedTo} relationships to other resources.</li>
 * </ul>
 *
 * @author Luca Frosini (ISTI - CNR)
 */
@JsonDeserialize(as=LinkedEntityImpl.class)
@TypeMetadata(name = LinkedEntity.NAME, description = " A convenient type to define a Resource in terms of: mandatory/recommended ConsistsOf->Facets; suggested IsRelatedTo->Resource.", version = Version.MINIMAL_VERSION_STRING)
@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
@Final
public interface LinkedEntity extends PropertyElement, Comparable<LinkedEntity> {

	/** The name of the LinkedEntity property type. */
	public static final String NAME = "LinkedEntity"; // LinkedEntity.class.getSimpleName();
	
	/** The property name for the source of the link. */
	public static final String SOURCE_PROPERTY = "source";
	/** The property name for the relation of the link. */
	public static final String RELATION_PROPERTY = "relation";
	/** The property name for the target of the link. */
	public static final String TARGET_PROPERTY = "target";
	/** The property name for the description of the link. */
	public static final String DESCRIPTION_PROPERTY = "description";
	/** The property name for the minimum cardinality. */
	public static final String MIN_PROPERTY = "min";
	/** The property name for the maximum cardinality. */
	public static final String MAX_PROPERTY = "max";
	
	/**
	 * Returns the name of the source entity type.
	 *
	 * @return The source type name.
	 */
	@ISProperty(name = SOURCE_PROPERTY, readonly = true, mandatory = true, nullable = false)
	public String getSource();
	
	/**
	 * Sets the name of the source entity type.
	 *
	 * @param source The source type name.
	 */
	public void setSource(String source);
	
	/**
	 * Returns the name of the relation type.
	 *
	 * @return The relation type name.
	 */
	@ISProperty(name = RELATION_PROPERTY, readonly = true, mandatory = true, nullable = false)
	public String getRelation();
	
	/**
	 * Sets the name of the relation type.
	 *
	 * @param relation The relation type name.
	 */
	public void setRelation(String relation);

	/**
	 * Returns the name of the target entity type.
	 *
	 * @return The target type name.
	 */
	@ISProperty(name = TARGET_PROPERTY, readonly = true, mandatory = true, nullable = false)
	public String getTarget();
	
	/**
	 * Sets the name of the target entity type.
	 *
	 * @param target The target type name.
	 */
	public void setTarget(String target);

	/**
	 * Returns the description of the link.
	 *
	 * @return The description.
	 */
	@ISProperty(name = DESCRIPTION_PROPERTY, readonly = true, mandatory = true, nullable = false)
	public String getDescription();
	
	/**
	 * Sets the description of the link.
	 *
	 * @param description The description.
	 */
	public void setDescription(String description);
	
	/**
	 * Returns the minimum cardinality of the link.
	 *
	 * @return The minimum cardinality.
	 */
	@ISProperty(name = MIN_PROPERTY, readonly = true, mandatory = true, nullable = false)
	public Integer getMin();
	
	/**
	 * Sets the minimum cardinality of the link.
	 *
	 * @param min The minimum cardinality.
	 */
	public void setMin(Integer min);
	
	/**
	 * Returns the maximum cardinality of the link.
	 *
	 * @return The maximum cardinality.
	 */
	@ISProperty(name = MAX_PROPERTY, readonly = true, mandatory = true, nullable = true)
	public Integer getMax();
	
	/**
	 * Sets the maximum cardinality of the link.
	 *
	 * @param max The maximum cardinality.
	 */
	public void setMax(Integer max);
	
}