package org.gcube.informationsystem.types.reference;

import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.annotation.JsonGetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import org.gcube.com.fasterxml.jackson.annotation.JsonPropertyOrder;
import org.gcube.com.fasterxml.jackson.annotation.JsonSetter;
import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.model.reference.properties.Metadata;
import org.gcube.informationsystem.types.annotations.Abstract;
import org.gcube.informationsystem.types.reference.properties.PropertyDefinition;
import org.gcube.informationsystem.utils.Version;

/**
 * The base interface for all type definitions in the Information System model.
 * <p>
 * A type definition describes the structure of an element, including its name,
 * version, properties, and inheritance relationships.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
@Abstract
@JsonIgnoreProperties(ignoreUnknown=true)
@JsonPropertyOrder({ Element.TYPE_PROPERTY, IdentifiableElement.ID_PROPERTY, IdentifiableElement.METADATA_PROPERTY })
@TypeMetadata(name = Type.NAME, description = "This is the base type to define any Type", version = Version.MINIMAL_VERSION_STRING)
@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
//@JsonDeserialize(as=TypeImpl.class) Do not uncomment to manage subclasses
public interface Type extends IdentifiableElement {
	
	/** The name of the base Type. */
	public static final String NAME = "Type"; // Type.class.getSimpleName();
	
	/** The property name for the type's name. */
	public static final String NAME_PROPERTY = "name";
	/** The property name for the type's description. */
	public static final String DESCRIPTION_PROPERTY = "description";
	/** The property name for the type's version. */
	public static final String VERSION_PROPERTY = "version";
	/** The property name for the type's changelog. */
	public static final String CHANGELOG_PROPERTY = "changelog";
	/** The property name for the 'abstract' flag. */
	public static final String ABSTRACT_PROPERTY = "abstract";
	/** The property name for the 'final' flag. */
	public static final String FINAL_PROPERTY = "final";
	/** The property name for the set of extended types. */
	public static final String EXTENDED_TYPES_PROPERTY = "extendedTypes";
	/** The property name for the set of property definitions. */
	public static final String PROPERTIES_PROPERTY = "properties";
	
	/**
	 * {@inheritDoc}
	 */
	@JsonGetter(value = ID_PROPERTY)
	@Override
	public UUID getID();

	/**
	 * {@inheritDoc}
	 */
	@JsonSetter(value = ID_PROPERTY)
	@Override
	public void setID(UUID uuid);
	
	/**
	 * {@inheritDoc}
	 */
	@JsonGetter(value = METADATA_PROPERTY)
	@Override
	public Metadata getMetadata();
	
	/**
	 * {@inheritDoc}
	 */
	@JsonSetter(value = METADATA_PROPERTY)
	@Override
	public void setMetadata(Metadata metadata);
	
	/**
	 * Returns the name of the type.
	 *
	 * @return The type name.
	 */
	public String getName();
	
	/**
	 * Returns the description of the type.
	 *
	 * @return The type description.
	 */
	public String getDescription();

	/**
	 * Returns the version of the type.
	 *
	 * @return The type version.
	 */
	@JsonIgnore
	public Version getVersion();
	
	/**
	 * Returns the version of the type as a string.
	 *
	 * @return The version string.
	 */
	@JsonGetter(value = VERSION_PROPERTY)
	public String getVersionAsString();

	/**
	 * Returns the changelog for this type.
	 *
	 * @return A map of versions to change descriptions.
	 */
	@JsonIgnore
	public Map<Version, String> getChangelog();
	
	/**
	 * Returns the changelog with versions represented as strings.
	 *
	 * @return A map of version strings to change descriptions.
	 */
	@JsonGetter(value = CHANGELOG_PROPERTY)
	public Map<String, String> getChangelogWithVersionAsString();
	
	/**
	 * Checks if the type is abstract.
	 *
	 * @return {@code true} if the type is abstract, {@code false} otherwise.
	 */
	@JsonGetter(value = ABSTRACT_PROPERTY)
	public boolean isAbstract();

	/**
	 * Checks if the type is final.
	 *
	 * @return {@code true} if the type is final, {@code false} otherwise.
	 */
	@JsonGetter(value = FINAL_PROPERTY)
	public boolean isFinal();
	
	/**
	 * Returns the set of types that this type extends.
	 *
	 * @return A set of parent type names.
	 */
	@JsonGetter(value = EXTENDED_TYPES_PROPERTY)
	public Set<String> getExtendedTypes();
	
	/**
	 * Returns the set of properties defined by this type.
	 *
	 * @return A set of {@link PropertyDefinition}s.
	 */
	public Set<PropertyDefinition> getProperties();

	/**
	 * Returns the {@link AccessType} of this type definition.
	 *
	 * @return The access type.
	 */
	@JsonIgnore
	public AccessType getAccessType();
	
}
