package org.gcube.informationsystem.types.knowledge;

import java.util.Calendar;
import java.util.Collection;
import java.util.concurrent.TimeUnit;

import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.model.knowledge.ModelKnowledge;
import org.gcube.informationsystem.model.knowledge.TypesDiscoverer;
import org.gcube.informationsystem.types.reference.Type;

/**
 * Manages the knowledge about types.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public class TypesKnowledge {

	private static TypesKnowledge instance;
	
	/**
	 * Returns the singleton instance of this class.
	 *
	 * @return The singleton instance.
	 */
	public synchronized static TypesKnowledge getInstance() {
		if(instance==null) {
			instance = new TypesKnowledge();
		}
		return instance;
	}
	
	// in millisec
	/** The default expiring timeout. */
	public static final long DEFAULT_EXPIRING_TIMEOUT;
	
	static {
		DEFAULT_EXPIRING_TIMEOUT = TimeUnit.HOURS.toMillis(6);
		
	}
	
	/** A flag indicating whether the knowledge has been initialized. */
	protected boolean initialized;
	/** The expiring timeout. */
	public int expiringTimeout;
	
	/** The creation time. */
	// in millisec used only for logging and debugging
	protected Calendar creationTime;
	/** The expiring time. */
	// in millisec
	protected Calendar expiringTime;
		
	/** The model knowledge. */
	protected ModelKnowledge<Type, TypeInformation> modelKnowledge;
	/** The types discoverer. */
	protected TypesDiscoverer<Type> typesDiscoverer;
	
	/**
	 * Constructs a new {@code TypesKnowledge}.
	 */
	public TypesKnowledge() {
		initialized = false;
		expiringTimeout = (int) DEFAULT_EXPIRING_TIMEOUT;
		modelKnowledge = new ModelKnowledge<>(new TypeInformation());
	}
	
	/**
	 * Sets the expiring timeout.
	 *
	 * @param expiringTimeout The expiring timeout in milliseconds.
	 */
	public void setExpiringTimeout(int expiringTimeout) {
		this.expiringTimeout = expiringTimeout;
	}
	
	/**
	 * Returns the types discoverer.
	 *
	 * @return The types discoverer.
	 */
	public TypesDiscoverer<Type> getTypesDiscoverer() {
		return typesDiscoverer;
	}

	/**
	 * Sets the types discoverer.
	 *
	 * @param typesDiscoverer The types discoverer.
	 */
	public void setTypesDiscoverer(TypesDiscoverer<Type> typesDiscoverer) {
		this.typesDiscoverer = typesDiscoverer;
	}
	
	/**
	 * Returns the model knowledge.
	 *
	 * @return The model knowledge.
	 */
	public ModelKnowledge<Type, TypeInformation> getModelKnowledge() {
		if(!initialized) {
			discover();
		}else {
			Calendar now = Calendar.getInstance();
			if(now.after(expiringTime)) {
				renew();
			}
		}
		return modelKnowledge;
	}
	
	/**
	 * Initializes the knowledge.
	 *
	 * @param forceReinitialization A flag indicating whether to force reinitialization.
	 */
	protected synchronized void init(boolean forceReinitialization) {
		if(typesDiscoverer!=null && (initialized==false || forceReinitialization)) {
			initialized = false;
			modelKnowledge = new ModelKnowledge<>(new TypeInformation());
			AccessType[] modelTypes = AccessType.getModelTypes();
			for(AccessType modelType : modelTypes) {
				Collection<Type> types = typesDiscoverer.discover(modelType);
				modelKnowledge.addAllType(types);
			}
			initialized = true;
			this.creationTime = Calendar.getInstance();
			this.expiringTime = Calendar.getInstance();
			this.expiringTime.setTimeInMillis(creationTime.getTimeInMillis());
			this.expiringTime.add(Calendar.MILLISECOND, -1);
			this.expiringTime.add(Calendar.MILLISECOND, expiringTimeout);
		}
	}
	
	/**
	 * This method do nothing if TypesDiscoverer
	 * was not set. 
	 * Otherwise initialized the ModelKnowledge
	 * if it was not already initialized.
	 * To enforce rediscovery use renew method.
	 */
	public void discover() {
		init(false);
	}
	
	/**
	 * Force reinitialization of 
	 */
	public void renew() {
		init(true);
	}
}
