package org.gcube.informationsystem.types.impl.validator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A report of the validation of an object node.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public class ValidatorReport {

    /** A flag indicating whether the validation was successful. */
    protected Boolean valid;
    /** The attribute validator reports. */
    protected Map<String, AttributeValidatorReport> attributeValidatorReports;
    /** The attribute validator reports, grouped by validity. */
    protected Map<Boolean, List<AttributeValidatorReport>> attributeValidatorReportByValidity;
    
    /**
     * Constructs a new validator report.
     */
    public ValidatorReport() {
    	this.valid = null;
    	this.attributeValidatorReports = new HashMap<>();
    	this.attributeValidatorReportByValidity = new HashMap<>();
    	for (Boolean b : new Boolean[] { true, false }) {
    		this.attributeValidatorReportByValidity.put(b, new ArrayList<>());
    	}
    }

    /**
     * Returns the attribute validator reports.
     *
     * @return The attribute validator reports.
     */
    public Collection<AttributeValidatorReport> getAttributeValidatorReports() {
    	return attributeValidatorReports.values();
    }

    /**
     * Adds an attribute validator report.
     *
     * @param attributeValidatorReport The attribute validator report.
     */
    public void addAttributeValidatorReport(AttributeValidatorReport attributeValidatorReport) {
    	boolean avrValid = attributeValidatorReport.isValid();
    	this.valid = avrValid && (valid == null ? true : valid);
    	List<AttributeValidatorReport> reports = this.attributeValidatorReportByValidity.get(avrValid);
    	reports.add(attributeValidatorReport);
    	this.attributeValidatorReports.put(attributeValidatorReport.getFieldName(), attributeValidatorReport);
    }

    /**
     * Returns the attribute validator report for the given field name.
     *
     * @param fieldName The field name.
     * @return The attribute validator report.
     */
    public AttributeValidatorReport getAttributeValidatorReport(String fieldName) {
    	return this.attributeValidatorReports.get(fieldName);
    }

    /**
     * Checks if the validation was successful.
     *
     * @return {@code true} if the validation was successful.
     */
    public Boolean isValid() {
    	return valid;
    }

    /**
     * Returns the list of errors.
     *
     * @return The list of errors.
     */
    public List<String> getErrors() {
    	List<String> errors = new ArrayList<>();
    	for (AttributeValidatorReport attributeValidatorReport : attributeValidatorReportByValidity.get(false)) {
    		if (!attributeValidatorReport.isValid()) {
    			errors.addAll(attributeValidatorReport.getMessages());
    		}
    	}
    	return errors;
    }

}
