package org.gcube.informationsystem.types.impl.validator;

import java.util.List;

import org.gcube.com.fasterxml.jackson.databind.node.ObjectNode;
import org.gcube.informationsystem.base.reference.AttributeDefinition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is used to validate an ObjectNode against a list of AttributeDefinitions.
 * It produces a report with the validation results.
 * @author Luca Frosini (ISTI - CNR)
 */
public class ObjectNodeValidator {

    /** The logger. */
    protected static Logger logger = LoggerFactory.getLogger(ObjectNodeValidator.class);

    
    /** The node to validate. */
    protected final ObjectNode nodeToValidate;
    /** The attribute definitions. */
    protected final List<? extends AttributeDefinition> attributeDefinitions;
    /** A flag indicating whether to stop on the first error. */
    protected final boolean stopOnError;

    /** The validator report. */
    protected ValidatorReport validatorReport;
    
    /**
     * Constructs a new validator.
     *
     * @param nodeToValidate       The node to validate.
     * @param attributeDefinitions The attribute definitions.
     */
    public ObjectNodeValidator(ObjectNode nodeToValidate, List<? extends AttributeDefinition> attributeDefinitions) {
    	this(nodeToValidate, attributeDefinitions, false);
    }

    /**
     * Constructs a new validator with the option to stop on the first error.
     *
     * @param nodeToValidate       The node to validate.
     * @param attributeDefinitions The attribute definitions.
     * @param stopOnError          {@code true} to stop on the first error.
     */
    public ObjectNodeValidator(ObjectNode nodeToValidate, List<? extends AttributeDefinition> attributeDefinitions, Boolean stopOnError) {
    	this.nodeToValidate = nodeToValidate;
    	this.attributeDefinitions = attributeDefinitions;
    	this.stopOnError = stopOnError;
    }

    /**
     * Returns the validator report.
     *
     * @return The validator report.
     */
    public ValidatorReport getValidatorReport() {
    	return validatorReport;
    }

    /**
     * Checks if the validation was successful.
     *
     * @return {@code true} if the validation was successful.
     */
    public boolean isValid() {
    	return validatorReport.isValid();
    }

    /**
     * Validates the node.
     *
     * @return {@code true} if the validation was successful.
     */
    public boolean validate() {
    	this.validatorReport = new ValidatorReport();
    	boolean valid = true;
    	for(AttributeDefinition ad : attributeDefinitions){
    		String key = ad.getName();
    		AttributeValidator attributeValidator = new AttributeValidator(ad, nodeToValidate.get(key), stopOnError);
    		valid = attributeValidator.validate() && valid;
    		validatorReport.addAttributeValidatorReport(attributeValidator.getAttributeValidatorReport());
    		if(!valid && stopOnError){
    			return false;
    		}
    	}
    	return validatorReport.isValid();
    }

    

}
