package org.gcube.informationsystem.types.impl.properties;

import java.lang.reflect.Method;
import java.net.URI;
import java.net.URL;
import java.util.Objects;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonSetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonTypeName;
import org.gcube.informationsystem.types.PropertyTypeName;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.annotations.GetReturnType;
import org.gcube.informationsystem.types.reference.properties.PropertyDefinition;
import org.gcube.informationsystem.utils.AttributeUtility;
import org.gcube.informationsystem.utils.TypeUtility;
import org.gcube.informationsystem.utils.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The default implementation of the {@link PropertyDefinition} interface.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
// @JsonAutoDetect(fieldVisibility=JsonAutoDetect.Visibility.ANY)
@JsonTypeName(value=PropertyDefinition.NAME)
public final class PropertyDefinitionImpl implements PropertyDefinition {

	/**
	 * The serial version UID.
	 */
	private static final long serialVersionUID = -5925314595659292025L;

	/**
	 * The logger.
	 */
	private static Logger logger = LoggerFactory.getLogger(PropertyDefinitionImpl.class);
	
	/**
	 * The regex for UUIDs.
	 */
	public final static String UUID_REGEX = "^([a-fA-F0-9]{8}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{12}){1}$";
	/**
	 * The regex for URIs.
	 */
	public final static String URI_REGEX = null;
	/**
	 * The regex for URLs.
	 */
	public final static String URL_REGEX = null;
	
	/**
	 * The name of the property.
	 */
	private String name= "";
	/**
	 * The description of the property.
	 */
	private String description= "";
	/**
	 * Whether the property is mandatory.
	 */
	private boolean mandatory = false;
	/**
	 * Whether the property is readonly.
	 */
	private boolean readonly = false;
	/**
	 * Whether the property is not null.
	 */
	private boolean notnull = false;
	/**
	 * The maximum value of the property.
	 */
	private Integer max= null;
	/**
	 * The minimum value of the property.
	 */
	private Integer min= null;
	/**
	 * The regex of the property.
	 */
	private String regexp= null;
	/**
	 * The type name of the property.
	 */
	private PropertyTypeName propertyTypeName = null;
	/**
	 * The default value of the property.
	 */
	private Object defaultValue = null;

	/**
	 * Gets the property name from the method name.
	 *
	 * @param method the method
	 * @return the property name
	 */
	private static String getPropertyNameFromMethodName(Method method){
		String name = method.getName();
		if(name.startsWith("get")){
			name = name.replace("get", "");

		}
		if(name.startsWith("is")){
			name = name.replace("is", "");
		}
		
		if(name.length() > 0){
			name = Character.toLowerCase(name.charAt(0)) + (name.length() > 1 ? name.substring(1) : "");
		}
		
		return name;
	}
	
	/**
	 * Default constructor.
	 */
	protected PropertyDefinitionImpl() {
		
	}
	
	/**
	 * Constructs a new PropertyDefinitionImpl from an annotation and a method.
	 *
	 * @param propertyAnnotation the annotation
	 * @param method             the method
	 */
	public PropertyDefinitionImpl(ISProperty propertyAnnotation, Method method) {
		String name = propertyAnnotation.name().isEmpty()?getPropertyNameFromMethodName(method):propertyAnnotation.name();
		this.name = name;
		this.description = propertyAnnotation.description();
		this.mandatory= propertyAnnotation.mandatory();
		this.notnull = !propertyAnnotation.nullable();
		this.readonly = propertyAnnotation.readonly();
		if(propertyAnnotation.max()>0) {
			this.max = propertyAnnotation.max();
		}
		if(propertyAnnotation.max()>=propertyAnnotation.min() && propertyAnnotation.min()>0) {
			this.min = propertyAnnotation.min();
		}
		
		Class<?> clz = propertyAnnotation.type();
		if(clz==GetReturnType.class) {
			// We have to read the return of the method
			this.propertyTypeName = new PropertyTypeName(method);
			clz = method.getReturnType();
			logger.trace("Return Type for method {} is {}", method, clz);
		}else {
			this.propertyTypeName = new PropertyTypeName(clz);
		}
		
		String defaultValueAsString = propertyAnnotation.defaultValue();
		defaultValueAsString = AttributeUtility.evaluateNullForDefaultValue(defaultValueAsString);
		
		// The default value is evaluated to test if compliant with the declared type
		this.defaultValue = AttributeUtility.evaluateValueStringAccordingBaseType(propertyTypeName.getBaseType(), defaultValueAsString);
		
		if(!propertyAnnotation.regexpr().isEmpty()) {
			this.regexp = propertyAnnotation.regexpr();
		}else {
			if(Enum.class.isAssignableFrom(clz)){
				Object[] constants = clz.getEnumConstants();
				StringBuilder stringBuilder = new StringBuilder("^(");
				for(int i=0; i<constants.length; i++){
					stringBuilder.append(constants[i].toString());
					if(i<constants.length-1){
						stringBuilder.append("|");
					}
				}
				stringBuilder.append(")$");
				this.regexp = stringBuilder.toString();
			}
			if(UUID.class.isAssignableFrom(clz)){
				this.regexp = UUID_REGEX;
			}
			if(URI.class.isAssignableFrom(clz)){
				this.regexp = URI_REGEX;
			}
			if(URL.class.isAssignableFrom(clz)){
				this.regexp = URL_REGEX;
			}
			if(Version.class.isAssignableFrom(clz)){
				this.regexp = Version.VERSION_REGEX;
			}
		}

	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return name;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		this.name = name;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return description;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {
		this.description = description;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isMandatory() {
		return mandatory;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMandatory(boolean mandatory) {
		this.mandatory = mandatory;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isReadonly() {
		return readonly;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setReadonly(boolean readonly) {
		this.readonly = readonly;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isNotnull() {
		return notnull;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setNotnull(boolean notnull) {
		this.notnull = notnull;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getMax() {
		return max;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMax(Integer max) {
		this.max = max;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getMin() {
		return min;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMin(Integer min) {
		this.min = min;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getRegexp() {
		return regexp;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setRegexp(String regexp) {
		AttributeUtility.checkRegex(regexp, null);
		this.regexp = regexp;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPropertyType() {
		return propertyTypeName.toString();
	}

	/**
	 * Sets the property type.
	 *
	 * @param type the type
	 */
	@JsonSetter(value = PropertyDefinition.PROPERTY_TYPE_PROPERTY)
	public void setPropertyType(String type) {
		this.propertyTypeName = new PropertyTypeName(type);
	}
	
	/**
	 * Gets the property type name.
	 *
	 * @return the property type name
	 */
	@JsonIgnore
	public PropertyTypeName getPropertyTypeName() {
		return propertyTypeName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object getDefaultValue() {
		return defaultValue;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDefaultValue(Object defaultValue) {
		this.defaultValue = defaultValue;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return "PropertyDefinition ["
				+ "name=" + name
				+ ", description=" + description
				+ ", mandatory=" + mandatory
				+ ", readonly=" + readonly
				+ ", notnull=" + notnull
				+ ", max=" + max
				+ ", min=" + min
				+ ", regexpr=" + regexp
				+ ", type=" + propertyTypeName.toString()
				+ ", defaultValue=" + (defaultValue == null ? "null" : defaultValue.toString())
				+ "]";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int hashCode() {
		return Objects.hash(name, description, mandatory, readonly, notnull, max, min, regexp, propertyTypeName.toString(), defaultValue);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		PropertyDefinitionImpl other = (PropertyDefinitionImpl) obj;
		return Objects.equals(name, other.name) &&
				Objects.equals(description, other.description) &&
				mandatory == other.mandatory &&
				readonly == other.readonly &&
				notnull == other.notnull &&
				Objects.equals(max, other.max) &&
				Objects.equals(min, other.min) &&
				Objects.equals(regexp, other.regexp) &&
				Objects.equals(propertyTypeName.toString(), other.propertyTypeName.toString()) &&
				Objects.equals(defaultValue, other.defaultValue);
	}

	/**
	 * Compares two integers.
	 *
	 * @param thisInt  the first integer
	 * @param otherInt the second integer
	 * @return the result of the comparison
	 */
	protected int compareIntegers(Integer thisInt, Integer otherInt) {
		Integer thisInteger = thisInt == null ? Integer.MAX_VALUE : thisInt;
		Integer otherInteger = otherInt == null ? Integer.MAX_VALUE : otherInt;
		return thisInteger.compareTo(otherInteger);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public int compareTo(PropertyDefinition other) {
		if (this == other) {
			return 0;
		}
		if (other == null) {
			return -1;
		}
		if (getClass() != other.getClass()) {
			return -1;
		}
		
		PropertyDefinitionImpl o = (PropertyDefinitionImpl) other;
		
		int ret = 0;
		ret = name.compareTo(o.name);
		if(ret != 0) {
			return ret;
		}
		
		ret = description.compareTo(o.description);
		if(ret != 0) {
			return ret;
		}
		
		ret = Boolean.compare(mandatory, o.mandatory);
		if(ret != 0) {
			return ret;
		}
		
		ret = Boolean.compare(readonly, o.readonly);
		if(ret != 0) {
			return ret;
		}
		
		ret = Boolean.compare(notnull, o.notnull);
		if(ret != 0) {
			return ret;
		}
		
		ret = compareIntegers(max, o.max);
		if(ret != 0) {
			return ret;
		}
		
		ret = compareIntegers(min, o.min);
		if(ret != 0) {
			return ret;
		}
		
		if(regexp==null && o.regexp!=null) {
			return -1;
		}
		
		if(o.regexp==null && regexp!=null) {
			return -1;
		}
		
		if(!(regexp==null && o.regexp==null)) {
			ret = regexp.compareTo(o.regexp);
			if(ret != 0) {
				return ret;
			}
		}
		
		ret = propertyTypeName.toString().compareTo(o.propertyTypeName.toString());
		if(ret != 0) {
			return ret;
		}
		
		
		if(defaultValue==null && o.defaultValue!=null) {
			return -1;
		}
		
		if(o.defaultValue==null && defaultValue!=null) {
			return -1;
		}
		
		if(defaultValue!=null && o.defaultValue!=null) {
			if(defaultValue.getClass()!=o.defaultValue.getClass()) {
				return -1;
			}
			
			// TODO
		}
		
		return ret;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getTypeName() {
		return TypeUtility.getTypeName(this.getClass());
	}
	
}
