package org.gcube.informationsystem.types.impl;

import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.TypeVariable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.annotation.JsonGetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonInclude;
import org.gcube.com.fasterxml.jackson.annotation.JsonInclude.Include;
import org.gcube.com.fasterxml.jackson.annotation.JsonProperty;
import org.gcube.com.fasterxml.jackson.annotation.JsonSetter;
import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.entities.EntityElement;
import org.gcube.informationsystem.base.reference.properties.PropertyElement;
import org.gcube.informationsystem.base.reference.relations.RelationElement;
import org.gcube.informationsystem.model.reference.properties.Metadata;
import org.gcube.informationsystem.types.TypeMapper;
import org.gcube.informationsystem.types.annotations.Abstract;
import org.gcube.informationsystem.types.annotations.Final;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.impl.entities.EntityTypeImpl;
import org.gcube.informationsystem.types.impl.properties.PropertyDefinitionImpl;
import org.gcube.informationsystem.types.impl.properties.PropertyTypeImpl;
import org.gcube.informationsystem.types.impl.relations.RelationTypeImpl;
import org.gcube.informationsystem.types.reference.Type;
import org.gcube.informationsystem.types.reference.properties.PropertyDefinition;
import org.gcube.informationsystem.utils.TypeUtility;
import org.gcube.informationsystem.utils.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The default implementation of the {@link Type} interface.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public class TypeImpl implements Type {

	/**
	 * The logger.
	 */
	private static Logger logger = LoggerFactory.getLogger(TypeImpl.class);

	/**
	 * Generated Serial version UID
	 */
	private static final long serialVersionUID = -4333954207969059451L;
	/**
	 * The default changelog map.
	 */
	public static final Map<Version, String> DEFAULT_CHANGELOG_MAP;
	/**
	 * The default changelog map with the key as a string.
	 */
	private static final Map<String, String> DEFAULT_CHANGELOG_MAP_KEY_AS_STRING;
	
	static {
		DEFAULT_CHANGELOG_MAP = new HashMap<>();
		DEFAULT_CHANGELOG_MAP.put(new Version(Version.MINIMAL_VERSION_STRING), Version.MINIMAL_VERSION_DESCRIPTION);
		DEFAULT_CHANGELOG_MAP_KEY_AS_STRING = new HashMap<>();
		DEFAULT_CHANGELOG_MAP_KEY_AS_STRING.put(Version.MINIMAL_VERSION_STRING, Version.MINIMAL_VERSION_DESCRIPTION);
	}
	
	/**
	 * The UUID of the type.
	 */
	protected UUID uuid;
	/**
	 * The metadata of the type.
	 */
	protected Metadata metadata;
	
	/**
	 * The name of the type.
	 */
	protected String name;
	/**
	 * The description of the type.
	 */
	protected String description;
	/**
	 * The version of the type.
	 */
	protected Version version;
	/**
	 * The changelog of the type.
	 */
	@JsonProperty(value = CHANGELOG_PROPERTY, required = false)
	@JsonInclude(Include.NON_NULL)
	protected Map<Version, String> changelog;

	/**
	 * Whether the type is abstract.
	 */
	@JsonProperty(value = ABSTRACT_PROPERTY)
	protected boolean abstractType;
	
	/**
	 * Whether the type is final.
	 */
	@JsonProperty(value = FINAL_PROPERTY)
	protected boolean finalClass;
	
	/**
	 * The extended types.
	 */
	protected Set<String> extendedTypes;

	/**
	 * The properties of the type.
	 */
	protected Set<PropertyDefinition> properties;

	/**
	 * Retrieves the super classes of a type.
	 *
	 * @param <E>           the element type
	 * @param type          the type
	 * @param baseClass     the base class
	 * @param topSuperClass the top super class
	 * @return the set of super classes
	 */
	protected <E extends Element> Set<String> retrieveSuperClasses(Class<? extends E> type, Class<E> baseClass,
			String topSuperClass) {
		Set<String> interfaceList = new HashSet<>();

		if (type == baseClass) {
			if (topSuperClass != null) {
				interfaceList.add(topSuperClass);
			}
			return interfaceList;
		}

		Class<?>[] interfaces = type.getInterfaces();

		for (Class<?> interfaceClass : interfaces) {

			if (!baseClass.isAssignableFrom(interfaceClass)) {
				continue;
			}

			@SuppressWarnings("unchecked")
			Class<? extends Element> clz = (Class<? extends Element>) interfaceClass;
			interfaceList.add(TypeMapper.getType(clz));
		}

		return interfaceList;
	}

	/**
	 * Retrieves the list of properties of a class.
	 *
	 * @param clz the class
	 * @return the set of properties
	 */
	protected Set<PropertyDefinition> retrieveListOfProperties(Class<?> clz) {
		Set<PropertyDefinition> properties = new TreeSet<>();
		for (Method m : clz.getDeclaredMethods()) {
			m.setAccessible(true);
			if (m.isAnnotationPresent(ISProperty.class)) {
				if (m.isBridge()) {
					continue;
				}
				ISProperty propAnnotation = m.getAnnotation(ISProperty.class);
				PropertyDefinition prop = new PropertyDefinitionImpl(propAnnotation, m);
				properties.add(prop);
				logger.trace("Property {} retrieved in type {} ", prop, clz.getSimpleName());
			}

		}
		if(properties.size()==0) {
			properties = null;
		}
		return properties;
	}

	/**
	 * Gets the generic class of a type.
	 *
	 * @param type the type
	 * @return the generic class
	 */
	protected Class<?> getGenericClass(java.lang.reflect.Type type) {
		TypeVariable<?> typeVariable = (TypeVariable<?>) type;
		java.lang.reflect.Type[] bounds = typeVariable.getBounds();
		java.lang.reflect.Type t = bounds[0];
		if (t instanceof ParameterizedType) {
			ParameterizedType parameterizedType = (ParameterizedType) t;
			return (Class<?>) parameterizedType.getRawType();
		}
		return (Class<?>) t;
	}

	/**
	 * Gets an instance of a type.
	 *
	 * @param clz the class
	 * @return the type instance
	 */
	@SuppressWarnings({ "unchecked" })
	public static Type getInstance(Class<? extends Element> clz) {
		Type typeDefinition = null;
		try {
			if (EntityElement.class.isAssignableFrom(clz)) {
				typeDefinition = EntityTypeImpl.getEntityTypeDefinitionInstance((Class<? extends EntityElement>) clz);
				return typeDefinition;
			} else if (RelationElement.class.isAssignableFrom(clz)) {
				typeDefinition = RelationTypeImpl
						.getRelationTypeDefinitionInstance((Class<? extends RelationElement<?, ?>>) clz);
				return typeDefinition;
			} else if (PropertyElement.class.isAssignableFrom(clz)) {
				typeDefinition = new PropertyTypeImpl((Class<? extends PropertyElement>) clz);
				return typeDefinition;
			} else if (Type.class.isAssignableFrom(clz)) {
				typeDefinition = new TypeImpl(clz);
				return typeDefinition;
			} else {
				throw new RuntimeException("Serialization required");
			}
		} finally {
			if (typeDefinition != null) {
				logger.debug("{} : {} ", clz, typeDefinition);
			}
		}
	}

	/**
	 * Default constructor.
	 */
	protected TypeImpl() {
	}
	
	/**
	 * Constructs a new TypeImpl from a class.
	 *
	 * @param clz the class
	 */
	protected TypeImpl(Class<? extends Element> clz) {
		this.name = TypeMapper.getType(clz);
		this.description = TypeMapper.getTypeDescription(clz);
		this.version = TypeMapper.getTypeVersion(clz);

		this.changelog = TypeMapper.getTypeChangelog(clz);
		
		this.abstractType = false;

		if (clz.isAnnotationPresent(Abstract.class)) {
			this.abstractType = true;
		}

		if(clz.isAnnotationPresent(Final.class)) {
			this.finalClass = true;
		}
		
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public UUID getID() {
		return uuid;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setID(UUID uuid) {
		this.uuid = uuid;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Metadata getMetadata() {
		return metadata;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMetadata(Metadata metadata) {
		this.metadata = metadata;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return name;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return description;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Version getVersion() {
		return version;
	}

	/**
	 * Gets the version as a string.
	 *
	 * @return the version as a string
	 */
	@JsonGetter(value = VERSION_PROPERTY)
	public String getVersionAsString() {
		return version.toString();
	}

	/**
	 * Sets the version from a string.
	 *
	 * @param version the version as a string
	 */
	@JsonSetter(value = VERSION_PROPERTY)
	public void setVersion(String version) {
		this.version = new Version(version);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Map<Version, String> getChangelog() {
		return changelog;
	}

	/**
	 * Gets the changelog with the version as a string.
	 *
	 * @return the changelog with the version as a string
	 */
	@JsonGetter(value = CHANGELOG_PROPERTY)
	@JsonInclude(Include.NON_NULL)
	public Map<String, String> getChangelogWithVersionAsString() {
		if(this.changelog==null) {
			return DEFAULT_CHANGELOG_MAP_KEY_AS_STRING;
		}
		Map<String, String> map = new HashMap<>();
		for (Version typeVersion : changelog.keySet()) {
			map.put(typeVersion.toString(), changelog.get(typeVersion));
		}
		return map;
	}

	/**
	 * Sets the changelog from a map with the version as a string.
	 *
	 * @param changelog the changelog with the version as a string
	 */
	@JsonSetter(value=CHANGELOG_PROPERTY)
	public void setChangelog(Map<String, String> changelog) {
		if(changelog==null) {
			this.changelog = DEFAULT_CHANGELOG_MAP;
			return;
		}
		this.changelog = new HashMap<>();
		for (String version : changelog.keySet()) {
			this.changelog.put(new Version(version), changelog.get(version));
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isAbstract() {
		return abstractType;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isFinal() {
		return finalClass;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Set<String> getExtendedTypes() {
		return extendedTypes;
	}

	/**
	 * Gets the properties of the type.
	 *
	 * @return the properties of the type
	 */
	@JsonInclude(Include.NON_EMPTY)
	public Set<PropertyDefinition> getProperties() {
		return properties;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	@JsonIgnore
	public AccessType getAccessType() {
		return null;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getTypeName() {
		return TypeUtility.getTypeName(this.getClass());
	}
	
}
