package org.gcube.informationsystem.types;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.gcube.com.fasterxml.jackson.annotation.JsonTypeName;
import org.gcube.com.fasterxml.jackson.databind.DeserializationFeature;
import org.gcube.com.fasterxml.jackson.databind.JavaType;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.types.annotations.Deserialize;
import org.gcube.informationsystem.types.impl.TypeImpl;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.Changelog;
import org.gcube.informationsystem.types.reference.Type;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.types.reference.entities.EntityType;
import org.gcube.informationsystem.types.reference.entities.FacetType;
import org.gcube.informationsystem.types.reference.entities.ResourceType;
import org.gcube.informationsystem.types.reference.properties.LinkedEntity;
import org.gcube.informationsystem.types.reference.properties.PropertyDefinition;
import org.gcube.informationsystem.types.reference.properties.PropertyType;
import org.gcube.informationsystem.types.reference.relations.ConsistsOfType;
import org.gcube.informationsystem.types.reference.relations.IsRelatedToType;
import org.gcube.informationsystem.types.reference.relations.RelationType;
import org.gcube.informationsystem.utils.Version;

/**
 * A utility class for mapping between Java classes and {@link Type} definitions,
 * and for serializing/deserializing these definitions to/from JSON.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public class TypeMapper {

	/** The object mapper. */
	protected static final ObjectMapper mapper;
	
	static {
		mapper = new ObjectMapper();
		mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		mapper.configure(DeserializationFeature.FAIL_ON_MISSING_CREATOR_PROPERTIES, false);
		
		mapper.registerSubtypes(Type.class);
		
		mapper.registerSubtypes(EntityType.class);
		mapper.registerSubtypes(ResourceType.class);
		mapper.registerSubtypes(FacetType.class);
		
		mapper.registerSubtypes(RelationType.class);
		mapper.registerSubtypes(IsRelatedToType.class);
		mapper.registerSubtypes(ConsistsOfType.class);
		
		mapper.registerSubtypes(PropertyType.class);
		mapper.registerSubtypes(PropertyDefinition.class);
		mapper.registerSubtypes(LinkedEntity.class);
	}
	
	/**
	 * Serializes a {@link Type} definition to a JSON string.
	 *
	 * @param type The type definition.
	 * @return The JSON string.
	 * @throws Exception if an error occurs during serialization.
	 */
	public static String serializeTypeDefinition(Type type) throws Exception{
		String json = mapper.writeValueAsString(type);
		return json;
	}

	/**
	 * Deserializes a {@link Type} definition from a JSON string.
	 *
	 * @param json The JSON string.
	 * @return The deserialized type definition.
	 * @throws Exception if an error occurs during deserialization.
	 */
	public static Type deserializeTypeDefinition(String json) throws Exception {
		Type type = mapper.readValue(json, Type.class);
		return type;
	}
	
	/**
	 * Serializes a list of {@link Type} definitions to a JSON string.
	 *
	 * @param typeDefinitions The list of type definitions.
	 * @return The JSON string.
	 * @throws Exception if an error occurs during serialization.
	 */
	public static String serializeTypeDefinitions(List<Type> typeDefinitions) throws Exception{
		JavaType javaType = mapper.getTypeFactory().constructCollectionType(List.class, Type.class);
		return mapper.writerFor(javaType).writeValueAsString(typeDefinitions);
	}
	
	/**
	 * Deserializes a list of {@link Type} definitions from a JSON string.
	 *
	 * @param json The JSON string.
	 * @return The deserialized list of type definitions.
	 * @throws Exception if an error occurs during deserialization.
	 */
	public static List<Type> deserializeTypeDefinitions(String json) throws Exception{
		JavaType javaType = mapper.getTypeFactory().constructCollectionType(ArrayList.class, Type.class);
		return mapper.readValue(json, javaType);
	}
	
	
	
	// TODO move somewhere else, probably in Element
	
	/**
	 * Creates a {@link Type} definition from a given {@link Element} class.
	 *
	 * @param <E> The type of the element.
	 * @param clz The element class.
	 * @return The generated type definition.
	 */
	public static <E extends Element> Type createTypeDefinition(Class<E> clz) {
		Type type = TypeImpl.getInstance(clz);
		return type;
	}
	
	/**
	 * Serializes the {@link Type} definition of a given {@link Element} class to JSON.
	 *
	 * @param <E> The type of the element.
	 * @param clz The element class.
	 * @return The JSON string representation of the type.
	 * @throws Exception if an error occurs during serialization.
	 */
	public static <E extends Element> String serializeType(Class<E> clz) throws Exception{
		Type type = createTypeDefinition(clz);
		return serializeTypeDefinition(type);
	}
	
	/**
	 * Returns the type name of an {@link Element} instance.
	 *
	 * @param <E> The type of the element.
	 * @param e   The element instance.
	 * @return The type name.
	 */
	public static <E extends Element> String getType(E e){
		return getType(e.getClass());
	}
	
	/**
	 * Gets the dynamic implementation class for a given interface, if specified
	 * via the {@link Deserialize} annotation.
	 *
	 * @param <E>     The type of the element.
	 * @param <EImpl> The type of the implementation.
	 * @param clz     The interface class.
	 * @return The implementation class, or {@code null}.
	 */
	public static <E extends Element, EImpl extends E> Class<EImpl> getDynamicImplementation(Class<E> clz){
		if(clz.isInterface() && clz.isAnnotationPresent(Deserialize.class)) {
			Deserialize deserialize = clz.getAnnotation(Deserialize.class);
			Class<?> annotatedClass = deserialize.as();
			if(clz.isAssignableFrom(annotatedClass) && !annotatedClass.isInterface()) {
				@SuppressWarnings("unchecked")
				Class<EImpl> implementationClz = (Class<EImpl>) annotatedClass;
				return implementationClz;
			}
		}
		return null;
	}
	
	/**
	 * Returns the type name of an {@link Element} class, derived from annotations.
	 *
	 * @param clz The element class.
	 * @return The type name.
	 */
	public static String getType(Class<? extends Element> clz){
		String classSimpleName = clz.getSimpleName();
		String name = null;
		
		if(!clz.isInterface()) {
			if(clz.isAnnotationPresent(JsonTypeName.class)) {
				JsonTypeName jsonTypeName = clz.getAnnotation(JsonTypeName.class);
				name = jsonTypeName.value();
				if(name==null || name.compareTo("")==0) {
					throw new RuntimeException("Invalid annotation @JsonTypeName for type " + classSimpleName);
				}
				return name;
			}
		}
		
		if(clz.isAnnotationPresent(TypeMetadata.class)) {
			TypeMetadata typeMetadata = clz.getAnnotation(TypeMetadata.class);
			String typeMetadataName = typeMetadata.name();
			if(typeMetadataName!=null && typeMetadataName.compareTo("")!=0) {
				/*
				if(name!=null && typeMetadataName.compareTo(name)!=0) {
					throw new RuntimeException("Name in annotation @TypeMetadata differ from annotation in @JsonTypeName for type " + classSimpleName + ". Please be coerent");
				}
				*/
				return typeMetadataName;
			}else {
				throw new RuntimeException("Invalid Name in annotation @TypeMetadata for type " + classSimpleName);
			}
		}else {
			throw new RuntimeException("You must provide @TypeMetadata for " + classSimpleName);
		}
		
	}
	
	/**
	 * Returns the description of an {@link Element} class from its {@link TypeMetadata}.
	 *
	 * @param clz The element class.
	 * @return The type description.
	 */
	public static String getTypeDescription(Class<? extends Element> clz){
		String classSimpleName = clz.getSimpleName();
		if(clz.isAnnotationPresent(TypeMetadata.class)) {
			TypeMetadata typeMetadata = clz.getAnnotation(TypeMetadata.class);
			String description = typeMetadata.description();
			if(description!=null && description.compareTo("")!=0) {
				return description;
			}else {
				throw new RuntimeException("Invalid Description in annotation @TypeMetadata for type " + classSimpleName);
			}
		}else {
			throw new RuntimeException("You must provide @TypeMetadata for " + classSimpleName);
		}
	}
	
	/**
	 * Returns the version of an {@link Element} class from its {@link TypeMetadata}.
	 *
	 * @param clz The element class.
	 * @return The type version.
	 */
	public static Version getTypeVersion(Class<? extends Element> clz){
		String classSimpleName = clz.getSimpleName();
		if(clz.isAnnotationPresent(TypeMetadata.class)) {
			TypeMetadata typeMetadata = clz.getAnnotation(TypeMetadata.class);
			return new Version(typeMetadata.version());
		}else {
			throw new RuntimeException("You must provide @TypeMetadata for " + classSimpleName);
		}
	}
	
	/**
	 * Returns the changelog of an {@link Element} class from its {@link Changelog} annotation.
	 *
	 * @param clz The element class.
	 * @return A map of versions to change descriptions.
	 */
	public static Map<Version, String> getTypeChangelog(Class<? extends Element> clz){
		Map<Version, String> map = new HashMap<>();
		
		if(clz.isAnnotationPresent(Changelog.class)) {
			Changelog changelog = clz.getAnnotation(Changelog.class);
			Change[] changes = changelog.value();
			for(Change change : changes) {
				String version = change.version();
				Version typeVersion = new Version(version);
				if(map.containsKey(typeVersion)) {
					throw new RuntimeException("Duplicated version " + version +" in @Change annotation");
				}
				
				String description = change.description();
				if(description==null || description.compareTo("")==0) {
					throw new RuntimeException("A valid description for version " + version +" must be provided in @Change annotation");
				}
				
				map.put(typeVersion, description);
			}
		}
		
		if(!map.containsKey(Version.MINIMAL_VERSION)) {
			map.putAll(TypeImpl.DEFAULT_CHANGELOG_MAP);
		}
		
		Version typeVersion = getTypeVersion(clz);
		if (!map.containsKey(typeVersion)) {
			throw new RuntimeException("The Type " + clz.getSimpleName() + " does not provided the appropriated changelog Map");
		}

		List<Version> versions = new ArrayList<>(map.keySet());
		versions.sort(new Comparator<Version>() {

			/**
			 * {@inheritDoc}
			 */
			@Override
			public int compare(Version o1, Version o2) {
				/* o2.compareTo(o1) and not vice-versa
				 * because we want descending order
				 */
				return o2.compareTo(o1);
			}
		});
		
		if(versions.get(0).compareTo(typeVersion)!=0) {
			throw new RuntimeException("The Type declared version (i.e."+ typeVersion.toString() +") does not match the highest version declared in changelog (i.e. "+ versions.get(0) + "). Please fix your type.");
		}
		
		return map;
	}
	
}
