package org.gcube.informationsystem.tree;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Represents a generic tree structure.
 *
 * @param <T> The type of the elements stored in the tree.
 * @author Luca Frosini (ISTI - CNR)
 */
public class Tree<T> {

	private boolean allowMultipleInheritance;
	
	private Node<T> rootNode;
	private NodeInformation<T> ni;
	private Map<String, Node<T>> locate;
	
	/**
	 * Constructs an empty tree.
	 */
	public Tree() {
		this.allowMultipleInheritance = true;
		this.locate = new HashMap<>();
	}
	
	/**
	 * Constructs a tree with the given node information provider.
	 *
	 * @param ni The node information provider.
	 */
	public Tree(NodeInformation<T> ni) {
		this();
		setNodeInformation(ni);
	}
	
	/**
	 * Constructs a tree with the given root element and node information provider.
	 *
	 * @param root The root element.
	 * @param ni   The node information provider.
	 */
	public Tree(T root, NodeInformation<T> ni)  {
		this(ni);
		setRoot(root);
	}
	
	/**
	 * Sets the node information provider for this tree, if not already set.
	 *
	 * @param ni The node information provider.
	 */
	public void setNodeInformation(NodeInformation<T> ni) {
		if(this.ni==null) {
			this.ni = ni;
		}
	}
	
	/**
	 * Sets the root element of this tree, if not already set.
	 *
	 * @param root The root element.
	 * @throws RuntimeException if the node information provider has not been set.
	 */
	public void setRoot(T root) throws RuntimeException {
		if(this.ni==null) {
			throw new RuntimeException("You must set the NodeInformation instance first");
		}
		if(this.rootNode==null) {
			this.rootNode = new Node<>(root);
			this.rootNode.setTree(this);
			String identifier = ni.getIdentifier(root);
			this.locate.put(identifier, rootNode);
		}
	}
	
	/**
	 * Sets whether multiple inheritance is allowed in this tree.
	 *
	 * @param allowMultipleInheritance {@code true} to allow multiple inheritance.
	 */
	public void setAllowMultipleInheritance(boolean allowMultipleInheritance) {
		this.allowMultipleInheritance = allowMultipleInheritance;
	}
	
	/**
	 * Returns the node information provider for this tree.
	 *
	 * @return The node information provider.
	 */
	public NodeInformation<T> getNodeInformation() {
		return ni;
	}
	
	/**
	 * Adds a new node to the tree.
	 *
	 * @param t The element to add.
	 * @return The newly created or existing node for the element.
	 */
	public Node<T> addNode(T t) {
		String identifier = ni.getIdentifier(t);
		if(locate.containsKey(identifier)) {
			// has been already added
			return locate.get(identifier);
		}
		
		Node<T> node = new Node<>(t);
		node.setTree(this);
		
		Set<String> parentIdentifiers = ni.getParentIdentifiers(rootNode!= null ? rootNode.getNodeElement() : null, t);
		if(parentIdentifiers==null || parentIdentifiers.size()==0) {
			if(this.rootNode==null) {
				this.rootNode = node;
			}else {
				throw new RuntimeException("A Tree cannot have two root. " + t.toString() + " has not parent.");
			}
		} else {
			for(String parentIdentifier : parentIdentifiers) {
				Node<T> parentNode = locate.get(parentIdentifier);
				if(parentNode==null) {
					throw new RuntimeException("I can find parent for " + identifier + ". Missing parent is " + parentIdentifier);
				}
				
				parentNode.addChild(node);
				
				if(!allowMultipleInheritance) {
					break;
				}
			}
		}
		
		this.locate.put(identifier, node);
		
		return node;
	}

	/**
	 * Returns the root node of the tree.
	 *
	 * @return The root node.
	 */
	public Node<T> getRootNode() {
		return rootNode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return rootNode.toString();
	}

	/**
	 * Traverses the tree and applies the given elaborator to each node.
	 *
	 * @param nodeElaborator The elaborator to apply.
	 * @throws Exception if an error occurs during elaboration.
	 */
	public void elaborate(NodeElaborator<T> nodeElaborator) throws Exception {
		rootNode.elaborate(nodeElaborator);
	}
	
	/**
	 * Returns the node with the given identifier.
	 *
	 * @param identifier The identifier of the node.
	 * @return The node, or {@code null} if not found.
	 */
	public Node<T> getNodeByIdentifier(String identifier){
		return locate.get(identifier);
	}


	/**
	 * Checks if a node is a child of another node.
	 *
	 * @param referenceIdentifier The identifier of the potential parent node.
	 * @param requiredChild       The identifier of the potential child node.
	 * @return {@code true} if the second node is a child of the first.
	 */
	public boolean isChildOf(String referenceIdentifier, String requiredChild){
		Node<T> referenceNode = locate.get(referenceIdentifier);
		Node<T> requiredChildNode = locate.get(requiredChild);
		if(referenceNode==null || requiredChildNode==null) {
			return false;
		}
		return referenceNode.getDescendants().contains(requiredChildNode);
	}

	/**
	 * Checks if a node is a parent of another node.
	 *
	 * @param referenceIdentifier The identifier of the potential child node.
	 * @param requiredParent      The identifier of the potential parent node.
	 * @return {@code true} if the second node is a parent of the first.
	 */
	public boolean isParentOf(String referenceIdentifier, String requiredParent){
		return isChildOf(requiredParent, referenceIdentifier);
	}

}