package org.gcube.informationsystem.tree;

import java.util.Comparator;
import java.util.Set;
import java.util.TreeSet;

/**
 * Represents a node in a {@link Tree}.
 *
 * @param <T> The type of the element stored in the node.
 * @author Luca Frosini (ISTI - CNR)
 */
public class Node<T> implements Comparable<Node<T>> {

	/** The string used for indentation when printing the tree. */
	public static String INDENTATION = "\t";
	
	/** The element stored in this node. */
	protected T t;
	/** The tree to which this node belongs. */
	protected Tree<T> tree;
	/** The elements of the children of this node. */
	protected Set<T> childrenElements;
	/** The elements of all descendants of this node. */
	protected Set<T> descendantElements;
	
	/** The parent of this node. */
	protected Node<T> parent;
	/** The children of this node. */
	protected Set<Node<T>> children;
	/** All descendants of this node. */
	protected Set<Node<T>> descendants;
	
	/**
	 * Constructs a new node with the given element.
	 *
	 * @param t The element to store in the node.
	 */
	public Node(T  t) {
		this.t = t;
		this.children = new TreeSet<>();
	}
	
	/**
	 * Returns the tree to which this node belongs.
	 *
	 * @return The parent tree.
	 */
	public Tree<T> getTree() {
		return tree;
	}

	/**
	 * Sets the tree to which this node belongs.
	 *
	 * @param tree The parent tree.
	 */
	void setTree(Tree<T> tree) {
		this.tree = tree;
	}
	
	/**
	 * Returns the element stored in this node.
	 *
	 * @return The node's element.
	 */
	public T getNodeElement() {
		return t;
	}
	
	/**
	 * Returns the parent of this node.
	 *
	 * @return The parent node, or {@code null} if this is the root.
	 */
	public Node<T> getParent() {
		return parent;
	}
	
	/**
	 * Sets the parent of this node.
	 *
	 * @param parent The parent node.
	 * @return This node.
	 */
	public Node<T> setParent(Node<T> parent) {
		this.parent = parent;
		return this;
	}

	/**
	 * Adds a child to this node.
	 *
	 * @param child The child node to add.
	 * @return This node.
	 */
	public Node<T> addChild(Node<T> child) {
		children.add(child);
		child.setParent(this);
		child.setTree(tree);
		return this;
	}

	/**
	 * Returns the identifier of this node's element.
	 *
	 * @return The identifier.
	 */
	public String getIdentifier() {
		return tree.getNodeInformation().getIdentifier(t);
	}

	/**
	 * Returns the children of this node.
	 *
	 * @return A set of child nodes.
	 */
	public Set<Node<T>> getChildren() {
		return children;
	}

	/**
	 * Returns all descendants of this node.
	 *
	 * @return A set of all descendant nodes.
	 */
	public Set<Node<T>> getDescendants() {
		if(descendants == null) {
			descendants = new TreeSet<>();
			for (Node<T> child : children) {
				descendants.add(child);
				descendants.addAll(child.getDescendants());
			}
		}
		return descendants;
	}
	
	/**
	 * Returns the elements of the children of this node.
	 *
	 * @return A set of the elements of the child nodes.
	 */
	public Set<T> getChildrenElements() {
		if(childrenElements == null) {
			NodeInformation<T> ni = tree.getNodeInformation();
			childrenElements = new TreeSet<>(new Comparator<T>() {
				/**
				 * {@inheritDoc}
				 */
				@Override
				public int compare(T t1, T t2) {
					return ni.getIdentifier(t1).compareTo(ni.getIdentifier(t2));
				}
			});
			for (Node<T> child : this.children) {
				childrenElements.add(child.t);
			}
		}
		return childrenElements;
	}

	/**
	 * Returns the elements of all descendants of this node.
	 *
	 * @return A set of the elements of all descendant nodes.
	 */
	public Set<T> getDescendantElements() {
		if(descendantElements == null) {
			descendantElements = new TreeSet<>(new Comparator<T>() {
				/**
				 * {@inheritDoc}
				 */
				@Override
				public int compare(T t1, T t2) {
					NodeInformation<T> ni = tree.getNodeInformation();
					return ni.getIdentifier(t1).compareTo(ni.getIdentifier(t2));
				}
			});
			for (Node<T> child : children) {
				descendantElements.add(child.t);
				descendantElements.addAll(child.getDescendantElements());
			}
		}
		return descendantElements;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return printTree(0).toString();
	}

	/**
	 * Prints the tree structure starting from this node.
	 *
	 * @param level The indentation level.
	 * @return A string buffer containing the tree representation.
	 */
	private StringBuffer printTree(int level) {
		
		StringBuffer stringBuffer = new StringBuffer();
		
		NodeElaborator<T> nodeElaborator = new NodeElaborator<T>() {

			/**
			 * {@inheritDoc}
			 */
			@Override
			public void elaborate(Node<T> node, int level) throws Exception {
				for (int i = 0; i < level; ++i) {
					stringBuffer.append(Node.INDENTATION);
				}
				stringBuffer.append(tree.getNodeInformation().getIdentifier(node.getNodeElement()));
				stringBuffer.append("\n");
			}
						
		};
		
		try {
			elaborate(nodeElaborator, level);
		}catch (Exception e) {
			throw new RuntimeException(e);
		}
		
		return stringBuffer;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int compareTo(Node<T> other) {
		if (this == other) {
			return 0;
		}
		if (other == null) {
			return -1;
		}
		if (getClass() != other.getClass()) {
			return -1;
		}
		
		return this.getIdentifier().compareTo(other.getIdentifier());
	}
	
	
	/**
	 * Traverses the subtree starting from this node and applies the given elaborator.
	 *
	 * @param nodeElaborator The elaborator to apply to each node.
	 * @throws Exception if an error occurs during elaboration.
	 */
	public void elaborate(NodeElaborator<T> nodeElaborator) throws Exception {
		elaborate(nodeElaborator, 0);
	}
	
	/**
	 * Traverses the subtree starting from this node and applies the given elaborator.
	 *
	 * @param nodeElaborator The elaborator to apply to each node.
	 * @param level          The current level in the tree.
	 * @throws Exception if an error occurs during elaboration.
	 */
	protected void elaborate(NodeElaborator<T> nodeElaborator, int level) throws Exception {
		nodeElaborator.elaborate(this, level);
		for (Node<T> child : children) {
			child.elaborate(nodeElaborator, level+1);
		}
	}
}
