package org.gcube.informationsystem.serialization;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.gcube.com.fasterxml.jackson.core.JsonGenerationException;
import org.gcube.com.fasterxml.jackson.core.JsonParseException;
import org.gcube.com.fasterxml.jackson.core.JsonProcessingException;
import org.gcube.com.fasterxml.jackson.databind.DeserializationFeature;
import org.gcube.com.fasterxml.jackson.databind.JavaType;
import org.gcube.com.fasterxml.jackson.databind.JsonMappingException;
import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.com.fasterxml.jackson.databind.SerializationFeature;
import org.gcube.com.fasterxml.jackson.databind.exc.InvalidTypeIdException;
import org.gcube.com.fasterxml.jackson.databind.module.SimpleAbstractTypeResolver;
import org.gcube.com.fasterxml.jackson.databind.module.SimpleModule;
import org.gcube.com.fasterxml.jackson.databind.node.ArrayNode;
import org.gcube.com.fasterxml.jackson.databind.node.JsonNodeType;
import org.gcube.com.fasterxml.jackson.databind.node.ObjectNode;
import org.gcube.com.fasterxml.jackson.databind.node.TextNode;
import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.discovery.Discovery;
import org.gcube.informationsystem.discovery.knowledge.Knowledge;
import org.gcube.informationsystem.model.reference.ModelElement;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.properties.Property;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.types.TypeMapper;
import org.gcube.informationsystem.types.reference.Type;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A central class for handling JSON serialization and deserialization of
 * {@link Element} types.
 * <p>
 * This class configures a Jackson {@link ObjectMapper} with custom settings
 * for the Information System model, including subtype registration and fallback
 * mechanisms for unknown types.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
@SuppressWarnings("unchecked")
public class ElementMapper {

	/** The logger. */
	private static Logger logger = LoggerFactory.getLogger(ElementMapper.class);

	private static boolean initialized = false;


	/** The object mapper. */
	protected static final ObjectMapper mapper;

	/** A map of known types, keyed by type name. */
	protected static final Map<String, Class<? extends Element>> knownTypes;

	/** A map of interfaces to their implementation classes. */
	protected static final Map<Class<? extends Element>, Class<? extends Element>> interfaceToImplementation;

	/**
	 * Returns the configured {@link ObjectMapper} instance.
	 *
	 * @return The singleton {@code ObjectMapper}.
	 */
	public static ObjectMapper getObjectMapper() {
		return mapper;
	}

	static {
		logger.debug("Initializing {} in ClassLoader {}",
				ElementMapper.class.getSimpleName(),
				Thread.currentThread().getContextClassLoader());

		mapper = new ObjectMapper();
		mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		mapper.getSerializationConfig().with(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS);

		/*
		* Instructing Jackson to serialize any Date in the defined pattern
		*/
		SimpleDateFormat sdf = new SimpleDateFormat(Element.DATETIME_PATTERN);
		mapper.setDateFormat(sdf);

		knownTypes = new HashMap<>();
		interfaceToImplementation = new HashMap<>();

		Set<Package> packages = new HashSet<Package>();

		Class<Type> tdClz = Type.class;
		ElementMapper.registerSubtype(tdClz);
		packages.add(tdClz.getPackage());

		AccessType[] accessTypes = AccessType.values();
		for (AccessType accessType : accessTypes) {
			@SuppressWarnings("rawtypes")
			Class clz = accessType.getTypeClass();
			if (!Type.class.isAssignableFrom(clz)) {
				Class<? extends Element> dynamicImplementationClz = TypeMapper.getDynamicImplementation(clz);
				if (dynamicImplementationClz != null) {
					ElementMapper.addDynamicAssociation(clz, dynamicImplementationClz);
				} else {
					Class<Element> dummyClz = accessType.getDummyImplementationClass();
					if (dummyClz != null) {
						ElementMapper.registerSubtype(dummyClz);
					} else {
						ElementMapper.registerSubtype(clz);
					}
					packages.add(clz.getPackage());
				}
			}

		}

		Set<AccessType> baseTypes = new LinkedHashSet<>();
		baseTypes.add(AccessType.PROPERTY_ELEMENT);
		baseTypes.add(AccessType.ENTITY_ELEMENT);
		baseTypes.add(AccessType.RELATION_ELEMENT);

		for (AccessType accessType : baseTypes) {
			Class<Element> clz = accessType.getTypeClass();
			try {
				@SuppressWarnings("rawtypes")
				Discovery discovery = new Discovery<>(clz);
				switch (accessType) {
					case PROPERTY_ELEMENT:
						discovery.addStopClass(Property.class);
						break;
				
					case ENTITY_ELEMENT:
						discovery.addStopClass(Resource.class);
						discovery.addStopClass(Facet.class);
						break;
					
					case RELATION_ELEMENT:
						discovery.addStopClass(IsRelatedTo.class);
						discovery.addStopClass(ConsistsOf.class);
						break;

					default:
						break;
				}

				discovery.addPackages(packages);
				discovery.addDiscoveredElementActions(new ElementMappingAction());
				discovery.discover();
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
		}

		try {
			Knowledge.getInstance();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}

		for (@SuppressWarnings("rawtypes")
		Class clz : interfaceToImplementation.keySet()) {
			Class<? extends Element> implClz = interfaceToImplementation.get(clz);
			registerSubtypes(clz, implClz);
		}	
	}

	/**
	 * Adds a dynamic mapping between an interface and its implementation class.
	 *
	 * @param <El>                The interface type.
	 * @param <ELImpl>            The implementation type.
	 * @param interfaceClz        The interface class.
	 * @param implementationClass The implementation class.
	 */
	public static <El extends Element, ELImpl extends El> void addDynamicAssociation(Class<El> interfaceClz,
			Class<ELImpl> implementationClass) {
		interfaceToImplementation.put(interfaceClz, implementationClass);
	}

	/**
	 * Registers a subtype with the object mapper.
	 *
	 * @param <El>                The element type.
	 * @param <ELImpl>            The implementation type.
	 * @param interfaceClz        The interface class.
	 * @param implementationClass The implementation class.
	 */
	protected static <El extends Element, ELImpl extends El> void registerSubtypes(Class<El> interfaceClz,
			Class<ELImpl> implementationClass) {
		String typeName = TypeMapper.getType(interfaceClz);

		SimpleModule module = new SimpleModule(typeName);
		SimpleAbstractTypeResolver resolver = new SimpleAbstractTypeResolver();
		resolver.addMapping(interfaceClz, implementationClass);
		module.setAbstractTypes(resolver);

		mapper.registerModule(module);
		mapper.registerSubtypes(interfaceClz);
	}

	/**
	 * Registers a subtype with the object mapper.
	 *
	 * @param <El> The element type.
	 * @param clz  The class to register.
	 */
	public static <El extends Element> void registerSubtype(Class<El> clz) {
		String typeName = TypeMapper.getType(clz);
		knownTypes.put(typeName, clz);
		mapper.registerSubtypes(clz);
	}

	/**
	 * Serializes an object to an {@link OutputStream}.
	 *
	 * @param <T>    The type of the output stream.
	 * @param <El>   The type of the element.
	 * @param object The object to serialize.
	 * @param stream The output stream.
	 * @return The output stream.
	 * @throws IOException             if an error occurs during serialization.
	 * @throws JsonGenerationException if an error occurs during JSON generation.
	 * @throws JsonMappingException    if an error occurs during JSON mapping.
	 */
	public static <T extends OutputStream, El extends Element> T marshal(El object, T stream)
			throws JsonGenerationException, JsonMappingException, IOException {
		mapper.writeValue(stream, object);
		return stream;
	}

	/**
	 * Serializes an object to a {@link Writer}.
	 *
	 * @param <T>    The type of the writer.
	 * @param <El>   The type of the element.
	 * @param object The object to serialize.
	 * @param writer The writer.
	 * @return The writer.
	 * @throws IOException             if an error occurs during serialization.
	 * @throws JsonGenerationException if an error occurs during JSON generation.
	 * @throws JsonMappingException    if an error occurs during JSON mapping.
	 */
	public static <T extends Writer, El extends Element> T marshal(El object, T writer)
			throws JsonGenerationException, JsonMappingException, IOException {
		mapper.writeValue(writer, object);
		return writer;
	}

	/**
	 * Serializes an object to a JSON string.
	 *
	 * @param <El>   The type of the element.
	 * @param object The object to serialize.
	 * @return The JSON string.
	 * @throws JsonProcessingException if an error occurs during serialization.
	 */
	public static <El extends Element> String marshal(El object) throws JsonProcessingException {
		return mapper.writeValueAsString(object);
	}

	/**
	 * Serializes a list of objects to a JSON string.
	 *
	 * @param <El> The type of the elements in the list.
	 * @param list The list to serialize.
	 * @return The JSON string.
	 * @throws JsonProcessingException if an error occurs during serialization.
	 */
	public static <El extends Element> String marshal(List<El> list) throws JsonProcessingException {
		JavaType type = mapper.getTypeFactory().constructCollectionType(List.class, Element.class);
		return mapper.writerFor(type).writeValueAsString(list);
	}

	/**
	 * Serializes an array of objects to a JSON string.
	 *
	 * @param <El>  The type of the elements in the array.
	 * @param array The array to serialize.
	 * @return The JSON string.
	 * @throws JsonProcessingException if an error occurs during serialization.
	 */
	public static <El extends Element> String marshal(El[] array) throws JsonProcessingException {
		return mapper.writeValueAsString(array);
	}

	/**
	 * Returns an error message for an unknown type.
	 *
	 * @param unknownType The unknown type.
	 * @return The error message.
	 */
	protected static StringBuffer getError(String unknownType) {
		StringBuffer stringBuffer = new StringBuffer();
		stringBuffer.append(unknownType);
		stringBuffer.append(" is an unknown type. Please provide ");
		stringBuffer.append(ModelElement.SUPERTYPES_PROPERTY);
		stringBuffer.append(" property as string array to allow to instantiate the most appropriated class.");
		return stringBuffer;
	}

	/**
	 * Sets the type of an object to the best available supertype.
	 *
	 * @param objectNode The object node.
	 * @return The object node with the updated type.
	 */
	protected static ObjectNode setTypeToBestAvailable(ObjectNode objectNode) {
		String unknownType = objectNode.get(Element.TYPE_PROPERTY).asText();
		ArrayNode arrayNode = (ArrayNode) objectNode.get(ModelElement.SUPERTYPES_PROPERTY);

		String candidatedSupertype = null;
		if (arrayNode != null) {
			for (int i = 0; i < arrayNode.size(); i++) {
				String superType = arrayNode.get(i).asText();
				if (knownTypes.containsKey(superType)) {
					candidatedSupertype = superType;
					try {
						// Checking if it is one of the base type. In some cases we need to use dummy
						// implementation
						AccessType accessType = AccessType.getAccessType(superType);
						// It is one of the BaseType.
						// Looking if we need to set the dummy implementation class
						if (accessType.getDummyImplementationClass() != null) {
							// This should not happen because the type has been assigned already to the
							// dummy class.
							candidatedSupertype = accessType.getDummyImplementationClass().getSimpleName();
						}
					} catch (Exception ex) {
						// can continue discovery
					}
					break;
				}
			}
		}

		if (candidatedSupertype != null) {
			if (!objectNode.has(ModelElement.EXPECTED_TYPE_PROPERTY)) {
				objectNode.set(ModelElement.EXPECTED_TYPE_PROPERTY, objectNode.get(Element.TYPE_PROPERTY));
			}
			objectNode.set(Element.TYPE_PROPERTY, new TextNode(candidatedSupertype));
			objectNode.remove(ModelElement.SUPERTYPES_PROPERTY);
			return objectNode;
		}

		StringBuffer stringBuffer = getError(unknownType);
		logger.trace("Unable to unmarshall {}. {}", objectNode.toString(), stringBuffer.toString());
		throw new RuntimeException(stringBuffer.toString());
	}

	/**
	 * Analyzes the types in an object node.
	 *
	 * @param objectNode The object node.
	 * @return The analyzed object node.
	 */
	protected static JsonNode analizeTypes(ObjectNode objectNode) {
		String cls = null;

		JsonNode typeJN = objectNode.get(Element.TYPE_PROPERTY);
		if (typeJN != null) {
			cls = objectNode.get(Element.TYPE_PROPERTY).asText();
		}

		if (cls == null && objectNode.getNodeType() == JsonNodeType.OBJECT) {
			cls = Property.NAME;
		}

		if (!knownTypes.containsKey(cls)) {
			objectNode = setTypeToBestAvailable(objectNode);
		}
		Iterator<String> iterator = objectNode.fieldNames();
		while (iterator.hasNext()) {
			String fieldName = iterator.next();
			JsonNode jn = objectNode.get(fieldName);
			switch (jn.getNodeType()) {
				case OBJECT:
					jn = analizeTypes((ObjectNode) jn);
					break;

				case ARRAY:
					jn = analizeTypes((ArrayNode) jn);
					break;

				default:
					break;
			}
			objectNode.replace(fieldName, jn);
		}
		return objectNode;
	}

	/**
	 * Analyzes the types in an array node.
	 *
	 * @param arrayNode The array node.
	 * @return The analyzed array node.
	 */
	protected static ArrayNode analizeTypes(ArrayNode arrayNode) {
		ArrayNode ret = mapper.createArrayNode();

		for (JsonNode jsonNode : arrayNode) {

			switch (jsonNode.getNodeType()) {
				case OBJECT:
					jsonNode = analizeTypes((ObjectNode) jsonNode);
					break;

				case ARRAY:
					jsonNode = analizeTypes((ArrayNode) jsonNode);
					break;

				default:
					break;
			}

			ret.add(jsonNode);
		}

		return ret;
	}

	/**
	 * Deserializes an object from a {@link Reader}.
	 *
	 * @param <El>   The type of the element.
	 * @param clz    The class of the element.
	 * @param reader The reader.
	 * @return The deserialized object.
	 * @throws IOException          if an error occurs during deserialization.
	 * @throws JsonParseException   if an error occurs during JSON parsing.
	 * @throws JsonMappingException if an error occurs during JSON mapping.
	 */
	public static <El extends Element> El unmarshal(Class<El> clz, Reader reader)
			throws JsonParseException, JsonMappingException, IOException {
		try {
			return mapper.readValue(reader, clz);
		} catch (InvalidTypeIdException e) {
			if (!ModelElement.class.isAssignableFrom(clz)) {
				throw e;
			}
			JsonNode jsonNode = mapper.readTree(reader);
			jsonNode = analizeTypes((ObjectNode) jsonNode);
			try {
				return ElementMapper.unmarshal(clz, mapper.writeValueAsString(jsonNode));
			} catch (Throwable t) {
				throw e;
			}
		}
	}

	/**
	 * Deserializes an object from an {@link InputStream}.
	 *
	 * @param <El>   The type of the element.
	 * @param clz    The class of the element.
	 * @param stream The input stream.
	 * @return The deserialized object.
	 * @throws IOException          if an error occurs during deserialization.
	 * @throws JsonParseException   if an error occurs during JSON parsing.
	 * @throws JsonMappingException if an error occurs during JSON mapping.
	 */
	public static <El extends Element> El unmarshal(Class<El> clz, InputStream stream)
			throws JsonParseException, JsonMappingException, IOException {
		try {
			return mapper.readValue(stream, clz);
		} catch (InvalidTypeIdException e) {
			if (!ModelElement.class.isAssignableFrom(clz)) {
				throw e;
			}
			JsonNode jsonNode = mapper.readTree(stream);
			jsonNode = analizeTypes((ObjectNode) jsonNode);
			try {
				return ElementMapper.unmarshal(clz, mapper.writeValueAsString(jsonNode));
			} catch (Throwable t) {
				throw e;
			}
		}
	}

	/**
	 * Deserializes an object from a JSON string.
	 *
	 * @param <El>   The type of the element.
	 * @param clz    The class of the element.
	 * @param string The JSON string.
	 * @return The deserialized object.
	 * @throws IOException          if an error occurs during deserialization.
	 * @throws JsonParseException   if an error occurs during JSON parsing.
	 * @throws JsonMappingException if an error occurs during JSON mapping.
	 */
	public static <El extends Element> El unmarshal(Class<El> clz, String string)
			throws JsonParseException, JsonMappingException, IOException {
		try {
			return mapper.readValue(string, clz);
		} catch (InvalidTypeIdException e) {
			if (!ModelElement.class.isAssignableFrom(clz)) {
				throw e;
			}
			JsonNode jsonNode = mapper.readTree(string);
			jsonNode = analizeTypes((ObjectNode) jsonNode);
			try {
				return ElementMapper.unmarshal(clz, mapper.writeValueAsString(jsonNode));
			} catch (Throwable t) {
				throw e;
			}
		}
	}

	/**
	 * Deserializes a list of objects from a JSON string.
	 *
	 * @param <El>   The type of the elements in the list.
	 * @param clz    The class of the elements.
	 * @param string The JSON string.
	 * @return The deserialized list.
	 * @throws IOException          if an error occurs during deserialization.
	 * @throws JsonParseException   if an error occurs during JSON parsing.
	 * @throws JsonMappingException if an error occurs during JSON mapping.
	 */
	public static <El extends Element> List<El> unmarshalList(Class<El> clz, String string)
			throws JsonParseException, JsonMappingException, IOException {
		JavaType type = mapper.getTypeFactory().constructCollectionType(ArrayList.class, clz);
		try {
			return mapper.readValue(string, type);
		} catch (InvalidTypeIdException e) {
			if (!ModelElement.class.isAssignableFrom(clz)) {
				throw e;
			}
			List<El> ret = new ArrayList<>();
			ArrayNode arrayNode = (ArrayNode) mapper.readTree(string);
			try {
				for (JsonNode jsonNode : arrayNode) {
					jsonNode = analizeTypes((ObjectNode) jsonNode);
					ret.add(ElementMapper.unmarshal(clz, mapper.writeValueAsString(jsonNode)));
				}
			} catch (Throwable t) {
				throw e;
			}
			return ret;
		}
	}

	/**
	 * Deserializes a list of objects from a JSON string.
	 *
	 * @param <El>   The type of the elements in the list.
	 * @param string The JSON string.
	 * @return The deserialized list.
	 * @throws IOException          if an error occurs during deserialization.
	 * @throws JsonParseException   if an error occurs during JSON parsing.
	 * @throws JsonMappingException if an error occurs during JSON mapping.
	 */
	public static <El extends Element> List<El> unmarshalList(String string)
			throws JsonParseException, JsonMappingException, IOException {
		JavaType type = mapper.getTypeFactory().constructCollectionType(ArrayList.class, Element.class);
		try {
			return mapper.readValue(string, type);
		} catch (InvalidTypeIdException e) {
			List<El> ret = new ArrayList<>();
			ArrayNode arrayNode = (ArrayNode) mapper.readTree(string);
			try {
				for (JsonNode jsonNode : arrayNode) {
					jsonNode = analizeTypes((ObjectNode) jsonNode);
					ret.add((El) ElementMapper.unmarshal(Element.class, mapper.writeValueAsString(jsonNode)));
				}
			} catch (Throwable t) {
				throw e;
			}
			return ret;
		}
	}

}
