package org.gcube.informationsystem.queries.templates.reference.entities;

import java.io.IOException;
import java.util.Map;

import org.gcube.com.fasterxml.jackson.annotation.JsonGetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import org.gcube.com.fasterxml.jackson.annotation.JsonSetter;
import org.gcube.com.fasterxml.jackson.core.JsonProcessingException;
import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.gcube.com.fasterxml.jackson.databind.node.ObjectNode;
import org.gcube.informationsystem.base.reference.entities.EntityElement;
import org.gcube.informationsystem.queries.templates.impl.entities.QueryTemplateImpl;
import org.gcube.informationsystem.queries.templates.reference.properties.TemplateVariable;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.utils.Version;

/**
 * Represents a query template that can be used to generate runnable queries by
 * substituting variables.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
@JsonIgnoreProperties(ignoreUnknown=true)
@JsonDeserialize(as=QueryTemplateImpl.class)
@TypeMetadata(name = QueryTemplate.NAME, description = "The type used to store Query Templates", version = Version.MINIMAL_VERSION_STRING)
@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
public interface QueryTemplate extends EntityElement {
	
	/** The name of the QueryTemplate entity type. */
	public static final String NAME = "QueryTemplate"; //QueryTemplate.class.getSimpleName();
	
	/** The property name for the template's name. */
	public static final String NAME_PROPERTY = "name";
	/** The property name for the template's description. */
	public static final String DESCRIPTION_PROPERTY = "description";
	
	/** The property name for the template's body. */
	public static final String TEMPLATE_PROPERTY = "template";
	/** The property name for the template's variables. */
	public static final String TEMPLATE_VARIABLES_PROPERTY = "templateVariables";
	
	/**
	 * Returns the name of the query template.
	 *
	 * @return The template name.
	 */
	@ISProperty(name = NAME_PROPERTY, description = "The name of the Query Template. Among UUID univocally identifiy the Query Template.", readonly = true, mandatory = true, nullable = false)
	public String getName();
	
	/**
	 * Sets the name of the query template.
	 *
	 * @param name The name to set.
	 */
	public void setName(String name);
	
	/**
	 * Returns the description of the query template.
	 *
	 * @return The template description.
	 */
	@ISProperty(name = DESCRIPTION_PROPERTY, description = "The description of the Query Template.", readonly = false, mandatory = true, nullable = false)
	public String getDescription();
	
	/**
	 * Sets the description of the query template.
	 *
	 * @param description The description to set.
	 */
	public void setDescription(String description);

	/**
	 * Returns the template body as a string.
	 *
	 * @return The template as a string.
	 * @throws JsonProcessingException if an error occurs during serialization.
	 */
	@JsonIgnore
	@ISProperty(name = TEMPLATE_PROPERTY, description = "The Query Template. It can contains query variables to be replaced to obtain a runnable query.", readonly = false, mandatory = true, nullable = false)
	public String getTemplateAsString() throws JsonProcessingException;
	
	/**
	 * Sets the template body from a string.
	 *
	 * @param template The template string.
	 * @throws IOException if an error occurs during parsing.
	 * @throws JsonProcessingException if an error occurs during JSON processing.
	 */
	@JsonIgnore
	public void setTemplate(String template) throws JsonProcessingException, IOException;
	
	/**
	 * Sets the template body from a {@link JsonNode}.
	 *
	 * @param template The template as a {@code JsonNode}.
	 */
	@JsonSetter(value = TEMPLATE_PROPERTY)
	public void setTemplate(JsonNode template);
	
	/**
	 * Returns the template body as a {@link JsonNode}.
	 *
	 * @return The template as a {@code JsonNode}.
	 */
	@JsonGetter(value = TEMPLATE_PROPERTY)
	public JsonNode getTemplate();
	
	/**
	 * Returns the variables defined in this template.
	 *
	 * @return A map of template variables, with variable names as keys.
	 */
	@JsonGetter(value = TEMPLATE_VARIABLES_PROPERTY)
	@ISProperty(name = TEMPLATE_VARIABLES_PROPERTY, description = "The Query Template Variables. It can contains Query Template Variable to be replaced to obtain a runnable query.", readonly = false, mandatory = true, nullable = false)
	public Map<String, TemplateVariable> getTemplateVariables();
	
	/**
	 * Adds a variable to the template. If a variable with the same name already
	 * exists, it will be overridden.
	 *
	 * @param templateVariable The variable to add.
	 */
	@JsonIgnore
	public void addTemplateVariable(TemplateVariable templateVariable);
	
	/**
	 * Creates an {@link ObjectNode} containing the default values of all template
	 * variables.
	 *
	 * @return An {@code ObjectNode} with default parameter values.
	 */
	@JsonIgnore
	public ObjectNode getParamsFromDefaultValues();
	
	/**
	 * Generates a runnable query by substituting variables with their default values.
	 *
	 * @return The generated query as a {@link JsonNode}.
	 * @throws Exception if an error occurs during query generation.
	 */
	@JsonIgnore
	public JsonNode getJsonQuery() throws Exception;
	
	/**
	 * Generates a runnable query by substituting variables with the provided values.
	 *
	 * @param values A {@link JsonNode} containing the values to substitute.
	 * @return The generated query as a {@link JsonNode}.
	 * @throws Exception if an error occurs during query generation.
	 */
	@JsonIgnore
	public JsonNode getJsonQuery(JsonNode values) throws Exception;

	
	
}