/**
 * 
 */
package org.gcube.informationsystem.model.reference.relations;

import java.util.Map;

import org.gcube.com.fasterxml.jackson.annotation.JsonAnyGetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonAnySetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonGetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import org.gcube.com.fasterxml.jackson.annotation.JsonInclude;
import org.gcube.com.fasterxml.jackson.annotation.JsonInclude.Include;
import org.gcube.com.fasterxml.jackson.annotation.JsonPropertyOrder;
import org.gcube.com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.base.reference.SchemaMixedElement;
import org.gcube.informationsystem.base.reference.relations.RelationElement;
import org.gcube.informationsystem.model.reference.ERElement;
import org.gcube.informationsystem.model.reference.ModelElement;
import org.gcube.informationsystem.model.reference.entities.Entity;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.properties.PropagationConstraint;
import org.gcube.informationsystem.serialization.AdditionalPropertiesSerializer;
import org.gcube.informationsystem.types.annotations.Abstract;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.utils.Version;

/**
 * The base interface for all relations in the facet-based resource model.
 * <p>
 * A relation connects a source {@link Resource} to a target {@link Entity} and
 * can have its own properties. It supports schema-mixed mode for additional
 * attributes. For more details, refer to the
 * <a href="https://wiki.gcube-system.org/gcube/Facet_Based_Resource_Model#Relation">
 * gCube Facet Based Resource Model Wiki</a>.
 *
 * @param <S> The type of the source resource.
 * @param <T> The type of the target entity.
 * @author Luca Frosini (ISTI - CNR)
 * @see RelationElement
 * @see SchemaMixedElement
 * @see ERElement
 */
@Abstract
@JsonPropertyOrder({ Element.TYPE_PROPERTY, ModelElement.SUPERTYPES_PROPERTY, ModelElement.EXPECTED_TYPE_PROPERTY, IdentifiableElement.ID_PROPERTY, IdentifiableElement.METADATA_PROPERTY, ERElement.CONTEXTS_PROPERTY, Relation.PROPAGATION_CONSTRAINT_PROPERTY })
// @JsonDeserialize(as=RelationImpl.class) Do not uncomment to manage subclasses
@TypeMetadata(name = Relation.NAME, description = "This is the base type for any Relation", version = Version.MINIMAL_VERSION_STRING)
@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
public interface Relation<S extends Resource, T extends Entity> extends RelationElement<S,T>, SchemaMixedElement, ERElement {
	
	/** The name of the relation type. */
	public static final String NAME = "Relation"; //Relation.class.getSimpleName();
	
	/** The property name for the propagation constraint. */
	public static final String PROPAGATION_CONSTRAINT_PROPERTY = "propagationConstraint";
	
	/**
	 * {@inheritDoc}
	 */
	@JsonInclude(Include.NON_NULL)
	@JsonIgnoreProperties({Resource.CONSISTS_OF_PROPERTY, Resource.IS_RELATED_TO_PROPERTY})
	@JsonGetter(value = SOURCE_PROPERTY)
	@Override
	public S getSource();
	
	/**
	 * {@inheritDoc}
	 */
	@JsonIgnore
	@Override
	public void setSource(S source);
	
	/**
	 * {@inheritDoc}
	 */
	@JsonGetter(value = TARGET_PROPERTY)
	@Override
	public T getTarget();
	
	/**
	 * {@inheritDoc}
	 */
	@JsonIgnore
	@Override
	public void setTarget(T target);
	
	/**
	 * Returns the propagation constraint for this relation.
	 *
	 * @return The {@link PropagationConstraint}.
	 */
	@JsonGetter(value = PROPAGATION_CONSTRAINT_PROPERTY)
	@ISProperty(name = PROPAGATION_CONSTRAINT_PROPERTY)
	public PropagationConstraint getPropagationConstraint();

	/**
	 * Sets the propagation constraint for this relation.
	 *
	 * @param propagationConstraint The {@link PropagationConstraint} to set.
	 */
	@JsonIgnore
	public void setPropagationConstraint(PropagationConstraint propagationConstraint);

	/**
	 * {@inheritDoc}
	 */
	@JsonAnyGetter
	@JsonSerialize(using = AdditionalPropertiesSerializer.class)
	@Override
	public Map<String,Object> getAdditionalProperties();
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAdditionalProperties(Map<String,Object> additionalProperties);
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object getAdditionalProperty(String key);
	
	/**
	 * {@inheritDoc}
	 */
	@JsonAnySetter
	@Override
	public void setAdditionalProperty(String key, Object value);
	
}
